#include "objects_update_move.h"

#include "maps/wikimap/mapspro/services/editor/src/branch_helpers.h"
#include "maps/wikimap/mapspro/services/editor/src/check_permissions.h"
#include "maps/wikimap/mapspro/services/editor/src/collection.h"
#include "maps/wikimap/mapspro/services/editor/src/commit.h"
#include "maps/wikimap/mapspro/services/editor/src/objects_cache.h"
#include "maps/wikimap/mapspro/services/editor/src/serialize/request_data.h"
#include "maps/wikimap/mapspro/services/editor/src/serialize/xml_parser.h"

#include <maps/libs/geolib/include/vector.h>

#include <geos/geom/CoordinateFilter.h>

namespace maps {
namespace wiki {

namespace {
const std::string TASK_METHOD_NAME = "ObjectsUpdateMove";
const std::string ACTION_ID_MOVE = "move";
}

ObjectsUpdateMove::ObjectsUpdateMove(
        const ObserverCollection& observers,
        const Request& request,
        taskutils::TaskID asyncTaskID)
    : controller::BaseController<ObjectsUpdateMove>(BOOST_CURRENT_FUNCTION, asyncTaskID)
    , request_(request)
    , observers_(observers)
{}

const std::string&
ObjectsUpdateMove::taskName()
{
    return TASK_METHOD_NAME;
}

std::string
ObjectsUpdateMove::Request::dump() const
{
    std::stringstream ss;
    ss << " uid: " << userId();
    ss << " branch: " << branchId;
    ss << " body: " << body;
    if (feedbackTaskId) {
        ss << " feedback-task-id: " << *feedbackTaskId;
    }
    return ss.str();
}

std::string
ObjectsUpdateMove::printRequest() const
{
    return request_.dump();
}

namespace {

class ObjectsUpdateMoveRequestData : public RequestData
{
public:
    ObjectsUpdateMoveRequestData(common::FormatType format, const std::string& body, ObjectsCache& cache)
        : dLon_(0)
        , dLat_(0)
    {
        parse(format, body, cache);
    }

    double dLon() const { return dLon_; }
    double dLat() const { return dLat_; }

protected:
    void parseContextDataJson(const json::Value& contextObject, ObjectsCache&) override
    {
        ASSERT(contextObject.hasField(STR_DELTA_LON));
        ASSERT(contextObject[STR_DELTA_LON].isNumber());
        dLon_ = contextObject[STR_DELTA_LON].as<double>();

        ASSERT(contextObject.hasField(STR_DELTA_LAT));
        ASSERT(contextObject[STR_DELTA_LAT].isNumber());
        dLat_ = contextObject[STR_DELTA_LAT].as<double>();
    }

    void parseContextDataXml(const maps::xml3::Node& contextNode, ObjectsCache&) override
    {
        dLon_ = contextNode.node("wmp:move/wmp:dlon").value<double>();
        dLat_ = contextNode.node("wmp:move/wmp:dlat").value<double>();
    }

private:
    double dLon_;
    double dLat_;
};

class MoveFilter : public geos::geom::CoordinateFilter
{
public:
    MoveFilter(const geolib3::Vector2& mercatorOffset)
        : dx_(mercatorOffset.x())
        , dy_(mercatorOffset.y())
    {}

    void filter_rw(geos::geom::Coordinate* coord) const override
    {
        coord->x += dx_;
        coord->y += dy_;
    }

    void filter_ro (const geos::geom::Coordinate*) override {}

private:
    double dx_;
    double dy_;
};

Geom
moveGeomByLL(const Geom& geom, double dLon, double dLat)
{
    REQUIRE(!geom.isNull(), "Trying to move null geometry.");
    auto curCenterMercGeom = geom.center();
    auto curCenterMercCoord = curCenterMercGeom->getCoordinate();
    auto centerLL = TGeoPoint(common::mercatorToGeodetic(
                        curCenterMercCoord->x,
                        curCenterMercCoord->y));
    MoveFilter filter(
                    common::geodeticTomercator(
                       centerLL.x() + dLon,
                       centerLL.y() + dLat) -
                    TMercatorPoint(curCenterMercCoord->x, curCenterMercCoord->y));

    auto newGeom = geom.geosGeometryPtr()->clone();
    newGeom->apply_rw(&filter);
    return Geom(std::move(newGeom));
}

std::string
doubleToString(double value)
{
    std::stringstream ss;
    ss.precision(DOUBLE_FORMAT_PRECISION);
    ss << value;
    return ss.str();
}

StringMap
createActionNotes(const ObjectsUpdateMoveRequestData& requestData)
{
    StringMap actionNotes;
    actionNotes[common::COMMIT_PROPKEY_GROUP_MOVED_OBJECTS_COUNT] =
        std::to_string(requestData.oids().size());
    actionNotes[common::COMMIT_PROPKEY_GROUP_MOVED_DLON] =
        doubleToString(requestData.dLon());
    actionNotes[common::COMMIT_PROPKEY_GROUP_MOVED_DLAT] =
        doubleToString(requestData.dLat());
    return actionNotes;
}

} // namespace

void
ObjectsUpdateMove::control()
{
    auto branchContext = BranchContextFacade::acquireWrite(request_.branchId, request_.userId());
    ObjectsCache cache(
        branchContext,
        boost::none);
    ObjectsUpdateMoveRequestData requestData(request_.format, request_.body, cache);

    GeoObjectCollection objects = cache.get(requestData.oids());
    cfg()->editor()->objectsUpdateRestrictions().checkObjects(ACTION_ID_MOVE, objects);

    StringSet categoryIds;
    for (const auto& oid : requestData.oids()) {
        ObjectPtr object = objects.getById(oid);
        WIKI_REQUIRE(object, ERR_MISSING_OBJECT, "Not found object id: " << oid);
        WIKI_REQUIRE(!object->geom().isNull(), ERR_BAD_REQUEST, "Object " << oid << " has no geometry");
        categoryIds.insert(object->categoryId());
    }
    CheckPermissions(request_.userId(), branchContext.txnCore()).checkPermissionsForGroupMove(
        categoryIds);

    for (const auto& oid : requestData.oids()) {
        ObjectPtr object = objects.getById(oid);
        object->setGeometry(moveGeomByLL(object->geom(),
            requestData.dLon(), requestData.dLat()));
        object->primaryEdit(true);
        object->calcModified();
    }
    saveAndNotify(objects, createActionNotes(requestData), cache);
}

void
ObjectsUpdateMove::saveAndNotify(
    GeoObjectCollection& /* objectContainer */,
    const StringMap& actionNotes,
    ObjectsCache& cache)
{
    cache.save(request_.userId(), common::COMMIT_PROPVAL_GROUP_MOVED, actionNotes);
    result_->taskName = taskName();
    CommitContext commitContext{request_.feedbackTaskId};
    result_->token = observers_.doCommit(cache, request_.userContext, commitContext);
    result_->commit = cache.savedCommit();
}

} // namespace wiki
} // namespace maps
