#include "objects_update_relations.h"

#include "maps/wikimap/mapspro/services/editor/src/revisions_facade.h"
#include "maps/wikimap/mapspro/services/editor/src/relations_manager.h"
#include "maps/wikimap/mapspro/services/editor/src/branch_helpers.h"
#include "maps/wikimap/mapspro/services/editor/src/check_permissions.h"
#include "maps/wikimap/mapspro/services/editor/src/collection.h"
#include "maps/wikimap/mapspro/services/editor/src/commit.h"
#include "maps/wikimap/mapspro/services/editor/src/objects_cache.h"
#include "maps/wikimap/mapspro/services/editor/src/serialize/objects_update_relations_parser.h"

namespace maps {
namespace wiki {

namespace {
const std::string TASK_METHOD_NAME = "ObjectsUpdateRelations";
const std::string ACTION_ID_UPDATE_RELATIONS = "update-relations";
}//namespace

ObjectsUpdateRelations::ObjectsUpdateRelations(
        const ObserverCollection& observers,
        Request request,
        taskutils::TaskID asyncTaskID)
    : controller::BaseController<ObjectsUpdateRelations>(BOOST_CURRENT_FUNCTION, asyncTaskID)
    , request_(std::move(request))
    , observers_(observers)
{}

const std::string&
ObjectsUpdateRelations::taskName()
{
    return TASK_METHOD_NAME;
}

std::string
ObjectsUpdateRelations::Request::dump() const
{
    std::stringstream ss;
    ss << " uid: " << userId();
    ss << " branch: " << branchId;
    if (feedbackTaskId) {
        ss << " feedback-task-id: " << *feedbackTaskId;
    }
    ss << " body: " << body;
    return ss.str();
}

std::string
ObjectsUpdateRelations::printRequest() const
{
    return request_.dump();
}

namespace {
TOIds revisionsToIds(const RevisionIds& revisionIds)
{
    TOIds oids;
    for (const auto& revisionId : revisionIds) {
        oids.insert(revisionId.objectId());
    }
    return oids;
}

void checkConflicts(ObjectsCache& cache, const RelativesDiffRevisionsByRole& rolesDiffs)
{
    for (const auto& roleDiff : rolesDiffs) {
        const auto& diff = roleDiff.second;
        if (!diff.added.empty()) {
            cache.revisionsFacade().gateway().checkConflicts(diff.added);
        }
        if (!diff.removed.empty()) {
            cache.revisionsFacade().gateway().checkConflicts(diff.removed);
        }
    }
}

void
deleteRelations(TOIds oids, const ObjectPtr& object, RelationType relationType, const std::string& role)
{
    RelationsManager& relationsManager = object->cache().relationsManager();
    for (const auto oid : oids) {
        auto masterId =
            relationType == RelationType::Slave
                ? object->id()
                : oid;
        auto slaveId =
            relationType == RelationType::Slave
                ? oid
                : object->id();
        if (!relationsManager.isRelationExists(masterId, slaveId, role)) {
            continue;
        }
        relationsManager.deleteRelation(masterId, slaveId, role);
    }
}

void
deleteAllRelations(const ObjectPtr& object, RelationType relationType, const std::string& role)
{
    TOIds oids;
    auto range = object->relations(relationType).range(role);
    for (const auto& rel : range) {
        oids.insert(rel.relative()->id());
    }
    deleteRelations(oids, object, relationType, role);
}


void
createRelations(TOIds oids, const ObjectPtr& object, RelationType relationType, const std::string& role)
{
    RelationsManager& relationsManager = object->cache().relationsManager();
    for (const auto oid : oids) {
        auto masterId =
            relationType == RelationType::Slave
                ? object->id()
                : oid;
        auto slaveId =
            relationType == RelationType::Slave
                ? oid
                : object->id();
        if (relationsManager.isRelationExists(masterId, slaveId, role)) {
            continue;
        }
        relationsManager.createRelation(masterId, slaveId, role);
    }
}

void
updateRelations(
    const RelationType relationType,
    const RelativesDiffRevisionsByRole& rolesDiffs,
    const RelativesToSetRevisionsByRole& rolesToSet,
    const ObjectPtr& object)
{
    for (const auto& roleDiff : rolesDiffs) {
        const auto& role = roleDiff.first;
        const auto& diff = roleDiff.second;
        deleteRelations(revisionsToIds(diff.removed), object, relationType, role);
        createRelations(revisionsToIds(diff.added), object, relationType, role);
    }
    for (const auto& [role, revisions] : rolesToSet) {
        deleteAllRelations(object, relationType, role);
        createRelations(revisionsToIds(revisions), object, relationType, role);
    }
}
} // namespace

void
ObjectsUpdateRelations::control()
{
    auto branchContext = BranchContextFacade::acquireWrite(request_.branchId, request_.userId());
    ObjectsCache cache(branchContext, boost::none);
    ObjectsUpdateRelationsParser bodyParser;
    bodyParser.parse(request_.body);
    cache.revisionsFacade().gateway().checkConflicts(bodyParser.objectsRevisionsToUpdate());
    checkConflicts(cache, bodyParser.mastersDiff());
    checkConflicts(cache, bodyParser.slavesDiff());
    const auto oids = revisionsToIds(bodyParser.objectsRevisionsToUpdate());
    GeoObjectCollection objects = cache.get(oids);
    const auto editorCfg = cfg()->editor();
    editorCfg->objectsUpdateRestrictions().checkObjects(
        ACTION_ID_UPDATE_RELATIONS,
        objects,
        ObjectsUpdateRestrictions::CategoriesCheckPolicy::ListedOnly);

    StringSet categoryIds;
    for (const auto& oid : oids) {
        ObjectPtr object = cache.getExisting(oid);
        categoryIds.insert(object->categoryId());
    }
    CheckPermissions(request_.userId(), branchContext.txnCore()).checkPermissionsForGroupUpdateRelations(
        categoryIds);

    for (const auto& oid : oids) {
        ObjectPtr object = objects.getById(oid);
        object->primaryEdit(true);
        updateRelations(RelationType::Master, bodyParser.mastersDiff(), bodyParser.mastersToSet(), object);
        updateRelations(RelationType::Slave, bodyParser.slavesDiff(), bodyParser.slavesToSet(), object);
        object->calcModified();
    }
    saveAndNotify(objects, cache);
}

void
ObjectsUpdateRelations::saveAndNotify(
    GeoObjectCollection& /* objectContainer */,
    ObjectsCache& cache)
{
    cache.save(request_.userId(), common::COMMIT_PROPVAL_GROUP_MODIFIED_RELATIONS);
    result_->taskName = taskName();
    CommitContext commitContext{request_.feedbackTaskId};
    result_->token = observers_.doCommit(cache, request_.userContext, commitContext);
    result_->commit = cache.savedCommit();
}

} // namespace wiki
} // namespace maps
