#include "dynamic_route_graph.h"

#include "util.h"

#include <maps/wikimap/mapspro/services/editor/src/actions/magic_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>

#include <maps/libs/geolib/include/intersection.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/vector.h>

#include <yandex/maps/wiki/common/geom_utils.h>
#include <yandex/maps/wiki/common/misc.h>
#include <yandex/maps/wiki/revision/objectrevision.h>
#include <yandex/maps/wiki/configs/editor/topology_groups.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <algorithm>
#include <utility>

namespace maps {
namespace wiki {

namespace {

bool isIncidenceRole(const std::string& roleId)
{
    return common::isIn(roleId, {ROLE_START, ROLE_END});
}

bool isSuffix(const ConditionPath& path, const ConditionPath& suffix)
{
    return (suffix.size() <= path.size()) && std::equal(
        suffix.rbegin(), suffix.rend(),
        path.rbegin()
    );
}

std::string getJunctionCategoryId(const std::string& elementCategoryId)
{
    const auto* topoGroup = cfg()->editor()->topologyGroups().findGroup(elementCategoryId);
    return topoGroup->junctionsCategory();
}

} // namespace

DynamicRouteGraph::DynamicRouteGraph(
        revision::Snapshot snapshot,
        Transaction& viewTxn,
        RoutingConfig config)
    : snapshot_(std::move(snapshot))
    , junctionIndex_(
        viewTxn,
        snapshot_.reader().branchId(),
        getJunctionCategoryId(config.elementCategoryId))
    , config_(std::move(config))
{}

const CompoundNodeID& DynamicRouteGraph::getCompoundNodeId(graph::NodeID nodeId) const
{
    return nodeIdMap_[nodeId];
}

const RouteElement& DynamicRouteGraph::getElement(TOid elementId) const
{
    const auto it = idToElement_.find(elementId);
    REQUIRE(
        it != idToElement_.end(),
        "There is no element with id " << elementId
    );
    return it->second;
}

const RouteCondition& DynamicRouteGraph::getCondition(TOid conditionId) const
{
    const auto it = idToCondition_.find(conditionId);
    REQUIRE(
        it != idToCondition_.end(),
        "There is no condition with id " << conditionId
    );
    return it->second;
}

TOIds DynamicRouteGraph::cacheJunctionIds(const geolib3::BoundingBox& bbox)
{
    TOIds junctionIds = junctionIndex_.idsByBox(bbox);
    for (auto it = junctionIds.begin(); it != junctionIds.end(); ) {
        if (!cachedJunctionIds_.insert(*it).second) {
            it = junctionIds.erase(it);
        } else {
            ++it;
        }
    }

    return junctionIds;
}

revision::Revisions DynamicRouteGraph::loadRelationsWithElements(const TOIds& junctionIds) const
{
    auto revisions = snapshot_.loadMasterRelations(junctionIds);
    revisions.remove_if(
        [](const revision::ObjectRevision& revision) {
            const std::string& roleId = extractAttributes(revision).at(ATTR_REL_ROLE);
            return !isIncidenceRole(roleId);
        }
    );

    return revisions;
}

TOIds DynamicRouteGraph::cacheFilteredElementsAndReturnIds(const TOIds& elementIds)
{
    TOIds filteredElementIds;
    for (const auto& pair: snapshot_.objectRevisions(elementIds)) {
        const TOid elementId = pair.first;
        RouteElement element(pair.second);
        if (config_.filter(element)) {
            filteredElementIds.insert(elementId);
            idToElement_.insert({elementId, std::move(element)});
        } else {
            discardedElementIds_.insert(elementId);
        }
    }

    return filteredElementIds;
}

void DynamicRouteGraph::cacheFilteredCondtionsForElements(const TOIds& elementIds)
{
    TOIds notCachedConditionIds;
    for (const auto& revision: snapshot_.loadMasterRelations(elementIds)) {
        const auto& roleId = extractAttributes(revision).at(ATTR_REL_ROLE);
        const TOid masterId = extractRelation(revision).masterObjectId();
        if (common::isIn(roleId, {ROLE_FROM, ROLE_TO})
                && !idToCondition_.count(masterId)
                && !discardedConditionIds_.count(masterId))
        {
                notCachedConditionIds.insert(masterId);
        }
    }

    for (auto& condition: RouteCondition::load(snapshot_, notCachedConditionIds)) {
        if (config_.condition->filter(condition)) {
            auto seqNum = RouteCondition::FROM_SEQ_NUM;
            for (; seqNum <= condition.lastSeqNum(); ++seqNum) {
                const auto pair = std::make_pair(
                    config_.condition->type(condition),
                    condition.at(seqNum)
                );
                conditionPositionsMap_[pair].emplace_back(condition.id(), seqNum);
            }
            idToCondition_.insert({condition.id(), std::move(condition)});
        } else {
            discardedConditionIds_.insert(condition.id());
        }
    }
}

Incidence DynamicRouteGraph::computeIncidence(
        const revision::Revisions& junctionRelationsWithElements) const
{
    Incidence incidence;

    for (const auto& relationRevision: junctionRelationsWithElements) {
        const auto& roleId = extractAttributes(relationRevision).at(ATTR_REL_ROLE);
        const auto& relation = extractRelation(relationRevision);
        const TOid elementId = relation.masterObjectId();
        const TOid junctionId = relation.slaveObjectId();

        const RouteElement& element = getElement(elementId);
        const auto startZlev = element[STR_START_ZLEV].as<int>(0);
        const auto endZlev = element[STR_END_ZLEV].as<int>(0);
        const auto direction = config_.direction(element);

        if (isSet(Direction::Forward, direction)) {
            if (roleId == ROLE_START) {
                LevelJunctionID id{junctionId, startZlev};
                incidence.out[id].emplace_back(elementId, Direction::Forward);
            } else {
                LevelJunctionID id{junctionId, endZlev};
                incidence.in[id].emplace_back(elementId, Direction::Forward);
            }
        }

        if (isSet(Direction::Backward, direction)) {
            if (roleId == ROLE_START) {
                LevelJunctionID id{junctionId, startZlev};
                incidence.in[id].emplace_back(elementId, Direction::Backward);
            } else {
                LevelJunctionID id{junctionId, endZlev};
                incidence.out[id].emplace_back(elementId, Direction::Backward);
            }
        }
    }

    return incidence;
}

const ConditionPositions& DynamicRouteGraph::getConditionPositions(
        RouteConditionType type,
        TOid elementId) const
{
    const static ConditionPositions NO_CONDITION_POSITIONS = {};

    if (!config_.condition) {
        return NO_CONDITION_POSITIONS;
    }
    const auto it = conditionPositionsMap_.find(std::make_pair(type, elementId));

    return it != conditionPositionsMap_.end()
        ? it->second
        : NO_CONDITION_POSITIONS;
}

bool DynamicRouteGraph::isTurnaboutPossible(TOid elementId, TOid junctionId) const
{
    const auto& positions = getConditionPositions(RouteConditionType::Turnabout, elementId);

    return std::any_of(
        positions.begin(), positions.end(),
        [&, this](const ConditionPosition& pos) {
            const auto& cond = getCondition(pos.conditionId);
            return cond.isFromSeqNum(pos.seqNum)
                && cond.viaJunctionId() == junctionId
                && cond.isLastSeqNum(0)
                && cond.at(0) == elementId;
        }
    );
}

IsStartOfForbiddenCondition DynamicRouteGraph::isStartOfForbiddenCondition(
        TOid fromElId,
        TOid viaJcId,
        TOid toElId) const
{
    IsStartOfForbiddenCondition result = IsStartOfForbiddenCondition::No;

    for (const auto& position: getConditionPositions(RouteConditionType::Forbidden, fromElId)) {
        const auto& condition =  getCondition(position.conditionId);

        const bool isStart = condition.fromElementId() == fromElId
            && condition.viaJunctionId() == viaJcId
            && !condition.toElementIds().empty()
            && condition.toElementIds().at(0) == toElId;

        if (isStart) {
            if (condition.toElementIds().size() == 1) {
                result = IsStartOfForbiddenCondition::YesAndMoveIsForbidden;
                break;
            }

            result = IsStartOfForbiddenCondition::Yes;
        }
    }

    return result;
}

bool DynamicRouteGraph::anyConditionForbidPath(const ConditionPath& path) const
{
    REQUIRE(
        path.size() >= 3,
        "Path must have 3 element at least"
    );

    const auto& positions = getConditionPositions(RouteConditionType::Forbidden, path.back());

    return std::any_of(
        positions.begin(), positions.end(),
        [&](const ConditionPosition& pos) {
            const auto& cond = getCondition(pos.conditionId);
            return cond.isLastSeqNum(pos.seqNum) && isSuffix(path, cond.pathTo(pos.seqNum));
        }
    );
}

ConditionPath DynamicRouteGraph::getLongestPathSuffixForbiddenConditionPrefix(
        const ConditionPath& path) const
{
    REQUIRE(
        path.size() >= 3,
        "Path must have 3 element at least"
    );

    const auto positions = getConditionPositions(RouteConditionType::Forbidden, path.back());

    ConditionPath longestSuffixPath;
    for (const auto& pos: positions) {
        const auto& cond = getCondition(pos.conditionId);
        auto condPath = cond.pathTo(pos.seqNum);

        if (!isSuffix(path, condPath)) {
            continue;
        }

        if (longestSuffixPath.size() < condPath.size()) {
            longestSuffixPath = std::move(condPath);
        }
    }

    return longestSuffixPath.size() > 1
        ? longestSuffixPath
        : ConditionPath();
}

void DynamicRouteGraph::addNodes(const Incidence& incidence)
{
    for (const auto& pair: incidence.in) {
        const LevelJunctionID& viaJcId = pair.first;
        const auto it = incidence.out.find(viaJcId);

        for (const DirectedElementID& fromElId: pair.second) {
            const graph::NodeID fromNodeId = nodeIdMap_(CompoundNodeID{fromElId});

            if (it == incidence.out.end()) { // sink node
                nodeIdToEdges_[fromNodeId] = graph::Edges();
                continue;
            }

            const auto& toElIds = it->second;
            const RouteElement& fromElement = getElement(fromElId.id());

            graph::Edges edgeList;
            for (const auto& toElId: toElIds) {
                if (fromElId.id() == toElId.id()
                        && !isTurnaboutPossible(fromElId.id(), viaJcId.id))
                {
                    continue;
                }

                const RouteElement& toElement = getElement(toElId.id());

                switch (isStartOfForbiddenCondition(fromElId.id(), viaJcId.id, toElId.id())) {
                    case IsStartOfForbiddenCondition::No:
                        edgeList.emplace_back(
                            fromNodeId,
                            nodeIdMap_(CompoundNodeID{toElId}),
                            config_.weight(fromElement, toElement)
                        );
                        break;

                    case IsStartOfForbiddenCondition::Yes:
                        edgeList.emplace_back(
                            fromNodeId,
                            nodeIdMap_(
                                CompoundNodeID{toElId, {fromElId.id(), toElId.id()}}
                            ),
                            config_.weight(fromElement, toElement)
                        );
                        break;

                    case IsStartOfForbiddenCondition::YesAndMoveIsForbidden:
                        // do nothing
                        break;
                }
            }

            nodeIdToEdges_[fromNodeId] = std::move(edgeList);
        }
    }
}

void DynamicRouteGraph::cacheByBBox(const geolib3::BoundingBox& bbox)
{
    DEBUG() << "Cache by bbox " << geolib3::WKT::toString(bbox.polygon());

    WIKI_REQUIRE(
        idToElement_.size() < config_.loadElementsLimit,
        ERR_ROUTING_SEARCH_AREA_TOO_LARGE,
        "Element cache is overflowed"
    );

    const auto cacheBox = geolib3::intersection(config_.aoi, bbox);
    if (!cacheBox) {
        DEBUG() << "There is no intersection with aoi";
        return;
    }

    auto relationsWithElements = loadRelationsWithElements(cacheJunctionIds(*cacheBox));

    TOIds notCachedElementIds;
    for (const auto& relationRevision: relationsWithElements) {
        const TOid elementId = extractRelation(relationRevision).masterObjectId();
        if (!idToElement_.count(elementId) && !discardedElementIds_.count(elementId)) {
            notCachedElementIds.insert(elementId);
        }
    }
    const TOIds elementIds = cacheFilteredElementsAndReturnIds(notCachedElementIds);

    relationsWithElements.remove_if(
        [this](const revision::ObjectRevision& revision) {
            const TOid elementId = extractRelation(revision).masterObjectId();
            return discardedElementIds_.count(elementId);
        }
    );

    if (config_.condition) {
        cacheFilteredCondtionsForElements(elementIds);
    }

    addNodes(computeIncidence(relationsWithElements));
}

void DynamicRouteGraph::cacheByNodeId(graph::NodeID nodeId)
{
    const TOid elementId = getCompoundNodeId(nodeId).directedElementId.id();
    const geolib3::Polyline2& polyline = getElement(elementId).geom();

    const geolib3::BoundingBox bbox = polyline.boundingBox();
    const double margin = geolib3::toMercatorUnits(config_.cacheRadiusMeters, bbox.center());

    cacheByBBox(geolib3::resizeByValue(bbox, margin));
}

void DynamicRouteGraph::cacheByPoint(const geolib3::Point2& point)
{
    const size_t size = geolib3::toMercatorUnits(2 * config_.cacheRadiusMeters, point);
    const geolib3::BoundingBox bbox(point, size, size);
    cacheByBBox(bbox);
}

std::vector<graph::NodeID> DynamicRouteGraph::nodesAtDistance(
        const geolib3::Point2& point,
        double radiusMeters)
{
    cacheByPoint(point);

    std::vector<graph::NodeID> result;
    const double maxPossibleDistance = geolib3::toMercatorUnits(radiusMeters, point);

    for (const auto& pair: idToElement_) {
        const RouteElement& element = pair.second;
        const geolib3::Polyline2& polyline = element.geom();
        const auto segment = polyline.segmentAt(polyline.closestPointSegmentIndex(point));
        const double distance = geolib3::distance(point, segment);

        if (distance > maxPossibleDistance) {
            continue;
        }

        const geolib3::Vector2 roadVector = segment.vector();
        const geolib3::Vector2 pointVector = point - segment.start();

        auto getNode = [this](TOid elementId, Direction direction) {
            DirectedElementID directedElementId{elementId, direction};
            return nodeIdMap_(CompoundNodeID{directedElementId});
        };

        const auto direction = config_.direction(element);
        if (geolib3::crossProduct(roadVector, pointVector) <= 0) {
            if (isSet(Direction::Forward, direction)) {
                result.push_back(getNode(element.id(), Direction::Forward));
            }
        } else {
            if (isSet(Direction::Backward, direction)) {
                result.push_back(getNode(element.id(), Direction::Backward));
            }
        }
    }

    return result;
}

boost::optional<graph::NodeID> DynamicRouteGraph::node(
        const DirectedElementID& directedElementId,
        boost::optional<geolib3::BoundingBox> cacheHint)
{
    CompoundNodeID compoundNodeId{directedElementId};

    if (nodeIdMap_.has(compoundNodeId)) {
        return nodeIdMap_(compoundNodeId);
    }

    if (!cacheHint) {
        const auto revision = snapshot_.objectRevision(directedElementId.id());

        if (revision && !revision->data().deleted) {
            const RouteElement element = RouteElement(*revision);
            const geolib3::BoundingBox bbox = element.geom().boundingBox();

            cacheHint = geolib3::resizeByValue(
                bbox,
                geolib3::toMercatorUnits(config_.cacheRadiusMeters, bbox.center())
            );
        }
    }

    if (cacheHint) {
        cacheByBBox(*cacheHint);
        if (nodeIdMap_.has(compoundNodeId)) {
            return nodeIdMap_(compoundNodeId);
        }
    }

    return boost::none;
}

const graph::Edges DynamicRouteGraph::edges(graph::NodeID nodeId)
{
    auto it = nodeIdToEdges_.find(nodeId);
    if (it == nodeIdToEdges_.end()) {
        const auto& fromNodeId = getCompoundNodeId(nodeId);
        if (!fromNodeId.conditionPath.empty()) {
            graph::Edges edgeList;

            const CompoundNodeID fromNodeIdWithoutPath(fromNodeId.directedElementId);
            for (const auto& edge: edges(nodeIdMap_(fromNodeIdWithoutPath))) {
                const auto& toNodeId = getCompoundNodeId(edge.endNodeId());
                auto path = fromNodeId.conditionPath;
                path.push_back(toNodeId.directedElementId.id());

                if (!anyConditionForbidPath(path)) {
                    edgeList.emplace_back(
                        nodeId,
                        nodeIdMap_(
                            CompoundNodeID{
                                toNodeId.directedElementId,
                                getLongestPathSuffixForbiddenConditionPrefix(path)
                            }
                        ),
                        edge.weight()
                    );
                }
            }

            return nodeIdToEdges_[nodeId] = std::move(edgeList);
        }

        cacheByNodeId(nodeId);
        it = nodeIdToEdges_.find(nodeId);
    }

    if (it == nodeIdToEdges_.end()) {
        return nodeIdToEdges_[nodeId] = graph::Edges();
    }

    return it->second;
}

} // namespace wiki
} // namespace maps
