#include "objects_query_path.h"

#include "common.h"

#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/common.h>
#include <maps/wikimap/mapspro/services/editor/src/magic_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/utils.h>

#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>

#include "helper.h"
#include "result_helper.h"
#include "route_element.h"
#include "dynamic_route_graph.h"

#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/graph/shortest_path.h>

#include <yandex/maps/wiki/common/geom_utils.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/point.h>

#include <boost/optional.hpp>

#include <map>
#include <memory>
#include <vector>


namespace maps {
namespace wiki {
namespace {

const std::string STR_TASK_METHOD_NAME = "ObjectsQueryPath";

constexpr double CACHE_RADIUS_METERS = 500;
constexpr double SEARCH_AREA_MARGIN_METERS = 1000;
constexpr double MAX_AREA_SIZE_METERS = 15000;
constexpr size_t LOAD_ELEMENTS_LIMIT = 15000;


RouteElements loadElements(revision::Snapshot& snapshot, const std::vector<TOid>& elementIds)
{
    RouteElements elements;
    for (const auto& revision: loadSequence(snapshot, elementIds)) {
        elements.emplace_back(revision);
    }

    const std::string& categoryId = elements.front().categoryId();
    REQUIRE(
        std::all_of(
            elements.begin(), elements.end(),
            [&categoryId](const RouteElement& el) {
                return el.categoryId() == categoryId;
            }
        ),
        "Loaded objects has different category id"
    );

    return elements;
}

RoutingConfig createRoutingConfig(const RouteElements& elements)
{
    WIKI_REQUIRE(
        elements.front().categoryId() == CATEGORY_RD_EL,
        ERR_ROUTING_UNSUPPORTED_ELEMENT_CATEGORY,
        "Now only " << CATEGORY_RD_EL << " category is supported"
    );

    geolib3::BoundingBox bbox = findBoundingBox(elements);
    bbox = resizeByValue(bbox, geolib3::toMercatorUnits(SEARCH_AREA_MARGIN_METERS, bbox.center()));

    WIKI_REQUIRE(
        !isSizeLimitExceeded(bbox, MAX_AREA_SIZE_METERS),
        ERR_ROUTING_SEARCH_AREA_TOO_LARGE,
        "Too large area of search"
    );

    return RoutingConfig{
        elements.front().categoryId(),
        /* aoi  = */ bbox,
        /* direction = */ [](const RouteElement&) { return Direction::Both; },
        /* filter = */ [](const RouteElement& el) {
            return el[STR_FC].as<int>() <= 7;
        },
        /* weight = */ [](const RouteElement& lhs, const RouteElement& /*rhs*/) {
            return lhs.lengthMeters();
        },
        /* condition = */ boost::none,
        CACHE_RADIUS_METERS,
        LOAD_ELEMENTS_LIMIT
    };
}

struct ObjectsQueryPathResult {
    graph::Path path;
    double length;

    explicit operator bool() const { return !path.empty(); }
};

const ObjectsQueryPathResult& best(
        const ObjectsQueryPathResult& lhs,
        const ObjectsQueryPathResult& rhs)
{
    if (lhs && (!rhs || lhs.length < rhs.length)) {
        return lhs;
    }

    return rhs;
}

ObjectsQueryPathResult findPath(
        DynamicRouteGraph& graph,
        const RouteElements& elements,
        routing::Direction startDirection)
{
    const geolib3::BoundingBox bbox = elements.front().geom().boundingBox();

    const auto startNodeId = graph.node(
        DirectedElementID{elements.front().id(), startDirection},
        resizeByValue(bbox, 2 * geolib3::toMercatorUnits(CACHE_RADIUS_METERS, bbox.center()))
    );

    WIKI_REQUIRE(
        startNodeId,
        ERR_ROUTING_IMPOSSIBLE_FIND_PATH,
        "Impossible to find element with id " << elements.front().id()
    );

    graph::NodeID fromNodeId = *startNodeId;
    ObjectsQueryPathResult result{{fromNodeId}, 0};
    for (size_t i = 1; i < elements.size(); ++i) {
        const auto shortestPathResult = graph::findShortestPath(
            /* outEdges = */ [&graph](const graph::NodeID nodeId) {
                return graph.edges(nodeId);
            },
            /* fromNodeId = */ fromNodeId,
            /* isFinishNode = */ [&](graph::NodeID nodeId) {
                return elements[i].id() == graph.getCompoundNodeId(nodeId).directedElementId.id();
            }
        );
        if (shortestPathResult.path().empty()) {
            return {{}, 0};
        }

        result.path.insert(
            result.path.end(),
            shortestPathResult.path().begin() + 1, shortestPathResult.path().end()
        );
        result.length += shortestPathResult.length();
        fromNodeId = shortestPathResult.path().back();
    }

    return result;
}

} // namespace

std::string ObjectsQueryPath::Request::dump() const
{
    std::stringstream ss;

    ss << " user: " << user
       << " token: " << token
       << " branch: " << branchId
       << " elementIds [" << common::join(elementIds, ", ") << "]"
       << " elementLimit: " << elementLimit;

    return ss.str();
}

ObjectsQueryPath::ObjectsQueryPath(
        const ObserverCollection&,
        const Request& request,
        taskutils::TaskID asyncTaskID)
    : controller::BaseController<ObjectsQueryPath>(BOOST_CURRENT_FUNCTION, asyncTaskID)
    , request_(request)
{}

std::string ObjectsQueryPath::printRequest() const { return request_.dump(); }

void ObjectsQueryPath::control()
{
    BranchContextFacade facade(request_.branchId);

    auto context = facade.acquireRead(request_.branchId, request_.token);
    auto gateway = revision::RevisionsGateway(context.txnCore(), context.branch);
    auto snapshot = gateway.snapshot(gateway.headCommitId());

    WIKI_REQUIRE(
        request_.elementIds.size() >= 2,
        ERR_BAD_REQUEST,
        "Need at least two elemens as input"
    );

    const RouteElements elements = loadElements(snapshot, request_.elementIds);
    const RoutingConfig config = createRoutingConfig(elements);

    DynamicRouteGraph graph{snapshot, context.txnView(), config};

    const auto resultWithForwardStart = findPath(graph, elements, Direction::Forward);
    const auto resultWithBackwardStart = findPath(graph, elements, Direction::Backward);

    WIKI_REQUIRE(
        resultWithForwardStart || resultWithBackwardStart,
        ERR_ROUTING_IMPOSSIBLE_FIND_PATH,
        "Impossible find path for elements ["
            << common::join(request_.elementIds, ", ")
            << "]"
    );


    TOIds elementIdSet;
    for (const auto id: best(resultWithForwardStart, resultWithBackwardStart).path) {
        elementIdSet.insert(graph.getCompoundNodeId(id).directedElementId.id());
    }

    if (request_.elementLimit) {
        WIKI_REQUIRE(
            elementIdSet.size() <= request_.elementLimit,
            ERR_ROUTING_TOO_MANY_ELEMENTS,
            "Too many elements " << elementIdSet.size()
        );
    }

    result_->elements = loadViewObjects(context.txnView(), request_.branchId, elementIdSet);
    result_->elementBBox = findBoundingBox(result_->elements);
}

const std::string& ObjectsQueryPath::taskName() { return STR_TASK_METHOD_NAME; }

} // namespace wiki
} // namespace maps
