#include "objects_query_route_diff.h"

#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/magic_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/utils.h>

#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>

#include "helper.h"
#include "result_helper.h"
#include "route_condition.h"
#include "route_element.h"
#include "dynamic_route_graph.h"

#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/routing/exception.h>
#include <yandex/maps/wiki/routing/route.h>

#include <yandex/maps/wiki/common/misc.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/point.h>

#include <boost/optional.hpp>

#include <functional>
#include <map>
#include <memory>
#include <sstream>
#include <string>
#include <vector>


namespace maps {
namespace wiki {
namespace {

const std::string STR_TASK_METHOD_NAME = "ObjectsQueryRouteDiff";

boost::optional<geolib3::BoundingBox> joinOptionalBBoxes(
        const boost::optional<geolib3::BoundingBox>& lhs,
        const boost::optional<geolib3::BoundingBox>& rhs)
{
    if (!lhs) {
        return rhs;
    }

    if (!rhs) {
        return lhs;
    }

    return geolib3::expand(lhs->boundingBox(), rhs->boundingBox());
}

} // namespace

std::string ObjectsQueryRouteDiff::Request::dump() const
{
    std::stringstream out;

    out << " user: " << user
        << " token: " << token
        << " branch: " << branchId

        << " revisionId: " << revisionId
        << " categoryId: " << categoryId

        << " addElementIds [" << common::join(addElementIds, ',') << ']'
        << " removeElementIds [" << common::join(removeElementIds, ',') << ']'

        << " threadStops: " << threadStopSequence.toString()
        << " fromThreadStopIdx << " << fromThreadStopIdx
        << " toThreadStopIdx << " << toThreadStopIdx

        << " elementLimit: " << elementLimit;

    return out.str();
}

ObjectsQueryRouteDiff::ObjectsQueryRouteDiff(
        const ObserverCollection&,
        const Request& request,
        taskutils::TaskID asyncTaskID)
    : controller::BaseController<ObjectsQueryRouteDiff>(BOOST_CURRENT_FUNCTION, asyncTaskID)
    , request_(request)
{}

std::string ObjectsQueryRouteDiff::printRequest() const { return request_.dump(); }
namespace {
TOIds
elementsIds(const routing::RestoreResult& restoreResult) {
    TOIds resultElementsIds;
    for (const auto& point: restoreResult.trace()) {
        resultElementsIds.insert(point.directedElementId().id());
    }
    return resultElementsIds;
}

const double DISTANCE_TO_BUFFER_RATIO = 0.2;
const double DISTANCE_TO_BUFFER_EXTRA_GAP_MERCATOR = 200;

TOIds filterElementsInStationsBuffer(
    const routing::Point& stationPt1, const routing::Point& stationPt2,
    const IdToRevision& idToElementRevision,
    const TOIds& elementsIds)
{
    Geom lineBetweenStations = polylineToGeom(
        geolib3::Polyline2 {
                geolib3::PointsVector {stationPt1, stationPt2}
        });
    double bufferWidth =
        lineBetweenStations.realLength() * DISTANCE_TO_BUFFER_RATIO
        + DISTANCE_TO_BUFFER_EXTRA_GAP_MERCATOR;
    Geom buffer = lineBetweenStations.createBuffer(bufferWidth);
    TOIds elementsInBuffer;
    for(const auto elementId: elementsIds) {
        auto elementRevisionIt = idToElementRevision.find(elementId);
        ASSERT(elementRevisionIt != idToElementRevision.end());
        const auto& elementRevision = elementRevisionIt->second.data();
        ASSERT(elementRevision.geometry);
        Geom elementGeom(*elementRevision.geometry);
        if (buffer->intersects(elementGeom.geosGeometryPtr())) {
            elementsInBuffer.insert(elementId);
        }
    }
    return elementsInBuffer;
}
} // namespace
void ObjectsQueryRouteDiff::control()
{
    BranchContextFacade facade(request_.branchId);

    auto context = facade.acquireRead(request_.branchId, request_.token);
    auto gateway = revision::RevisionsGateway(context.txnCore(), context.branch);
    auto snapshot = gateway.snapshot(gateway.headCommitId());

    restoreStopIds(snapshot, request_.threadStopSequence);
    const routing::Stops stops = loadStops(snapshot, request_.threadStopSequence.stopIds());

    WIKI_REQUIRE(
        request_.fromThreadStopIdx < request_.toThreadStopIdx
            && request_.toThreadStopIdx < stops.size(),
        ERR_BAD_REQUEST,
        "Invalid stops idx"
    );

    routing::Stops diffStops;
    for (size_t i = request_.fromThreadStopIdx; i <= request_.toThreadStopIdx; ++i) {
        diffStops.push_back(stops[i]);
    }

    const RoutingConfig config = defaultTransportRoutingConfig(request_.categoryId, diffStops);

    const TOIds threadElementIds = getThreadElementIds(
        snapshot,
        request_.revisionId.objectId(),
        request_.addElementIds,
        request_.removeElementIds
    );

    const auto idToElementRevision = load(snapshot, threadElementIds);

    TOIds oldElementIds;
    if (request_.fromThreadStopIdx == 0 && request_.toThreadStopIdx + 1 == stops.size()) {
        // force rebuild
        oldElementIds = TOIds(threadElementIds);
    } else {
        try {
            const auto allRoutingElements = toElements(snapshot, config, idToElementRevision);
            const auto routingConditions = loadConditions(snapshot, config, threadElementIds);
            auto restore = [&](const routing::Stops& stops) {
                return routing::restore(
                    allRoutingElements,
                    stops,
                    routingConditions,
                    DEFAULT_STOP_SEARCH_RADIUS_METERS
                );
            };
            const auto restoreResult = restore({diffStops.front(), diffStops.back()});
            if (restoreResult.isOk()) {
                oldElementIds = elementsIds(restoreResult);
            } else if (!restoreResult.noPathErrors().empty()) {
                TOIds keepElementsIds;
                bool tryToRemove = true;
                if (request_.fromThreadStopIdx > 0) {
                    routing::Stops beforeFromStop;
                    std::copy(stops.begin(), stops.begin() + request_.fromThreadStopIdx + 1,
                        std::back_inserter(beforeFromStop));
                    const auto restoreResultTo = restore(beforeFromStop);
                    if (restoreResultTo.isOk()) {
                        keepElementsIds = elementsIds(restoreResultTo);
                    } else {
                        tryToRemove = false;
                    }
                }
                if (tryToRemove && request_.toThreadStopIdx + 1 < stops.size()) {
                    routing::Stops afterToStop;
                    std::copy(stops.begin() + request_.toThreadStopIdx, stops.end(),
                        std::back_inserter(afterToStop));
                    const auto restoreResultFrom = restore(afterToStop);
                    if (restoreResultFrom.isOk()) {
                        const auto elementsIdsFrom = elementsIds(restoreResultFrom);
                        keepElementsIds.insert(elementsIdsFrom.begin(), elementsIdsFrom.end());
                    } else {
                        tryToRemove = false;
                    }
                }
                if (tryToRemove) {
                    std::set_difference(
                        threadElementIds.begin(), threadElementIds.end(),
                        keepElementsIds.begin(), keepElementsIds.end(),
                        std::inserter(oldElementIds, oldElementIds.begin())
                    );
                    oldElementIds = filterElementsInStationsBuffer(
                        diffStops.front().geom(), diffStops.back().geom(),
                        idToElementRevision, oldElementIds);
                }
            }
        } catch (routing::BaseRoutingError&) {
            INFO() << taskName() << " didn't remove elements, impossible to restore path";
        }
    }

    DynamicRouteGraph graph{snapshot, context.txnView(), config};
    const TOIds newElementIds = findThreadElementIds(graph, diffStops);

    TOIds addElementIds, removeElementIds;
    std::set_difference(
        newElementIds.begin(), newElementIds.end(),
        threadElementIds.begin(), threadElementIds.end(),
        std::inserter(addElementIds, addElementIds.begin())
    );
    std::set_difference(
        oldElementIds.begin(), oldElementIds.end(),
        newElementIds.begin(), newElementIds.end(),
        std::inserter(removeElementIds, removeElementIds.begin())
    );

    if (request_.elementLimit) {
        const size_t diffElementNum = addElementIds.size() + removeElementIds.size();
        WIKI_REQUIRE(
            diffElementNum < request_.elementLimit,
            ERR_ROUTING_TOO_MANY_ELEMENTS,
            "Too many elements in diff " << diffElementNum
        );
    }

    boost::optional<geolib3::BoundingBox> addBBox, removeBBox;

    if (!addElementIds.empty()) {
        result_->addElements = loadViewObjects(context.txnView(), request_.branchId, addElementIds);
        addBBox = findBoundingBox(result_->addElements);
    }

    if (!removeElementIds.empty()) {
        result_->removeElements = loadViewObjects(context.txnView(), request_.branchId, removeElementIds);
        removeBBox = findBoundingBox(result_->removeElements);
    }

    result_->elementBBox = joinOptionalBBoxes(addBBox, removeBBox);
}

const std::string& ObjectsQueryRouteDiff::taskName() { return STR_TASK_METHOD_NAME; }

} // namespace wiki
} // namespace maps
