#include "route_element.h"

#include "util.h"

#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/exception.h>
#include <maps/wikimap/mapspro/services/editor/src/magic_strings.h>

#include <yandex/maps/wiki/common/rd/speed_cat.h>


namespace maps {
namespace wiki {

namespace {

const constexpr double AVERAGE_TRAM_SPEED_KMH = 20;

} // namesace

RouteElement::RouteElement(const revision::ObjectRevision& revision)
    : RouteObject(revision)
    , geom_(extractGeom<geolib3::Polyline2>(revision))
{
    REQUIRE(
        geom_.pointsNumber() >= 2,
        "Geometry of element must have 2 points at least"
    );
}

const geolib3::Polyline2& RouteElement::geom() const
{
    return geom_;
}

double RouteElement::lengthMeters() const
{
    return mercatorRealLength(geom());
}

int RouteElement::speedCat() const {
    REQUIRE(
        categoryId() == CATEGORY_RD_EL,
        "Impossible find speed category for " << categoryId()
    );

    int speedCat = (*this)[STR_SPEED_CAT];

    if (!speedCat) {
        speedCat = common::predictSpeedCategory(
            (*this)[STR_FC].as<int>(),
            (*this)[STR_FOW].as<FOW>(),
            (*this)[STR_PAVED].as<bool>()
        );
    }

    return speedCat;
}


SpeedInterval RouteElement::speedIntervalMetersPerSecond() const
{
    if (categoryId() == CATEGORY_TRANSPORT_TRAM_EL) {
        return SpeedInterval {
            KMH_TO_MS_RATIO * AVERAGE_TRAM_SPEED_KMH,
            KMH_TO_MS_RATIO * AVERAGE_TRAM_SPEED_KMH
        };
    }

    if (categoryId() == CATEGORY_RD_EL) {
        const int category = speedCat();
        REQUIRE(category, "Impossible to define speed interval");

        const auto speedInterval = common::toSpeedInterval(category);

        return SpeedInterval {
            KMH_TO_MS_RATIO * speedInterval.min,
            KMH_TO_MS_RATIO * speedInterval.max
        };
    }

    THROW_WIKI_LOGIC_ERROR(
        ERR_ROUTING_UNSUPPORTED_ELEMENT_CATEGORY,
        "'" << categoryId() << "' is unsupported element category id"
    );
}

} // namesace wiki
} // namespace maps
