#include "thread_stop.h"

#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>

#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <sstream>
#include <utility>


namespace maps {
namespace wiki {

ThreadStop::ThreadStop(TOid id, TOid stopId, const StringMap& attrs)
    : id(id)
    , stopId(stopId)
    , attrs(attrs)
{}

ThreadStop::ThreadStop(TOid id, TOid stopId, const StringMultiMap& attrs)
    : id(id)
    , stopId(stopId)
    , attrs(attrs)
{}

ThreadStopSequence::ThreadStopSequence() = default;

ThreadStopSequence::ThreadStopSequence(ThreadStopVector&& threadStops)
    : threadStops_(std::move(threadStops))
{
    for (size_t idx = 0; idx < threadStops_.size(); ++idx) {
        if (threadStops_[idx].id) {
            threadStopIdToIdx_.emplace(threadStops_[idx].id, idx);
        }
    }
}

size_t ThreadStopSequence::size() const { return threadStops_.size(); }

ThreadStop& ThreadStopSequence::at(size_t idx)
{
    REQUIRE(idx < size(), "Invalid index " << idx);
    return threadStops_.at(idx);
}

const ThreadStop& ThreadStopSequence::at(size_t idx) const
{
    REQUIRE(idx < size(), "Invalid index " << idx);
    return threadStops_.at(idx);
}

ThreadStop& ThreadStopSequence::byId(TOid id)
{
    const auto it = threadStopIdToIdx_.find(id);
    REQUIRE(it != threadStopIdToIdx_.end(), "Invalid thread stop id " << id);
    return at(it->second);
}

const ThreadStop& ThreadStopSequence::byId(TOid id) const
{
    const auto it = threadStopIdToIdx_.find(id);
    REQUIRE(it != threadStopIdToIdx_.end(), "Invalid thread stop id " << id);
    return at(it->second);
}

std::vector<TOid> ThreadStopSequence::stopIds() const
{
    std::vector<TOid> stopIds;

    for (const auto& threadStop: threadStops_) {
        REQUIRE(threadStop.stopId, "No stop id for thread stop " << threadStop.id);
        stopIds.push_back(threadStop.stopId);
    }

    return stopIds;
}

TOIds ThreadStopSequence::threadStopsWithoutStopId() const {
    TOIds threadStopIds;

    for (const auto& threadStop: threadStops_) {
        if (!threadStop.stopId) {
            ASSERT(threadStop.id);
            threadStopIds.insert(threadStop.id);
        }
    }

    return threadStopIds;
}

TOIds ThreadStopSequence::threadStopsWithoutAttrs() const {
    TOIds threadStopIds;

    for (const auto& threadStop: threadStops_) {
        if (threadStop.attrs.empty()) {
            ASSERT(threadStop.id);
            threadStopIds.insert(threadStop.id);
        }
    }

    return threadStopIds;
}

namespace {

std::string threadStopToString(const ThreadStop& threadStop)
{
    std::stringstream out;

    out << "ThreadStop {"
        << "id=" << threadStop.id << ", "
        << "stopId=" << threadStop.stopId;

    if (threadStop.attrs[ATTR_TRANSPORT_THREAD_STOP_NO_EMBARKATION]) {
        out << ", " << ATTR_TRANSPORT_THREAD_STOP_NO_EMBARKATION;
    }

    if (threadStop.attrs[ATTR_TRANSPORT_THREAD_STOP_NO_DISEMBARKATION]) {
        out << ", " << ATTR_TRANSPORT_THREAD_STOP_NO_DISEMBARKATION;
    }

    out << "}";

    return out.str();
}

} // namespace

std::string ThreadStopSequence::toString() const {
    return '[' + common::join(threadStops_, threadStopToString, ',') + ']';
}

} // namespace wiki
} // namespace maps
