#pragma once

#include "maps/wikimap/mapspro/libs/controller/include/basecontroller.h"
#include "maps/wikimap/mapspro/libs/controller/include/async_control_support.h"
#include "maps/wikimap/mapspro/services/editor/src/observers/observer.h"
#include "maps/wikimap/mapspro/services/editor/src/view.h"
#include "maps/wikimap/mapspro/services/editor/src/collection.h"
#include "maps/wikimap/mapspro/services/editor/src/topo_storage.h"
#include "maps/wikimap/mapspro/services/editor/src/objectvisitor.h"
#include "maps/wikimap/mapspro/services/editor/src/topological/topological_callbacks.h"

#include <string>
#include <sstream>

namespace maps {
namespace wiki {

namespace topo {

class Editor;

} // namespace topo

class ObjectsCache;
class Context;
class TopologyEditContext;
class BranchContext;

class SaveObject;

template<>
struct controller::ResultType<SaveObject>
{
    GeoObjectCollection collection;
    std::shared_ptr<ObjectsCache> cache;
    std::string taskName;
    Token token;
    TUid userId;
};

class SaveObject: public controller::BaseController<SaveObject>
{
public:
    enum class IsLocalPolicy
    {
        Manual,
        Auto
    };
    enum class StickPolygonsPolicy
    {
        On,
        Off
    };

    struct Request
    {
        std::string dump() const;
        TUid userId() const { return userContext.uid(); }

        UserContext userContext;
        bool force;
        TBranchId branchId;
        ObjectsUpdateDataCollection objectsUpdateData;
        ObjectsEditContextsPtrMap objectsEditContexts;
        IsLocalPolicy isLocalPolicy;
        boost::optional<TId> feedbackTaskId;
        StringMap commitAttributes;
        std::string requestBody;
        StickPolygonsPolicy stickPolygonsPolicy = StickPolygonsPolicy::Off;
    };

    SaveObject(const ObserverCollection& observers, const Request& request,
               taskutils::TaskID asyncTaskID = 0);

    ~SaveObject() override;

    static const std::string& taskName();
    std::string printRequest() const override;

    static controller::AsyncTaskControlData asyncTaskControlData(const Request&);

protected:

    class ObjectSaver : public ObjectProcessor
    {
    public:
        ObjectSaver(
                ObjectsDataMap& newObjects,
                Context& context,
                const TopologyEditContext* topologyEditContext)
            : newObjects_(newObjects)
            , context_(context)
            , topologyEditContext_(topologyEditContext)
        {}

        void processGeoObject(GeoObject* obj) override;
        void processArealObject(ArealObject* obj) override;

        void processJunction(Junction*) override
        {
            REQUIRE(false, "Direct junction update is prohibited");
        }

        void processLinearElement(LinearElement*) override
        {
            REQUIRE(false, "Direct linear element update is prohibited");
        }

        void processLineObject(LineObject* obj) override;
        void processPointObject(PointObject* obj) override;
        void processComplexObject(ComplexObject* obj) override;
        void processAttrObject(AttrObject* obj) override;
        void processRelationObject(RelationObject* obj) override;
        void processModel3dObject(Model3dObject* obj) override;

        Context& context() { return context_; }

        const TopologyEditContext* topologyEditContext() const { return topologyEditContext_; }

    private:
        void saveObjectGeometry(GeoObject* obj);
        void saveObjectRelations(
            GeoObject* obj,
            const RelativesDiffByRoleMap& diff,
            RelationType relationType);

        ObjectsDataMap& newObjects_;
        Context& context_;
        const TopologyEditContext* topologyEditContext_;
    };

    void control() override;
    void doEditAndSave(maps::wiki::BranchContext&, CommitContext& commitContext);

    void preliminaryGabaritsCheck(Context&);
    void prepareObjectsDataForUpdate(Context&);
    void checkUpdateRevisionIds(const ObjectsUpdateDataCollection&);
    void loadExistingObjects(Context&);
    void createNewObjects(Context&);

    void prepareObjectEditContexts(Context&, const ObjectIdToParentIdsMap&);
    ObjectEditContextPtr editContextFromParents(
        const UniqueId& objectId,
        const ObjectIdToParentIdsMap& objectIdToParentIdsMap,
        const ObjectsEditContextsPtrMap& editContexts);

    void collectNewLinearElementsData();

    void saveTopologicalObjects(ObjectSaver& saver);
    void save(Context&);

    void splitArealObjects(Context&);
    void splitLineObjects(Context&);

    const ObserverCollection& observers_;

    Request request_;

    ObjectsDataMap newObjectsData_;

    TOid newLinearElementSourceIdGen_;
    NewLinearElementsDataMap newLinearElementsData_;

    bool createdNewPrimaryObjects_;

    TOIds modifiedGeomObjectIds_;
    std::unique_ptr<TopologyEditContext> topologyEditContext_;

    ObjectsUpdateDataCollection curObjectsUpdateData_;
    ObjectsEditContextsPtrMap curObjectsEditContexts_;
};

} // namespace wiki
} // namespace maps
