#include "saveobject_operations.h"

#include <maps/wikimap/mapspro/services/editor/src/context.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/category_traits.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/objects_cache.h>
#include <maps/wikimap/mapspro/services/editor/src/relations_manager.h>
#include <maps/wikimap/mapspro/services/editor/src/srv_attrs/ad.h>
#include <maps/wikimap/mapspro/services/editor/src/srv_attrs/calc.h>

#include <maps/wikimap/mapspro/services/editor/src/geom.h>
#include <maps/wikimap/mapspro/services/editor/src/utils.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/view_utils.h>

#include <maps/wikimap/mapspro/libs/views/include/query_builder.h>
#include <yandex/maps/wiki/configs/editor/categories.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <geos/geom/Envelope.h>

#include <set>
#include <vector>

namespace maps::wiki {

class Context;
class GeoObjectCollection;

namespace {

const StringVec LEVEL_KINDS_TO_CONSIDER_ORDERED =
    {
        AD_LEVEL_KIND_2_STATE,
        AD_LEVEL_KIND_1_COUNTRY
    };

const double ROAD_BY_COUNTRY_OVERLAP_FACTOR = 0.9;
const double MIN_RD_DX = 0.001;
const double MIN_RD_DY = 0.001;

std::vector<ObjectPtr>
newNmAttrObjects(TOid namedObjectId, ObjectsCache& cache)
{
    auto names = nmAttrObjects(cache.getExisting(namedObjectId), cache);
    names.erase(
        std::remove_if(names.begin(), names.end(),
            [](const ObjectPtr& obj) { return !obj->isCreated(); }),
        names.end());
    return names;
}

bool
updateIsLocalAndLang(ObjectPtr nmObj, const StringSet& localLangs,
    const std::string& nmIsLocalAttr, const std::string& nmLangAttr)
{
    auto objLang = nmObj->attributes().value(nmLangAttr);
    if (objLang == NO_LANG) {
        WIKI_REQUIRE(!localLangs.empty(), ERR_NO_LANG,
                "Can't determine default language to replace no_lang.");
        nmObj->attributes().setValue(nmLangAttr, *localLangs.begin());
        nmObj->attributes().setValue(nmIsLocalAttr, TRUE_VALUE);
        return true;
    }
    auto newValue = localLangs.count(nmObj->attributes().value(nmLangAttr))
            ? TRUE_VALUE
            : FALSE_VALUE;
    auto oldValue = nmObj->attributes().value(nmIsLocalAttr);
    if (oldValue.empty() == newValue.empty()) {
        return false;
    }
    nmObj->attributes().setValue(nmIsLocalAttr, newValue);
    return true;
}

struct ContianingAds
{
    std::string levelKind;
    TOIds adIds;
};

std::vector<ContianingAds>
findContainingAds(
    const ObjectPtr& object, ObjectsCache& cache,
    const ObjectEditContext* editContext,
    const StringVec& levelKinds)
{
    const auto& category = object->category();
    bool isSimpleGeom = isSimpleGeomCategory(category);

    const auto& viewCenter = editContext->viewCenter();
    Geom viewCenterBox(
            createGeom(
                viewCenter.x() - CALCULATION_TOLERANCE,
                viewCenter.y() - CALCULATION_TOLERANCE,
                viewCenter.x() + CALCULATION_TOLERANCE,
                viewCenter.y() + CALCULATION_TOLERANCE,
                SpatialRefSystem::Mercator));
    std::vector<Geom> geoms;
    if (object->isCreated() && !isSimpleGeom) {
        if (!isDirectGeomMasterCategory(category) &&
            !isIndirectGeomMasterCategory(category)) {
                return {};
        }
        auto slaveGeomPartRoles = object->category().slaveRoleIds(roles::filters::IsGeom);
        for (const auto& rel : object->slaveRelations().range(slaveGeomPartRoles)) {
            if (isSimpleGeomCategory(rel.categoryId())) {
                geoms.push_back(rel.relative()->geom());
            } else {
                auto partGeomPartRoles = rel.relative()->category().slaveRoleIds(roles::filters::IsGeom);
                for (const auto& partRel : rel.relative()->slaveRelations().range(partGeomPartRoles)) {
                    geoms.push_back(partRel.relative()->geom());
                }
            }
        }
        if (geoms.empty() && editContext != nullptr) {
            geoms.push_back(viewCenterBox);
        }
    } else {
        // Here object is simple and we may use it's original geometry
        // or it is old complex and we have to use bbox geometry instead
        geoms.push_back(isSimpleGeom
            ? object->geom()
            : (object->envelope().isNull()
                ? viewCenterBox
                : Geom(createGeom(object->envelope(), SpatialRefSystem::Mercator)))
            );
    }
    std::vector<ContianingAds> result;
    for (const auto& levelKind : levelKinds) {
        const auto adIds = findContainingAdByView(geoms,
            object->envelope().isNull()
                ? *viewCenterBox->getEnvelopeInternal()
                :  object->envelope(),
            {levelKind}, {}, true, cache.workView());
        TOIds withLocalLangs;
        for (const auto& adId : adIds) {
            if (!localLangs(adId, cache).empty()) {
                withLocalLangs.insert(adId);
            }
        }
        if (!withLocalLangs.empty()) {
            result.emplace_back(
                ContianingAds {
                    levelKind,
                    withLocalLangs
                }
            );
            if (withLocalLangs.size() == 1) {
                break;
            }
        }
    }
    return result;
}

TOid
findAdInMastersHierarchy(const GeoObject* object, const std::string& levelKind)
{
    TOid masterAdId = 0;
    std::deque<const GeoObject*> toVisit;
    std::unordered_set<TOid> visited;
    toVisit.push_back(object);
    visited.insert(object->id());
    while (!toVisit.empty()) {
        const auto* curObject = toVisit.front();
        toVisit.pop_front();
        const auto& objAttrs = curObject->attributes();
        if (objAttrs.value(ATTR_AD_LEVEL_KIND) == levelKind &&
            objAttrs.value(ATTR_AD_DISP_CLASS) != ATTR_VALUE_DISP_CLASS_IGNORE)
        {
            if (masterAdId) {
                return 0;
            }
            masterAdId = curObject->id();
        }
        for (const auto& rel : curObject->masterRelations().range()) {
            if (!visited.contains(rel.id()) && rel.roleId() != ROLE_EXCLUSION) {
                toVisit.push_back(rel.relative());
                visited.insert(rel.id());
            }
        }
    }
    return masterAdId;
}

void
correctIsLocalAndLang(const ObjectPtr& object, ObjectsCache& cache, const ObjectEditContext* editContext)
{
    const auto& categoryId = object->categoryId();
    if (!isNamedCategory(categoryId)) {
        return;
    }
    auto newNmObjects = newNmAttrObjects(object->id(), cache);
    if (newNmObjects.empty()) {
        return;
    }
    const auto& objNmAttrId = nameAttrId(categoryId);
    const auto& nmIsLocalAttr = findAttrColumnNameBySuffix(objNmAttrId, NM_IS_LOCAL_SUFFIX);
    const auto& nmLangAttr = findAttrColumnNameBySuffix(objNmAttrId, NM_LANG_SUFFIX);
    bool hasNewLangs = object->isCreated();
    bool hasNoLang =
        newNmObjects.end() !=
        std::find_if(newNmObjects.begin(), newNmObjects.end(),
                    [&](ObjectPtr nmObject) {
                        return nmObject->attributes().value(nmLangAttr) == NO_LANG;
                    });
    StringSet localLangs;
    if (object->revision().valid()) {
        ObjectsCache prevCache(cache.branchContext(), cache.headCommitId());
        auto objectPrevLangs = langs(object->id(), prevCache);
        localLangs = maps::wiki::localLangs(object->id(), prevCache);
        for (auto nmObject : newNmObjects) {
            const auto& newNameLang = nmObject->attributes().value(nmLangAttr);
            auto newNameIsLocal = !nmObject->attributes().value(nmIsLocalAttr).empty();
            if (!objectPrevLangs.count(newNameLang) ||
                (newNameIsLocal == !localLangs.count(newNameLang)))
            {
                hasNewLangs = true;
                break;
            }
        }
    }
    if (hasNewLangs || hasNoLang) {
        const auto allContainigAds = findContainingAds(
            object,
            cache,
            editContext,
            hasNoLang
                ? StringVec {AD_LEVEL_KIND_1_COUNTRY}
                : LEVEL_KINDS_TO_CONSIDER_ORDERED);
        for (const auto& containigAds : allContainigAds) {
            TOid uniqueAdId = 0;
            if (containigAds.adIds.size() == 1) {
                uniqueAdId = *containigAds.adIds.begin();
            } else if (!containigAds.adIds.empty()) {
                const auto adIdByHierarchy =
                    findAdInMastersHierarchy(object.get(), containigAds.levelKind);
                if (containigAds.adIds.contains(adIdByHierarchy)) {
                    uniqueAdId = adIdByHierarchy;
                }
            }
            if (uniqueAdId) {
                StringSet adLocalLangs = maps::wiki::localLangs(uniqueAdId, cache);
                localLangs.insert(adLocalLangs.begin(), adLocalLangs.end());
                break;
            }
        }
    }
    if (localLangs.empty() && !hasNoLang) {
        return;
    }
    bool someUpdated = false;
    for (auto nmObject : newNmObjects) {
        someUpdated =
            updateIsLocalAndLang(nmObject, localLangs, nmIsLocalAttr, nmLangAttr) ||
            someUpdated;
    }
    if (someUpdated) {
        readTableAttributesFromSlaveInfos(cache, object->id());
    }
}

std::string
joinIds(const TOIds& oids)
{
    return common::join(oids, ',');
}

const GeoObject*
findParentCountry(const GeoObject* ad)
{
    const auto& levelKind = ad->attributes().value(ATTR_AD_LEVEL_KIND);
    if (levelKind == AD_LEVEL_KIND_1_COUNTRY) {
        return ad;
    }
    const auto& masterRels = ad->masterRelations().range(ROLE_CHILD);
    return
        masterRels.empty()
        ? nullptr
        : findParentCountry(masterRels.begin()->relative());
}

void
checkCountryEnvelopeContainsRdEnvelope(
    const geos::geom::Envelope& rdEnvelope, const ObjectPtr& rd,
    const geos::geom::Envelope& adEnvelope, const GeoObject* ad)
{
    WIKI_REQUIRE(adEnvelope.contains(rdEnvelope),
        ERR_RD_OUTSIDE_PARENT, "Road: " << rd->id() << " is outside of ad: " << ad->id());
    const double AREA_PART = rdEnvelope.getArea() * ROAD_BY_COUNTRY_OVERLAP_FACTOR;
    double area = 0;
    for (const auto& fc : ad->slaveRelations().range(ROLE_PART)) {
        geos::geom::Envelope intersection;
        fc.relative()->envelope().intersection(rdEnvelope, intersection);
        area += intersection.getArea();
        if (area >= AREA_PART) {
            return;
        }
    }
    WIKI_REQUIRE(area >= AREA_PART,
        ERR_RD_OUTSIDE_PARENT, "Road: " << rd->id() << " is outside of ad: " << ad->id());
}

void
checkRdDoesntCrossCountryBorder(
    TOid adId,
    const ObjectPtr& rd,
    const BranchContext& branchCtx,
    const TOIds& addedElements,
    const TOIds& removedElements)
{
    auto branchId = branchCtx.branch.id();

    auto check = [&] (std::string whereClause, bool needRdToEl) {
        views::QueryBuilder qb(branchId);
        qb.selectFields("rdel.id");
        qb.fromTable(views::TABLE_OBJECTS_L, "rdel", "adel");
        qb.fromTable(views::TABLE_OBJECTS_R, "adtofc", "fctoel");
        if (needRdToEl) {
            qb.fromTable(views::TABLE_OBJECTS_R, "rdtoel");
        }
        qb.whereClause(std::move(whereClause));

        WIKI_REQUIRE(
            branchCtx.txnView().exec(qb.query() + " LIMIT 1").empty(),
            ERR_RD_CROSSES_AD_BORDER,
            "Some road: " << rd->id() << " elements cross ad: " << adId << " borders.");
    };

    check(
        " rdtoel.domain_attrs -> 'rel:slave'='rd_el'"
        " AND adel.service_attrs ? 'ad_el:level_kind_1'"
        " AND adel.domain_attrs ? 'cat:ad_el' "
        " AND rdtoel.slave_id=rdel.id AND rdtoel.master_id=" + std::to_string(rd->id()) +
        " AND adtofc.master_id=" + std::to_string(adId) +
        " AND adtofc.slave_id=fctoel.master_id AND fctoel.slave_id=adel.id"
        " AND ST_Intersects(rdel.the_geom, adel.the_geom)" +
        (removedElements.empty()
            ? s_emptyString
            : " AND rdel.id NOT IN(" + joinIds(removedElements) + ")"),
        true);

    if (!addedElements.empty()) {
        check(
            " adel.service_attrs ? 'ad_el:level_kind_1'"
            " AND adel.domain_attrs ? 'cat:ad_el' "
            " AND adtofc.master_id=" + std::to_string(adId) +
            " AND adtofc.slave_id=fctoel.master_id AND fctoel.slave_id=adel.id"
            " AND ST_Intersects(rdel.the_geom, adel.the_geom)"
            " AND rdel.id IN (" + joinIds(addedElements) + ")",
        false);
    }
}

void
checkRdIntersectsParentObject(
    const geos::geom::Envelope& rdEnvelope,
    const ObjectPtr& rd,
    const ObjectPtr& parentObj,
    const BranchContext& branchCtx,
    const TOIds& addedElements,
    const TOIds& removedElements)
{
    auto branchId = branchCtx.branch.id();
    ASSERT(branchId == revision::TRUNK_BRANCH_ID);

    auto isEmpty = [&](const std::string& query) {
        return branchCtx.txnView().exec(query + " LIMIT 1").empty();
    };

    std::stringstream rdBox;
    rdBox.precision(DOUBLE_FORMAT_PRECISION);
    const double dXFix = rdEnvelope.getWidth() < MIN_RD_DX ? MIN_RD_DX : 0;
    const double dYFix = rdEnvelope.getHeight() < MIN_RD_DY ? MIN_RD_DY : 0;
    rdBox << "ST_SetSRID(ST_MakeBox2D("
      << "ST_Point(" << (rdEnvelope.getMinX() - dXFix) << "," << (rdEnvelope.getMinY() - dYFix) <<  "),"
      << "ST_Point("<< (rdEnvelope.getMaxX() + dXFix) << "," << (rdEnvelope.getMaxY() + dYFix) << ")),3395)";
    const auto& rdBoxClause = rdBox.str();
    const std::string contourGeomQuery =
        "SELECT ST_Intersection(the_geom, " + rdBoxClause + ") g "
        " FROM contour_objects_geom WHERE object_id="
        + std::to_string(parentObj->id()) +
        " AND ST_Intersects(the_geom, " + rdBoxClause + ")";
    const std::string intersectsClause = " AND ST_Intersects(rdel.the_geom,  cgeom.g)";

    auto commonQueryBuilder = [&] {
        views::QueryBuilder qb(branchId);
        qb.with("cgeom", contourGeomQuery);
        qb.selectFields("rdel.id");
        qb.fromTable(views::TABLE_OBJECTS_L, "rdel");
        qb.fromWith("cgeom");
        return qb;
    };

    if (!addedElements.empty()) {
        auto qb = commonQueryBuilder();
        qb.whereClause(
            "rdel.id IN (" + joinIds(addedElements) + ")" +
            intersectsClause);
        if (!isEmpty(qb.query())) {
            return;
        }
    }

    auto qb = commonQueryBuilder();
    qb.fromTable(views::TABLE_OBJECTS_R, "rdtoel");
    qb.whereClause(
        "rdtoel.slave_id=rdel.id AND rdtoel.master_id=" + std::to_string(rd->id()) +
        intersectsClause +
        (removedElements.empty()
            ? std::string()
            : " AND NOT rdel.id IN (" + joinIds(removedElements) + ")"));

    WIKI_REQUIRE(
        !isEmpty(qb.query()),
        rd->categoryId() == CATEGORY_RD ? ERR_RD_OUTSIDE_PARENT : ERR_SPORT_TRACK_OUTSIDE_PARENT,
        rd->categoryId() << ": " << rd->id() << " is outside of parent: " << parentObj->id());
}

void
validateAssociationOfRdWithParent(
    const ObjectPtr& rd,
    const std::set<std::string>& rolesAssociation,
    const std::string& rolePart,
    Context& context)
{
    auto& cache = context.cache();
    auto mastersDiff = rd->masterRelations().diff();
    if (mastersDiff.added.empty()) {
        return;
    }
    TOid newMasterId = 0;
    for (const auto& rel : mastersDiff.added) {
        if (common::isIn(rel.roleId(), rolesAssociation)) {
            newMasterId = rel.id();
            break;
        }
    }
    if (!newMasterId) {
        return;
    }
    auto master = cache.getExisting(newMasterId);
    auto rdEnvelope = rd->envelope();
    TOIds addedElements;
    TOIds removedElements;
    for (const auto& rel : rd->slaveRelations().diff().added) {
        if (rel.roleId() == rolePart) {
            addedElements.insert(rel.id());
            auto rdElEnvelope = rel.relative()->envelope();
            rdEnvelope.expandToInclude(&rdElEnvelope);
        }
    }
    for (const auto& rel : rd->slaveRelations().diff().deleted) {
        if (rel.roleId() == rolePart) {
            removedElements.insert(rel.id());
        }
    }

    if (rd->categoryId() == CATEGORY_SPORT_TRACK ||
        master->categoryId() != CATEGORY_AD)
    {
        checkRdIntersectsParentObject(
            rdEnvelope, rd, master, cache.branchContext(),
            addedElements, removedElements);
        return;
    }

    ASSERT(master->categoryId() == CATEGORY_AD);
    const auto* country = findParentCountry(master.get());
    WIKI_REQUIRE(country,
        ERR_NO_COUNTRY_IN_AD_HIERARCHY,
        "Ad: " << master->id() << " is not in any country hierachy.");
    auto countryEnvelope = country->envelope();
    checkCountryEnvelopeContainsRdEnvelope(rdEnvelope, rd, countryEnvelope, country);
    checkRdDoesntCrossCountryBorder(
        country->id(), rd, cache.branchContext(),
        addedElements, removedElements);
    if(master->id() != country->id()) {
        checkRdIntersectsParentObject(
            rdEnvelope, rd, master, cache.branchContext(),
            addedElements, removedElements);
    }
}

void
associateRdWithAd(const ObjectPtr& obj, Context& context)
{
    auto& cache = context.cache();
    if (cache.branchContext().branch.id() != revision::TRUNK_BRANCH_ID) {
        return;
    }

    if (!obj->isCreated() || !obj->masterRelations().range().empty()) {
        if (obj->categoryId() == CATEGORY_RD) {
            validateAssociationOfRdWithParent(
                obj,
                {ROLE_ASSOCIATED_WITH, ROLE_RD_ASSOCIATED_WITH},
                ROLE_PART,
                context);
        } else if (obj->categoryId() == CATEGORY_SPORT_TRACK) {
            validateAssociationOfRdWithParent(
                obj,
                {ROLE_SPORT_TRACK_ASSOCIATED_WITH},
                ROLE_SPORT_TRACK_PART,
                context);
        }
        return;
    }

    geos::geom::Envelope rdEnvelope;
    const auto& range = obj->slaveRelations().range(ROLE_PART);
    if (range.empty()) {
        return;
    }
    std::vector<Geom> elementsGeoms;
    for (const auto& rdElRel : range) {
        elementsGeoms.push_back(rdElRel.relative()->geom());
        rdEnvelope.expandToInclude(rdElRel.relative()->geom()->getEnvelopeInternal());
    }
    auto adIds = findContainingAdByView(
            elementsGeoms, rdEnvelope, {AD_LEVEL_KIND_4_LOCALITY}, {}, false, cache.workView());
    if (adIds.size() == 1) {
        cache.relationsManager().createRelation(*adIds.begin(), obj->id(), ROLE_ASSOCIATED_WITH);
    }
}

TOid
threadRouteId(const GeoObject* thread)
{
    if (thread->isDeleted()) {
        return 0;
    }
    const auto& routesRange =
        thread->masterRelations().range(ROLE_ASSIGNED_THREAD);
    ASSERT(routesRange.size() == 1);
    return routesRange.begin()->id();
}

void
updateStopRouteRelations(const GeoObject* stop, Context& context, TOid primaryThreadId)
{
    if (stop->isDeleted() || stop->slaveRelations().diff().empty()) {
        return;
    }
    TOIds stopRoutes;
    for (const auto& curRouteRel :  stop->masterRelations().range(ROLE_ASSIGNED)) {
        stopRoutes.insert(curRouteRel.id());
    }
    TOIds stopThreadsRoutes;
    bool stopIsInPrimaryThread = false;
    for (const auto& rel : stop->slaveRelations().range(ROLE_ASSIGNED_THREAD_STOP)) {
        ASSERT(rel.relative()->masterRelations().range(ROLE_PART).size() == 1);
        const auto thread = rel.relative()->masterRelations().range(ROLE_PART).begin()->relative();
        stopIsInPrimaryThread |= (primaryThreadId == thread->id());
        stopThreadsRoutes.insert(threadRouteId(thread));
    }
    auto& cache = context.cache();
    TOid threadRoute = threadRouteId(cache.getExisting(primaryThreadId).get());
    if (stopIsInPrimaryThread) {
        if (stopRoutes.count(threadRoute)) {
            return;
        }
        cache.relationsManager().createRelation(threadRoute, stop->id(), ROLE_ASSIGNED);
    } else {
        if (!stopRoutes.count(threadRoute)) {
            return;
        }
        if (stopThreadsRoutes.count(threadRoute)) {
            return;
        }
        cache.relationsManager().deleteRelation(threadRoute, stop->id(), ROLE_ASSIGNED);
    }
}

boost::optional<TOid>
findPrimaryThreadId(GeoObjectCollection& collection)
{
    for (auto& obj : collection) {
        if (isTransportThread(obj->categoryId()) &&
            obj->primaryEdit()) {
            return obj->id();
        }
    }
    return boost::none;
}

void
updateThreadStationToRouteAssignment(Context& context, GeoObjectCollection& collection)
{
    auto primaryThreadId = findPrimaryThreadId(collection);
    if (!primaryThreadId) {
        return;
    }
    auto thread = collection.getById(*primaryThreadId);
    ASSERT(thread);
    if (thread->categoryId() == CATEGORY_TRANSPORT_METRO_THREAD) {
        return;
    }
    for (auto& obj : collection) {
        if (obj->categoryId() != CATEGORY_TRANSPORT_THREAD_STOP
            || obj->masterRelations().diff().empty()) {
            continue;
        }
        for (const auto& added : obj->masterRelations().diff().added) {
            if (isTransportStation(added.categoryId())) {
                updateStopRouteRelations(added.relative(), context, *primaryThreadId);
            }
        }
        for (const auto& deleted : obj->masterRelations().diff().deleted) {
            if (isTransportStation(deleted.categoryId())) {
                updateStopRouteRelations(deleted.relative(), context, *primaryThreadId);
            }
        }
    }
}

} // namespace

void calculateNmIsLocalAndLang(Context& context, GeoObjectCollection& collection)
{
    for (auto& obj : collection) {
        correctIsLocalAndLang(obj, context.cache(), context.editContext(UniqueId(obj->id())));
    }
}

void createDefaultRelations(Context& context, GeoObjectCollection& collection)
{
    for (auto& obj : collection) {
        if (common::isIn(obj->categoryId(), {CATEGORY_RD, CATEGORY_SPORT_TRACK})) {
            associateRdWithAd(obj, context);
        }
    }
    updateThreadStationToRouteAssignment(context, collection);
}

void resetPoiImportSourceWithPermalink(GeoObjectCollection& collection)
{
    for (auto& obj : collection) {
        if (isPermalinkReset(obj.get())) {
            if (obj->attributes().isDefined(ATTR_SYS_IMPORT_SOURCE)) {
                obj->attributes().setValue(ATTR_SYS_IMPORT_SOURCE, s_emptyString);
                obj->setModifiedAttr();
            }
            if (obj->attributes().isDefined(ATTR_SYS_IMPORT_SOURCE_ID)) {
                obj->attributes().setValue(ATTR_SYS_IMPORT_SOURCE_ID, s_emptyString);
                obj->setModifiedAttr();
            }
        }
    }
}

} // namespace maps::wiki
