#include "set_default_speed_limit.h"

#include <maps/wikimap/mapspro/services/editor/src/srv_attrs/ad.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/category_traits.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/linear_element.h>
#include <maps/wikimap/mapspro/services/editor/src/objects_cache.h>

#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>

namespace maps {
namespace wiki {
namespace {
const double MAXIMUN_ROAD_ELEMENT_BY_CITIES_OVERLAP_FACTOR = 0.1;
const double RD_EL_OVERLAP_BUFFER_WIDTH = 0.1;

std::string
defaultSpeedLimit(Context& context, const ObjectPtr& linearElement)
{
    Geom leGeomBuffer = linearElement->geom().createBuffer(RD_EL_OVERLAP_BUFFER_WIDTH);
    auto& work = context.cache().workView();
    std::string query =
        "SELECT json_agg(object_id) as ids,  "
        " ST_UnaryUnion(ST_Collect(the_geom)) as geom, "
        " domain_attrs->'ad:level_kind' as lk"
        " FROM contour_objects_geom WHERE "
        " ST_Intersects(the_geom,  ST_GeomFromWKB('" + work.esc_raw(leGeomBuffer.wkb()) + "', 3395))"
        " AND "
            "(domain_attrs @> hstore('ad:level_kind', '4')"
            " OR "
            "domain_attrs @> hstore('ad:level_kind', '1'))"
        " AND "
        " (NOT domain_attrs ? 'ad:municipality') "
        " AND domain_attrs ? 'cat:ad'"
        " GROUP BY domain_attrs->'ad:level_kind'";
    auto rows = work.exec(query);
    if (rows.empty()) {
        return s_emptyString;
    }
    TOid adId = 0;
    bool isInCities = false;
    for (const auto& row : rows) {
        const auto& lk = row["lk"].as<std::string>();
        const auto& jsonIdsArray = json::Value::fromString(row["ids"].c_str());
        ASSERT(jsonIdsArray.isArray());
        if (lk == AD_LEVEL_KIND_1_COUNTRY) {
            ASSERT(jsonIdsArray.size());
            adId = jsonIdsArray[0].as<TOid>();
            for (const auto& otherId : jsonIdsArray) {
                if (adId != otherId.as<TOid>()) {
                    return s_emptyString;
                }
            }
        } else {
            Geom citiesGeom(row["geom"]);
            std::unique_ptr<geos::geom::Geometry> intersection(citiesGeom->intersection(leGeomBuffer.geosGeometryPtr()));
            isInCities =
                leGeomBuffer->getArea() * MAXIMUN_ROAD_ELEMENT_BY_CITIES_OVERLAP_FACTOR
                < intersection->getArea();
        }
    }
    if (!adId) {
        return s_emptyString;
    }
    const auto& ad = context.cache().getExisting(adId);
    const auto& isocode = ad->attributes().value(ATTR_AD_ISOCODE);

    const auto& countries = cfg()->editor()->countries();
    if (!countries.hasCountryWithISOCODE(isocode)) {
        return s_emptyString;
    }
    const auto& country = countries.findByISOCODE(isocode);
    auto curFC = boost::lexical_cast<size_t>(linearElement->attributes().value(ATTR_RD_EL_FC));
    auto speedLimits = country.fcRangeSpeedLimits(curFC);
    if (!speedLimits) {
        return s_emptyString;
    }
    return std::to_string(
        isInCities
        ? speedLimits->inLocality()
        : speedLimits->outLocality());
}
}//namepsace

void setDefaultSpeedLimit(Context& context)
{
    if (context.cache().branchContext().branch.id() != revision::TRUNK_BRANCH_ID) {
        return;
    }
    bool primaryIsNewRoadElement = false;
    auto filter = [&](const GeoObject* object) -> bool {
        if (object->categoryId() != CATEGORY_RD_EL
            || object->revision().valid()) {
            return false;
        }
        primaryIsNewRoadElement =
            primaryIsNewRoadElement || object->primaryEdit();
        return
            object->primaryEdit() || as<LinearElement>(object)->isPartOfPrimary();
    };
    auto collection = context.cache().find(filter);
    if (!primaryIsNewRoadElement) {
        return;
    }
    for (auto& obj : collection) {
        const auto& speedLimit = obj->attributes().value(ATTR_RD_EL_SPEED_LIMIT);
        if (!speedLimit.empty()) {
            continue;
        }
        const auto& fc = obj->attributes().value(ATTR_RD_EL_FC);
        if (fc.empty()) {
            continue;
        }
        obj->attributes().setValue(ATTR_RD_EL_SPEED_LIMIT, defaultSpeedLimit(context, obj));
    }
}
} // namespace wiki
} // namespace maps
