#include "delete.h"
#include "logging.h"

#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/check_permissions.h>
#include <maps/wikimap/mapspro/services/editor/src/exception.h>

#include <maps/wikimap/mapspro/libs/acl_utils/include/moderation.h>

#include <yandex/maps/wiki/common/moderation.h>
#include <yandex/maps/wiki/social/gateway.h>

namespace maps {
namespace wiki {

namespace {

const std::string TASK_METHOD_NAME = "CommentsDelete";

} // namespace

CommentsDelete::Request::Request(UserContext userContext, TId commentId)
    : userContext(std::move(userContext))
    , commentId(commentId)
{
    CHECK_REQUEST_PARAM(commentId);
}

std::string
CommentsDelete::Request::dump() const
{
    std::stringstream ss;
    ss << " uid: " << userId()
       << " comment id: " << commentId;
    return ss.str();
}


CommentsDelete::CommentsDelete(
        const ObserverCollection&,
        const Request& request,
        taskutils::TaskID asyncTaskID)
    : controller::BaseController<CommentsDelete>(BOOST_CURRENT_FUNCTION, asyncTaskID)
    , request_(request)
{}

std::string
CommentsDelete::printRequest() const
{
    return request_.dump();
}

void
CommentsDelete::control()
{
    auto branchCtx = CheckedTrunkBranchContextFacade().acquireWrite();

    const auto& user = request_.userContext.aclUser(branchCtx);
    user.checkActiveStatus();

    social::Gateway socialGateway(branchCtx.txnSocial());
    auto comments = socialGateway.loadComments({request_.commentId});
    WIKI_REQUIRE(
        !comments.empty(), ERR_NOT_FOUND,
        "can't load comment : " << request_.commentId);
    ASSERT(comments.size() == 1);

    result_->comment = comments.front();

    const auto& moderationStatus = request_.userContext.moderationStatus(branchCtx);
    WIKI_REQUIRE(
        result_->comment->createdBy() == user.uid() ||
        (acl_utils::isModerator(moderationStatus) &&
        CheckPermissions(request_.userId(), branchCtx.txnCore()).canUserDeleteOtherUserComment()),
        ERR_FORBIDDEN,
        "User can delete only his own comments: " << request_.commentId);

    // skip checking already deleted comments
    result_->comment->deleteBy(branchCtx.txnSocial(), request_.userId());

    request_.userContext.saveActivity(
        branchCtx, social::UserActivityAction::DeleteComment, result_->comment->id());

    logDeleteComment(*result_->comment);
    result_->token = branchCtx.commit();
}

const std::string&
CommentsDelete::taskName()
{
    return TASK_METHOD_NAME;
}

} // namespace wiki
} // namespace maps
