#include "feed_params.h"

#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/exception.h>

#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/social/comment.h>
#include <yandex/maps/wiki/social/common.h>

#include <maps/wikimap/mapspro/libs/acl_utils/include/moderation.h>

#include <optional>

namespace maps {
namespace wiki {

namespace {

void
checkAuthorization(TUid uid)
{
    WIKI_REQUIRE(
        uid,
        ERR_FORBIDDEN,
        "Non-authorized access for inbox/outbox");
}

social::CommentsFeedParams
paramsForOutbox(
    TUid uid,
    TUid createdBy,
    const social::CommentTypes& commentTypes,
    social::comments::DeletedPolicy deletedPolicy)
{
    checkAuthorization(uid);
    WIKI_REQUIRE(
        uid == createdBy,
        ERR_FORBIDDEN,
        "Non-authorized access for outbox " << createdBy << " " << uid);
    social::CommentsFeedParams params;
    params.setCreatedBy(createdBy);
    params.setTypes(commentTypes);
    params.setDeletedPolicy(deletedPolicy);
    return params;
}

social::CommentsFeedParams
paramsForObject(
    TOid objectId,
    const social::CommentTypes& commentTypes,
    social::comments::DeletedPolicy deletedPolicy)
{
    social::CommentsFeedParams params;
    params.setObjectId(objectId);
    params.setTypes(commentTypes);
    params.setDeletedPolicy(deletedPolicy);
    return params;
}

social::CommentsFeedParams
paramsForFeedbackTask(
    std::optional<TId> feedbackTaskId,
    const social::CommentTypes& commentTypes,
    social::comments::DeletedPolicy deletedPolicy)
{
    social::CommentsFeedParams params;
    params.setFeedbackTaskId(feedbackTaskId);
    params.setTypes(commentTypes);
    params.setDeletedPolicy(deletedPolicy);
    return params;
}

social::CommentsFeedParams
paramsForAoi(
    TOid aoiId,
    TUid createdBy,
    const social::CommentTypes& commentTypes,
    social::comments::DeletedPolicy deletedPolicy)
{
    social::CommentsFeedParams params;
    params.setAoiId(aoiId);
    params.setTypes(commentTypes);
    params.setDeletedPolicy(deletedPolicy);
    params.setCreatedBy(createdBy);
    return params;
}

social::CommentsFeedParams
paramsForCommit(
    TCommitId commitId,
    const social::CommentTypes& commentTypes,
    social::comments::DeletedPolicy deletedPolicy)
{
    social::CommentsFeedParams params;
    params.setCommitId(commitId);
    params.setTypes(commentTypes);
    params.setDeletedPolicy(deletedPolicy);
    return params;
}

std::optional<social::Comment::Internal>
getFilterForInternalInfo(TUid uid, const Token& token)
{
    if (!uid) {
        return social::Comment::Internal::No;
    }

    auto branchCtx = CheckedTrunkBranchContextFacade().acquireReadCoreSocial(token);
    acl::ACLGateway aclGw(branchCtx.txnCore());
    auto moderationStatus = acl_utils::moderationStatus(aclGw, aclGw.user(uid));

    if (acl_utils::isYandexModerator(moderationStatus) ||
        acl_utils::isCartographer(moderationStatus))
    {
        return std::nullopt;
    }

    return social::Comment::Internal::No;
}

social::CommentsFeedParams
createParams(
    TUid uid,
    TOid objectId,
    TCommitId commitId,
    std::optional<TId> feedbackTaskId,
    TOid aoiId,
    TUid createdBy,
    const social::CommentTypes& commentTypes,
    social::comments::DeletedPolicy deletedPolicy)
{
    if (objectId) {
        return paramsForObject(objectId, commentTypes, deletedPolicy);
    }
    if (aoiId) {
        return paramsForAoi(aoiId, createdBy, commentTypes, deletedPolicy);
    }
    if (createdBy) {
        return paramsForOutbox(uid, createdBy, commentTypes, deletedPolicy);
    }
    if (commitId) {
        return paramsForCommit(commitId, commentTypes, deletedPolicy);
    }
    if (feedbackTaskId) {
        return paramsForFeedbackTask(feedbackTaskId, commentTypes, deletedPolicy);
    }
    THROW_WIKI_LOGIC_ERROR(
        ERR_BAD_REQUEST,
        "Empty request parameters oid,aoi,created-by");
}

} // namespace

FeedParams::FeedParams(TUid uid, social::CommentsFeedParams params)
    : uid_(uid)
    , params_(std::move(params))
{}

FeedParams FeedParams::create(
    TUid uid,
    TOid objectId,
    TCommitId commitId,
    std::optional<TId> feedbackTaskId,
    TOid aoiId,
    TUid createdBy,
    social::CommentTypes commentTypes,
    social::comments::DeletedPolicy deletedPolicy,
    const Token& token)
{
    FeedParams feedParams{
        uid,
        createParams(uid, objectId, commitId, feedbackTaskId, aoiId,
                     createdBy, commentTypes, deletedPolicy)
    };
    feedParams.params_.setInternal(getFilterForInternalInfo(uid, token));
    return feedParams;
}

std::string FeedParams::dump() const
{
    std::stringstream ss;
    ss << " uid: " << uid_
       << " created-by: " << params_.createdBy();
    ss << " oid: " << params_.objectId()
       << " commit-id: " << params_.commitId();
    if (params_.feedbackTaskId()) {
        ss << " feedback-task-id: " << *params_.feedbackTaskId();
    }
    ss << " aoi: " << params_.aoiId();
    if (!params_.types().empty()) {
        ss << " types: " << maps::wiki::common::join(params_.types(),',');
    }
    if (params_.deletedPolicy() == social::comments::DeletedPolicy::Show) {
        ss << " (show deleted)";
    }
    return ss.str();
}

} // namespace wiki
} // namespace maps
