#include "get_feed.h"

#include "get_feed_helper.h"
#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/serialize/common.h>
#include <maps/wikimap/mapspro/services/editor/src/social_utils.h>
#include <maps/wikimap/mapspro/services/editor/src/moderation.h>
#include <maps/wikimap/mapspro/services/editor/src/exception.h>

#include <yandex/maps/wiki/common/moderation.h>
#include <yandex/maps/wiki/common/paged_result.h>
#include <yandex/maps/wiki/social/gateway.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <maps/wikimap/mapspro/libs/acl/include/check_context.h>
#include <maps/wikimap/mapspro/libs/acl/include/subject_path.h>

namespace maps {
namespace wiki {

namespace {

const size_t SUSPICIOUS_MAX_COUNT_LIMIT = 5000;

} // namespace


GetSuspiciousSocialFeed::GetSuspiciousSocialFeed(const Request& request)
    : request_(request)
{
    CHECK_REQUEST_PARAM(request.perPage);
    WIKI_REQUIRE(
        !request_.afterEventId || !request_.beforeEventId,
        ERR_BAD_REQUEST,
        "non-empty after,before request parameters");

    result_->branchId = request_.branchId;
    result_->feedType = social::FeedType::Suspicious;
    result_->subscriber = 0;

    result_->totalCount = 0;
    result_->page = 0;
    result_->perPage = request_.perPage;
}

std::string
GetSuspiciousSocialFeed::printRequest() const
{
    std::stringstream ss;
    ss << " uid: " << request_.uid
       << " branch: " << request_.branchId;
    ss << " target-uids: [";
    for (const auto& uid: request_.targetUids) {
        ss << uid << ", ";
    }
    ss << "]";
    ss << " target-mode: ";
    if (request_.targetMode) {
        ss << moderation::toAclRoleName(*request_.targetMode);
    } else {
        ss << "none";
    }
    ss << " since: ";
    if (request_.since) {
        ss << chrono::formatIsoDateTime(*request_.since);
    } else {
        ss << "none";
    }
    ss << " till: ";
    if (request_.till) {
        ss << chrono::formatIsoDateTime(*request_.till);
    } else {
        ss << "none";
    }
    ss << " actions: [";
    for (const auto& action: request_.allowedActions) {
        ss << action << ", ";
    }
    ss << "]";
    ss << " per-page: " << request_.perPage
       << " after-event-id: " << request_.afterEventId
       << " before-event-id: " << request_.beforeEventId
       << " token: " << request_.token;

    return ss.str();
}

namespace {

std::set<acl::ID> userIdsByRoleName(acl::ACLGateway& aclGw, const std::string& roleName)
{
    auto roles = aclGw.roles(acl::InclusionType::StartsWith, roleName);
    std::set<acl::ID> roleIds;
    for (const acl::Role& role : roles) {
        roleIds.emplace(role.id());
    }
    return aclGw.userIdsByRoles(roleIds);
}

std::vector<acl::ID> userIdsByMode(acl::ACLGateway& aclGw, const std::string& moderationStatus)
{
    std::vector<acl::ID> result;
    if (moderationStatus == common::MODERATION_STATUS_CARTOGRAPHER) {
        auto cartographerUserIds = userIdsByRoleName(aclGw, common::MODERATION_STATUS_CARTOGRAPHER);
        std::copy(cartographerUserIds.begin(), cartographerUserIds.end(), std::back_inserter(result));
    } else if (moderationStatus == common::MODERATION_STATUS_YANDEX_MODERATOR) {
        auto cartographerUserIds =  userIdsByRoleName(aclGw, common::MODERATION_STATUS_CARTOGRAPHER);
        auto yaModeratorUserIds = userIdsByRoleName(aclGw, common::MODERATION_STATUS_YANDEX_MODERATOR);
        for (auto userId: yaModeratorUserIds) {
            if (cartographerUserIds.count(userId) == 0) {
                result.push_back(userId);
            }
        }
    } else if (moderationStatus == common::MODERATION_STATUS_MODERATOR) {
        auto cartographerUserIds =  userIdsByRoleName(aclGw, common::MODERATION_STATUS_CARTOGRAPHER);
        auto yaModeratorUserIds = userIdsByRoleName(aclGw, common::MODERATION_STATUS_YANDEX_MODERATOR);
        auto moderatorUserIds = userIdsByRoleName(aclGw, common::MODERATION_STATUS_MODERATOR);
        for (auto userId: moderatorUserIds) {
            if (yaModeratorUserIds.count(userId) == 0 && cartographerUserIds.count(userId) == 0) {
                result.push_back(userId);
            }
        }
    } else {
        THROW_WIKI_LOGIC_ERROR(
            ERR_BAD_REQUEST,
            "unsupported moderation status " << moderationStatus
        );
    }
    return result;
}

std::vector<TUid> uidsByMode(acl::ACLGateway& aclGw, const std::string& moderationStatus)
{
    auto userIds = userIdsByMode(aclGw, moderationStatus);
    auto users = aclGw.usersByIds({userIds.begin(), userIds.end()});
    std::vector<TUid> result;
    for (const auto& user: users) {
        result.push_back(user.uid());
    }
    return result;

}

} // namespace

void
GetSuspiciousSocialFeed::control()
{
    auto branchCtx = BranchContextFacade(request_.branchId)
        .acquireReadCoreSocial(request_.token);

    const acl::SubjectPath aclPath("mpro/social/suspicious-feed");
    aclPath.check(
        acl::CheckContext(request_.uid, {}, branchCtx.txnCore(), {acl::User::Status::Active}));

    social::FeedFilter filter;
    if (request_.since) {
        filter.createdAfter(*request_.since);
    }
    if (request_.till) {
        filter.createdBefore(*request_.till);
    }
    if (!request_.allowedActions.empty()) {
        filter.actionsAllowed(request_.allowedActions);
    }

    WIKI_REQUIRE(
        request_.targetUids.empty() || !request_.targetMode,
        ERR_BAD_REQUEST,
        "target-uid and target-mode have been specified simultaneously");

    if (!request_.targetUids.empty()) {
        filter.createdBy(request_.targetUids);
    }
    if (request_.targetMode) {
        acl::ACLGateway aclGw(branchCtx.txnCore());
        filter.createdBy(
            uidsByMode(aclGw, moderation::toAclRoleName(*request_.targetMode))
        );
    }

    auto feed = social::Gateway(branchCtx.txnSocial()).suspiciousFeed(
        request_.branchId, std::move(filter));


    result_->totalCount = feed.limitedCount(SUSPICIOUS_MAX_COUNT_LIMIT);

    auto processEventsMore = [&](std::pair<social::Events, social::HasMore>&& pair)
    {
        fillWithEvents(
            branchCtx,
            *result_,
            std::move(pair.first)
        );
        result_->hasMore = pair.second == social::HasMore::Yes;
    };

    if (request_.beforeEventId) {
        processEventsMore(feed.eventsBefore(request_.beforeEventId, request_.perPage));
    } else if (request_.afterEventId) {
        processEventsMore(feed.eventsAfter(request_.afterEventId, request_.perPage));
    } else {
        processEventsMore(feed.eventsHead(request_.perPage));
    }
}


} // namespace wiki
} // namespace maps
