#pragma once

#include "maps/wikimap/mapspro/services/editor/src/exception.h"
#include "maps/wikimap/mapspro/services/editor/src/objects_cache.h"
#include "maps/wikimap/mapspro/services/editor/src/observers/observer.h"
#include "maps/wikimap/mapspro/libs/controller/include/basecontroller.h"
#include "maps/wikimap/mapspro/services/editor/src/actions/objects_deleter.h"
#include "helpers.h"

#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/social/task.h>

#include <string>

namespace maps {
namespace wiki {

namespace {

const std::string RESOLUTION = "resolution";

template <typename Resolution>
Resolution
requestResolution(const json::Value& jsonObject)
{
    Resolution resolution;
    auto resolutionStr = jsonObject[RESOLUTION].toString();
    try {
        resolution = boost::lexical_cast<Resolution>(resolutionStr);
    } catch(...) {
        THROW_WIKI_LOGIC_ERROR(
            ERR_BAD_REQUEST, "invalid resolution: " << resolutionStr);
    }
    return resolution;
}

} // namespace


class ObserverCollection;

// A base class for controllers which work with comments and complaints.
template<typename Controller, typename Resolution>
class SocialModerationCommentTasksProcessor: public controller::BaseController<Controller> {
public:
    struct Request {
        Request(
            UserContext userContext,
            boost::optional<TOid> aoiId,
            Resolution resolution,
            const social::TaskIds& taskIds);

        Request(
            UserContext userContext,
            boost::optional<TOid> aoiId,
            const std::string& json);

        std::string dump() const;
        TUid userId() const { return userContext.uid(); }

        UserContext userContext;
        const boost::optional<TOid> aoiId;
        Resolution resolution;
        social::TaskIds taskIds;
        boost::optional<RevertReason> revertReason;
    };

    SocialModerationCommentTasksProcessor(
        const ObserverCollection& observers,
        const Request& request,
        taskutils::TaskID asyncTaskID = 0);

    virtual std::string printRequest() const;

protected:
    template <typename R = Resolution>
    typename std::enable_if<std::is_same<R, social::ResolveResolution>::value>::type
    processComments(
        BranchContext& branchCtx,
        const social::Tasks& tasks)
    {
        switch (request_.resolution) {
            case Resolution::Accept:
                processCommentsAcceptAndApprove(branchCtx, tasks);
                break;
            case Resolution::Edit:
                processCommentsEdit(branchCtx, tasks);
                break;
            case Resolution::Revert:
                processCommentsRevert(branchCtx);
                break;
        }
    }


    template <typename R = Resolution>
    typename std::enable_if<std::is_same<R, social::CloseResolution>::value>::type
    processComments(
        BranchContext& branchCtx,
        const social::Tasks& tasks)
    {
        switch (request_.resolution) {
            case Resolution::Approve:
                processCommentsAcceptAndApprove(branchCtx, tasks);
                break;
            case Resolution::Edit:
                processCommentsEdit(branchCtx, tasks);
                break;
            case Resolution::Revert:
                processCommentsRevert(branchCtx);
                break;
        }
    }

    void checkResult(
        Resolution resolution,
        const social::TIds& sourceTaskIds, const social::TIds& resultTaskIds) const;

    Request request_;
    const ObserverCollection& observers_;

private:
    void processCommentsEdit(
        BranchContext& branchCtx,
        const social::Tasks& tasks);

    void processCommentsAcceptAndApprove(
        BranchContext& branchCtx,
        const social::Tasks& tasks);

    void processCommentsRevert(
        BranchContext& branchCtx);

    void closeTasks(
        BranchContext& branchCtx,
        social::ResolveResolution resolveResolution,
        social::CloseResolution closeResolution) const;
};


template<typename Controller, typename Resolution>
SocialModerationCommentTasksProcessor<Controller, Resolution>::Request::Request(
        UserContext userContext,
        boost::optional<TOid> aoi,
        Resolution resolution,
        const social::TaskIds& taskIds)
    : userContext(std::move(userContext))
    , aoiId(aoi)
    , resolution(resolution)
    , taskIds(taskIds)
{
    if (aoi) {
        CHECK_REQUEST_PARAM(*aoi);
    }
    moderation::checkRequestTaskIds(taskIds);
}


template<typename Controller, typename Resolution>
SocialModerationCommentTasksProcessor<Controller, Resolution>::Request::Request(
        UserContext userContext,
        boost::optional<TOid> aoi,
        const std::string& json)
    : userContext(std::move(userContext))
    , aoiId(aoi)
{
    if (aoi) {
        CHECK_REQUEST_PARAM(*aoi);
    }

    auto jsonObject = moderation::requestJsonBody(json, Controller::taskName());
    moderation::checkJsonObject(jsonObject);

    taskIds = moderation::requestTaskIds(jsonObject);
    resolution = requestResolution<Resolution>(jsonObject);
    revertReason = moderation::requestRevertReason(jsonObject);

    WIKI_REQUIRE(
        !revertReason or resolution == Resolution::Revert,
        ERR_BAD_REQUEST,
        "Invalid request parameter: revert reason makes sense only when resolution is revert");
}


template<typename Controller, typename Resolution>
std::string
SocialModerationCommentTasksProcessor<Controller, Resolution>::Request::dump() const
{
    std::ostringstream os;
    os << " uid: " << userId();
    if (aoiId) {
        os << " aoi: " << *aoiId;
    }
    os << " resolution: " << resolution
       << " task ids: " << common::join(taskIds, ',');
    if (revertReason) {
        os << " revert reason: " << *revertReason;
    }
    return os.str();
}


template<typename Controller, typename Resolution>
SocialModerationCommentTasksProcessor<Controller, Resolution>::SocialModerationCommentTasksProcessor(
        const ObserverCollection& observers,
        const Request& request,
        taskutils::TaskID asyncTaskID)
    : controller::BaseController<Controller>(BOOST_CURRENT_FUNCTION, asyncTaskID)
    , request_(request)
    , observers_(observers)
{}


template<typename Controller, typename Resolution>
std::string
SocialModerationCommentTasksProcessor<Controller, Resolution>::printRequest() const
{
    return request_.dump();
}


template<typename Controller, typename Resolution>
void
SocialModerationCommentTasksProcessor<Controller, Resolution>::checkResult(
    Resolution resolution,
    const social::TIds& sourceTaskIds, const social::TIds& resultTaskIds) const
{
    for (auto taskId: sourceTaskIds) {
        WIKI_REQUIRE(
            resultTaskIds.count(taskId), ERR_MODERATION_CONFLICT,
            "Can not process: " << resolution << " moderation task: " << taskId);
    }
}


template<typename Controller, typename Resolution>
void
SocialModerationCommentTasksProcessor<Controller, Resolution>::processCommentsAcceptAndApprove(
    BranchContext& branchCtx,
    const social::Tasks& tasks)
{
    TOIds oids;

    moderation::checkTasksType(tasks, social::EventType::RequestForDeletion);

    for (const auto& task: tasks) {
        const auto& objectData = task.event().primaryObjectData();
        WIKI_REQUIRE(
            objectData,
            ERR_INVALID_OPERATION,
            "Can not find object data from comment task " << task.id());
        oids.insert(objectData->id());
    }

    ObjectsCache cache(branchCtx, boost::none);

    TOIds oidsToDelete;
    for (auto& obj: cache.get(oids)) {
        if (!obj->isDeleted()) {
            oidsToDelete.insert(obj->id());
        }
    }

    closeTasks(
        branchCtx,
        social::ResolveResolution::Accept,
        social::CloseResolution::Approve);

    if (oidsToDelete.empty()) {
        return;
    }

    if (ObjectsDeleter(cache).changeState(request_.userId(), oidsToDelete)) {
        cache.save(
            request_.userId(),
            oidsToDelete.size() == 1
                ? common::COMMIT_PROPVAL_OBJECT_DELETED
                : common::COMMIT_PROPVAL_GROUP_DELETED);
        controller::BaseController<Controller>::result_->token = observers_.doCommit(cache, request_.userContext);
    }
}


template<typename Controller, typename Resolution>
void
SocialModerationCommentTasksProcessor<Controller, Resolution>::processCommentsEdit(
    BranchContext& branchCtx,
    const social::Tasks& tasks)
{
    moderation::checkTasksType(tasks, social::EventType::Complaint);

    closeTasks(
        branchCtx,
        social::ResolveResolution::Edit,
        social::CloseResolution::Edit);
}


template<typename Controller, typename Resolution>
void
SocialModerationCommentTasksProcessor<Controller, Resolution>::processCommentsRevert(
    BranchContext& branchCtx)
{
    closeTasks(
        branchCtx,
        social::ResolveResolution::Revert,
        social::CloseResolution::Revert);
}


template<typename Controller, typename Resolution>
void
SocialModerationCommentTasksProcessor<Controller, Resolution>::closeTasks(
    BranchContext& branchCtx,
    social::ResolveResolution resolveResolution,
    social::CloseResolution closeResolution) const
{
    social::Gateway socialGateway(branchCtx.txnSocial());
    auto modConsole = socialGateway.superModerationConsole(request_.userId());

    if constexpr (std::is_same_v<Resolution, social::ResolveResolution>) {
        controller::BaseController<Controller>::result_->taskIds =
            modConsole.resolveTasksByUserAndCloseByRobot(request_.taskIds, resolveResolution, closeResolution);
    } else {
        controller::BaseController<Controller>::result_->taskIds =
            modConsole.resolveTasksByRobotAndCloseByUser(request_.taskIds, resolveResolution, closeResolution);
    }

    checkResult(request_.resolution, request_.taskIds, controller::BaseController<Controller>::result_->taskIds);
}


} // namespace wiki
} // namespace maps
