#include "get_stat.h"
#include "helpers.h"
#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/social_utils.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/utils.h>
#include <yandex/maps/wiki/configs/editor/category_groups.h>
#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <yandex/maps/wiki/social/gateway.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>

namespace maps::wiki {

namespace {

typedef std::map<TOid, StringSet> Aoi2CategoryGroupIds;

TOIds
loadFilteredAoiIds(pqxx::transaction_base& work, const TOIds& aoiIds)
{
    TOIds filteredAoiIds;
    if (aoiIds.empty()) {
        return filteredAoiIds;
    }

    revision::RevisionsGateway revGateway(work); // trunk
    auto snapshot = revGateway.snapshot(revGateway.headCommitId());

    auto filter = existingAoiFilter(aoiIds);
    for (const auto& revId : snapshot.revisionIdsByFilter(filter)) {
        filteredAoiIds.insert(revId.objectId());
    }
    return filteredAoiIds;
}


bool isInPolicies(
    TOid aoiId, social::ModerationMode mode,
    const std::string& categoryId,
    const moderation::RegionPolicies& policies)
{
    for (const auto& policy: policies) {
        if (policy.mode != mode or policy.aoiId != aoiId) {
            continue;
        }

        // Counters for common moderation tasks are returned in a form of an
        // empty category by lib social.
        if (categoryId.empty()) {
            if (policy.areCommonTasksPermitted) {
                return true;
            }
            continue;
        }

        auto policyCategoryIds =
            cfg()->editor()->categoryGroups().categoryIdsByGroups(policy.categoryGroupIds);

        if (policyCategoryIds.count(categoryId)) {
            return true;
        }
    }

    return false;
};


bool
hasTasksByModeAndPolicies(
    social::SuperModerationConsole& modConsole,
    social::ModerationMode mode,
    const moderation::RegionPolicies& policies)
{
    social::EventFilter eventFilter;
    eventFilter.deferred(social::Deferred::No);

    auto countsByAoiCategory = modConsole.countsByAoiCategoryId(
        mode, eventFilter, collectAoiIds(policies), cfg()->moderationTimeIntervals());
    if (countsByAoiCategory.taskCounts.empty()) {
        return false;
    }

    for (const auto& [aoiId, countsByCat] : countsByAoiCategory.taskCounts) {
        for (const auto& [cat, counts] : countsByCat) {
            if (!counts.acquired() && !counts.available()) {
                continue;
            }
            if (isInPolicies(aoiId, mode, cat, policies)) {
                return true;
            }
        }
    }
    return false;
}

} // namespace

GetSocialModerationStat::GetSocialModerationStat(const Request& request)
    : controller::BaseController<GetSocialModerationStat>(BOOST_CURRENT_FUNCTION)
    , request_(request)
{}

std::string
GetSocialModerationStat::printRequest() const
{
    std::stringstream ss;
    ss << " uid: " << request_.uid
       << " token: " << request_.token;
    return ss.str();
}

void
GetSocialModerationStat::control()
{
    result_->uid = request_.uid;
    result_->hasTasks = false;

    try {
        result_->hasTasks = hasTasks();
    } catch (const maps::Exception& ex) {
        WARN() << BOOST_CURRENT_FUNCTION << " " << ex.what();
    }
}

bool
GetSocialModerationStat::hasTasks() const
{
    auto branchCtx = BranchContextFacade()
        .acquireReadCoreSocial(request_.token);

    auto user = acl::ACLGateway(branchCtx.txnCore()).user(request_.uid);
    user.checkActiveStatus();

    const auto regionPolicies = moderation::createRegionPolicies(user);
    if (regionPolicies.empty()) {
        return false;
    }

    auto filteredAoiIds = loadFilteredAoiIds(
        branchCtx.txnCore(), moderation::collectAoiIds(regionPolicies));
    if (filteredAoiIds.empty()) {
        return false;
    }

    std::map<social::ModerationMode, moderation::RegionPolicies> modes2policies;
    for (const auto& regionPolicy : regionPolicies) {
        if (!filteredAoiIds.count(regionPolicy.aoiId)) {
            continue;
        }
        modes2policies[regionPolicy.mode].push_back(regionPolicy);
    }

    social::Gateway socialGateway(branchCtx.txnSocial());
    auto modConsole = socialGateway.superModerationConsole(request_.uid);

    for (const auto& mode2policies : modes2policies) {
        if (hasTasksByModeAndPolicies(
                modConsole, mode2policies.first, mode2policies.second))
        {
            return true;
        }
    }
    return false;
}

} // namespace maps::wiki
