#include "get_tasks.h"
#include "helpers.h"
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/exception.h>
#include <maps/wikimap/mapspro/services/editor/src/acl_utils.h>
#include <maps/wikimap/mapspro/services/editor/src/check_permissions.h>

#include <yandex/maps/wiki/configs/editor/category_groups.h>
#include <yandex/maps/wiki/social/gateway.h>

namespace maps {
namespace wiki {

namespace {

const std::string TASK_METHOD_NAME = "GetSocialModerationTasks";

} // namespace

GetSocialModerationTasks::Request::Request(
        social::TUid uid,
        Token token,
        social::TId before,
        social::TId after,
        size_t perPage,
        const std::optional<std::string>& categoryGroup,
        const std::optional<social::TUid>& resolvedBy,
        const std::optional<social::DateTimeCondition>& resolvedAt,
        const std::optional<social::ResolveResolution>& resolveResolution,
        const std::optional<social::TUid>& closedBy,
        const std::optional<social::DateTimeCondition>& closedAt,
        const std::optional<social::CloseResolution>& closeResolution)
    : uid(uid)
    , token(std::move(token))
    , before(before)
    , after(after)
    , perPage(perPage)
    , categoryGroup(categoryGroup)
    , resolvedBy(resolvedBy)
    , resolvedAt(resolvedAt)
    , resolveResolution(resolveResolution)
    , closedBy(closedBy)
    , closedAt(closedAt)
    , closeResolution(closeResolution)
{
    WIKI_REQUIRE(
            resolvedBy || closedBy,
            ERR_BAD_REQUEST,
            "Either 'resolved-by' or 'closed-by' should be specified");
    WIKI_REQUIRE(
            !(before && after),
            ERR_BAD_REQUEST,
            "using 'before' and 'after' simultaneously is unacceptable");
}

std::string
GetSocialModerationTasks::Request::dump() const
{
    std::stringstream ss;
    ss << " uid: " << uid
       << " token: " << token
       << " before: " << before
       << " after: " << after
       << " perPage: " << perPage;

    if (categoryGroup) {
       ss << " categoryGroup: " << *categoryGroup;
    }
    if (resolvedBy) {
        ss << " resolvedBy: " << *resolvedBy;
    }
    if (resolvedAt && resolvedAt->last()) {
        ss << " resolvedBefore: " << chrono::formatIsoDateTime(*resolvedAt->last());
    }
    if (resolvedAt && resolvedAt->first()) {
        ss << " resolvedAfter: " << chrono::formatIsoDateTime(*resolvedAt->first());
    }
    if (resolveResolution) {
        ss << " resolveResolution: " << toString(*resolveResolution);
    }
    if (closedBy) {
        ss << " closedBy: " << *closedBy;
    }
    if (closedAt && closedAt->last()) {
        ss << " closedBefore: " << chrono::formatIsoDateTime(*closedAt->last());
    }
    if (closedAt && closedAt->first()) {
        ss << " closedAfter: " << chrono::formatIsoDateTime(*closedAt->first());
    }
    if (closeResolution) {
        ss << " closeResolution: " << toString(*closeResolution);
    }

    return ss.str();
}


GetSocialModerationTasks::GetSocialModerationTasks(const Request& request)
    : controller::BaseController<GetSocialModerationTasks>(BOOST_CURRENT_FUNCTION)
    , request_(request)
{}

std::string
GetSocialModerationTasks::printRequest() const
{
    return request_.dump();
}

void
GetSocialModerationTasks::control()
{
    auto branchCtx = CheckedTrunkBranchContextFacade()
        .acquireReadCoreSocial(request_.token);
    auto permissionsChecker = CheckPermissions(
        request_.uid, branchCtx.txnCore());

    // check rights
    checkUserActiveStatus(branchCtx, request_.uid);
    if (request_.resolvedBy && *request_.resolvedBy != request_.uid ||
        request_.closedBy && *request_.closedBy != request_.uid)
    {
        permissionsChecker.checkPermissionForCompletedByOthersTaskFeed();
    }

    const auto orderBy = request_.closedBy
        ? social::TaskFeedParams::OrderBy::ClosedAt
        : social::TaskFeedParams::OrderBy::ResolvedAt;
    social::TaskFeedParams params(
        request_.before,
        request_.after,
        request_.perPage,
        orderBy);

    social::TaskFilter filter;
    if (request_.categoryGroup) {
        filter.setCategories(cfg()->editor()->categoryGroups().categoryIdsByGroup(*request_.categoryGroup));
    }
    if (request_.resolvedBy) {
        filter.setResolvedBy({*request_.resolvedBy});
    }
    filter.setResolvedAt(request_.resolvedAt);
    filter.setResolveResolution(request_.resolveResolution);
    if (request_.closedBy) {
        filter.setClosedBy({*request_.closedBy});
    }
    filter.setClosedAt(request_.closedAt);
    filter.setCloseResolution(request_.closeResolution);

    // get tasks
    social::Gateway socialGw(branchCtx.txnSocial());
    auto loadResult = socialGw.loadTasks(params, filter);
    result_->tasks = loadResult.tasks();
    result_->hasMore = loadResult.hasMore();

    // get alerts
    social::TIds taskIds;
    for (const auto& task : result_->tasks) {
        taskIds.insert(task.id());
    }
    for (auto& alert : socialGw.loadEventAlerts(taskIds)) {
        const auto categoryId = socialGw.loadEventsByIds({alert.eventId()})
            .front().getPrimaryObjectCategory();
        if (categoryId &&
            !permissionsChecker.isUserHasAccessToEventAlert(alert, *categoryId))
        {
            continue;
        }
        auto taskId = alert.eventId();
        result_->alertsByTaskId[taskId].push_back(std::move(alert));
    }

    // set user uid
    result_->uid = request_.uid;
}

const std::string&
GetSocialModerationTasks::taskName()
{
    return TASK_METHOD_NAME;
}

} // namespace wiki
} // namespace maps
