#include "helpers.h"

#include <maps/wikimap/mapspro/services/editor/src/acl_role_info.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/exception.h>
#include <maps/wikimap/mapspro/services/editor/src/moderation.h>
#include <maps/wikimap/mapspro/services/editor/src/social_utils.h>

#include <maps/wikimap/mapspro/libs/acl/include/check_context.h>
#include <maps/wikimap/mapspro/libs/acl/include/policy.h>
#include <maps/wikimap/mapspro/libs/acl/include/role.h>
#include <maps/wikimap/mapspro/libs/acl/include/subject_path.h>
#include <yandex/maps/wiki/common/robot.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/configs/editor/categories.h>

#include <boost/lexical_cast.hpp>

namespace maps::wiki::moderation {

namespace {

const std::string TASK_IDS = "taskIds";
const std::string REVERT_REASON = "revertReason";
const acl::SubjectPath TASKINFO_ACL_PATH("mpro/social/moderation-taskinfo");

bool
canUserCloseAllTasks(
    BranchContext& branchCtx,
    const social::Gateway& socialGw,
    const acl::User& user,
    const TCommitIds& commitIdsToClose,
    const TOIds& aoiIds,
    bool isUserCartographer)
{
    if (isUserCartographer) {
        return true;
    }

    // Only cartographers can close tasks without AOI.
    if (aoiIds.empty()) {
        return false;
    }

    const auto tasksToClose = socialGw.loadActiveEditTasksByCommitIds(commitIdsToClose);
    const auto autoApproveCommitIds = getAutoApproveCommitIds(
        branchCtx,
        socialGw,
        tasksToClose,
        user,
        aoiIds);

    return autoApproveCommitIds == commitIdsToClose;
}

} // namespace

void
checkJsonObject(const json::Value& jsonObject)
{
    WIKI_REQUIRE(jsonObject.isObject(), ERR_BAD_REQUEST, "json body is not object");
}

void
checkAclTaskInfo(pqxx::transaction_base& work, TUid uid)
{
    TASKINFO_ACL_PATH.check(
        acl::CheckContext(uid, {}, work, {acl::User::Status::Active}));
}

void
checkTasksType(const social::Tasks& tasks, social::EventType desiredType)
{
    for (const auto& task: tasks) {
        WIKI_REQUIRE(
            task.event().type() == desiredType,
            ERR_INVALID_OPERATION,
            "Wrong event type " << task.event().type() << " "
            "for task " << task.id() << ".");
    }
}

json::Value
requestJsonBody(const std::string& json, const std::string& taskName)
{
    try {
        return json::Value::fromString(json);
    } catch (const maps::Exception&) {
        THROW_WIKI_LOGIC_ERROR(
            ERR_BAD_REQUEST,
            "Invalid json format on parse request body in " << taskName);
    }
}

void
checkRequestTaskIds(const social::TaskIds& taskIds)
{
    CHECK_NON_EMPTY_REQUEST_PARAM(taskIds);

    for (auto taskId : taskIds) {
        WIKI_REQUIRE(taskId, ERR_BAD_REQUEST, "invalid task id");
    }
}

social::TaskIds
requestTaskIds(const json::Value& jsonObject)
{
    checkJsonObject(jsonObject);

    const auto& jsonIds = jsonObject[TASK_IDS];
    WIKI_REQUIRE(jsonIds.isArray(), ERR_BAD_REQUEST, TASK_IDS << " is not array");

    social::TaskIds taskIds;
    for (const auto& jsonValue : jsonIds) {
        auto id = boost::lexical_cast<TId>(jsonValue.toString());
        WIKI_REQUIRE(taskIds.insert(id).second, ERR_BAD_REQUEST,
            "duplicated task id: " << id);
    }
    checkRequestTaskIds(taskIds);
    return taskIds;
}

boost::optional<RevertReason>
requestRevertReason(const json::Value& jsonObject)
{
    if (!jsonObject.hasField(REVERT_REASON)) {
        return boost::none;
    }
    auto revertReasonStr = jsonObject[REVERT_REASON].toString();
    try {
        return boost::lexical_cast<RevertReason>(revertReasonStr);
    } catch(...) {
        THROW_WIKI_LOGIC_ERROR(
            ERR_BAD_REQUEST, "invalid revert reason: " << revertReasonStr);
    }
}

social::Tasks
loadTasks(
    const social::Gateway& socialGateway,
    const social::TaskIds& taskIds,
    TUid uid,
    ResolveLockedByOther resolveLockedByOther)
{
    auto tasks = socialGateway.loadActiveTasksByTaskIds(taskIds);
    WIKI_REQUIRE(
        tasks.size() == taskIds.size(), ERR_MODERATION_CONFLICT,
        "Some moderation tasks already closed : "
            << common::join(taskIds, ','));

    if (resolveLockedByOther == ResolveLockedByOther::Yes) {
        return tasks;
    }

    for (const auto& task : tasks) {
        WIKI_REQUIRE(
            task.isLocked(), ERR_MODERATION_CONFLICT,
            "Moderation task " << task.id() <<
            " not locked by current user " << uid);

        auto lockedUid = task.locked().uid();
        WIKI_REQUIRE(
            lockedUid == uid, ERR_MODERATION_CONFLICT,
            "Moderation task " << task.id() << " locked by another user"
            " (" << lockedUid << " != " << uid << ")");
    }

    return tasks;
}

void
checkTasks(const social::Tasks& tasks)
{
    size_t tasksWithComments{0};

    for (const auto& task: tasks) {
        if (task.event().comment()) {
            tasksWithComments++;
            WIKI_REQUIRE(
                task.event().type() != social::EventType::Edit,
                ERR_BAD_DATA,
                "Invalid event type '" << task.event().type() <<
                "' for comment task " << task.id() << ".");
        }
    }

    WIKI_REQUIRE(
        tasksWithComments == 0 || tasksWithComments == tasks.size(),
        ERR_INVALID_OPERATION,
        "Mixed tasks are forbidden.");
}

bool
hasComments(const social::Tasks& tasks)
{
    for (const auto& task: tasks) {
        if (task.event().comment()) {
            return true;
        }
    }
    return false;
}

bool
hasFeedbackTasks(const social::Tasks& tasks)
{
    return std::any_of(
        tasks.begin(), tasks.end(),
        [](const social::Task& task) {
            return task.event().type() == social::EventType::ClosedFeedback;
        }
    );
}

TCommitIds
commitIdsFromEditTasks(const social::Tasks& tasks)
{
    TCommitIds commitIds;
    for (const auto& task : tasks) {
        REQUIRE(task.event().type() == social::EventType::Edit,
                "unexpected non-edit task " << task.id() <<
                " type: " << task.event().type());
        auto commitId = task.commitId();
        if (commitId) {
            commitIds.insert(commitId);
        }
    }
    return commitIds;
}

TOIds
collectAoiIds(const RegionPolicies& regionPolicies)
{
    TOIds aoiIds;
    for (const auto& regionPolicy : regionPolicies) {
        aoiIds.insert(regionPolicy.aoiId);
    }
    return aoiIds;
}

RegionPolicies
createRegionPolicies(
    const acl::User& user,
    TOid aoiId,
    const boost::optional<social::ModerationMode>& mode)
{
    using AoiMode = std::pair<TOid, social::ModerationMode>;
    std::map<AoiMode, RegionPolicy> aoiMode2Policies;

    const auto& str2mode = getModerationModeResolver();

    // If there are several policies for the same AOI and mode, let's combine them.
    for (const auto& policy : user.allPolicies()) {
        auto policyAoiId = policy.aoiId();
        if (!policyAoiId || (aoiId != 0 && aoiId != policyAoiId)) {
            continue;
        }
        AclRoleInfo roleInfo(policy.role().name());
        if (roleInfo.trustLevel() == TrustLevel::NotTrusted) {
            continue;
        }
        auto modeIt = str2mode.find(roleInfo.roleName());
        if (modeIt == str2mode.end() || (mode && modeIt->second != *mode)) {
            continue;
        }

        auto& combinedPolicy = aoiMode2Policies[{policyAoiId, modeIt->second}];
        combinedPolicy.aoiId = policyAoiId;
        combinedPolicy.mode = modeIt->second;
        combinedPolicy.categoryGroupIds.insert(
            roleInfo.categoryGroupIds().begin(),
            roleInfo.categoryGroupIds().end());
        combinedPolicy.areCommonTasksPermitted |= roleInfo.hasAccessToCommonTasks();
    }

    RegionPolicies result;
    result.reserve(aoiMode2Policies.size());
    for (auto& aoiMode2Policy: aoiMode2Policies) {
        result.emplace_back(std::move(aoiMode2Policy.second));
    }
    return result;
}

bool areCommonTasksPermitted(const RegionPolicies& policies)
{
    for (const auto& policy: policies) {
        if (policy.areCommonTasksPermitted) return true;
    }
    return false;
}

TCommitIds
getAutoApproveCommitIds(
    const BranchContext& branchCtx,
    const social::Gateway& socialGateway,
    const social::Tasks& tasks,
    const acl::User& user,
    const TOIds& aoiIds)
{
    TCommitIds commitIds;
    const auto& categories = cfg()->editor()->categories();
    const auto& autoApproveCategoryIds =
        moderation::getAutoModeratedCategoryIds(user, aoiIds);

    for (const auto& task : tasks) {
        const auto& event = task.event();
        if (event.type() != social::EventType::Edit) {
            continue;
        }
        if (!event.commitData()) {
            continue;
        }
        const auto& primaryObjectData = event.primaryObjectData();
        if (!primaryObjectData) {
            continue;
        }
        const auto& bbox = event.commitData()->bbox();
        if (!bbox) {
            continue;
        }
        const auto& categoryId = primaryObjectData->categoryId();
        if (!categories.defined(categoryId)) {
            continue;
        }
        if (autoApproveCategoryIds.count(categoryId)
            && socialGateway.loadEventAlerts({task.id()}).empty())
        {
            commitIds.insert(event.commitData()->commitId());
            continue;
        }
        if (categories[categoryId].autoApprove()
            && !isCommitInImportantRegion(branchCtx, categoryId, *bbox)
            && !isCommitWithBigArea(categoryId, socialGateway, task.id()))
        {
            commitIds.insert(event.commitData()->commitId());
        }
    }
    return commitIds;
}

social::TaskIds
closeOrResolveRevertedTasks(
    BranchContext& branchCtx,
    const social::Gateway& socialGateway,
    const acl::User& user,
    const TCommitIds& revertedCommitIds,
    const TOIds& aoiIds,
    const TCommitIds& commitIds,
    bool isUserCartographer)
{
    using social::ResolveResolution;
    using social::CloseResolution;

    const auto indirectlyRevertedCommitIds = revertedCommitIds - commitIds;

    auto userConsole  = socialGateway.superModerationConsole(user.uid());
    auto robotConsole = socialGateway.superModerationConsole(common::ROBOT_UID);

    auto result =
        userConsole.resolveEditTasksByCommitIds(
            ResolveResolution::Revert, commitIds
        );
    result.merge(
        robotConsole.resolveEditTasksByCommitIds(
            ResolveResolution::Revert, indirectlyRevertedCommitIds
        )
    );

    const bool closeAll = canUserCloseAllTasks(
        branchCtx,
        socialGateway,
        user,
        revertedCommitIds,
        aoiIds,
        isUserCartographer);

    if (closeAll) {
        auto allCommitIds{commitIds};
        allCommitIds.insert(
            indirectlyRevertedCommitIds.cbegin(), indirectlyRevertedCommitIds.cend()
        );

        robotConsole.closeEditTasksByCommitIds(
            allCommitIds, ResolveResolution::Revert, CloseResolution::Revert
        );
    }

    return result;
}

} // namespace maps::wiki::moderation
