#include "regions.h"

#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/moderation.h>
#include <maps/wikimap/mapspro/services/editor/src/social_utils.h>
#include <maps/wikimap/mapspro/services/editor/src/utils.h>
#include "helpers.h"

#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <yandex/maps/wiki/configs/editor/category_groups.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/social/gateway.h>

namespace maps::wiki {

namespace {

const std::string STR_AOI = CATEGORY_AOI;
const std::string STR_AOI_NAME = "aoi:name";


social::TaskCounts
totalCounts(const TaskCountersByCategoryGroup& countersByCategoryGroupId)
{
    social::TaskCounts counts;
    for (const auto& [categoryGroupId, catCounters] : countersByCategoryGroupId) {
        counts += catCounters;
    }
    return counts;
}


// Returns tasks counters combined by category groups, counters for
// common moderation tasks are put into a fake group CATEGORY_GROUP_COMMON.
//
TaskCountersByCategoryGroup
countsByCategoryGroupId(
    const social::CountsByCategoryId& countsByCategoryId,
    const moderation::RegionPolicy& policy)
{
    TaskCountersByCategoryGroup result;
    const auto& categoryGroups = cfg()->editor()->categoryGroups();
    const auto& allowedCategoryIds = categoryGroups.categoryIdsByGroups(policy.categoryGroupIds);

    for (const auto& [categoryId, counts]: countsByCategoryId) {
        if (policy.areCommonTasksPermitted && categoryId.empty()) {
            result[CATEGORY_GROUP_COMMON] += counts;
            continue;
        }

        if (allowedCategoryIds.count(categoryId)) {
            const auto& groupId = categoryGroups.findGroupByCategoryId(categoryId)->id();
            result[groupId] += counts;
        }
    }

    return result;
}

void
fillCounts(
    ModerationRegion& region,
    const moderation::RegionPolicy& regionPolicy,
    const social::CountsByCategoryId& countsByCategoryId,
    size_t oldTaskAgeInHours,
    bool countByCategoryGroup)
{
    auto countsByGroup = countsByCategoryGroupId(countsByCategoryId, regionPolicy);

    region.taskCounters = totalCounts(countsByGroup);
    if (countByCategoryGroup) {
        region.taskCountersByCategoryGroup = countsByGroup;
    }
    region.oldTaskAgeInHours = oldTaskAgeInHours;
}

struct AoiData
{
    std::string name;
    Geom geom;
};

std::map<TOid, AoiData>
loadFilteredAoi(const BranchContext& branchCtx, const TOIds& aoiIds)
{
    std::map<TOid, AoiData> filteredAoi;
    auto filter = existingAoiFilter(aoiIds);

    revision::RevisionsGateway revGateway(branchCtx.txnCore()); // trunk
    auto snapshot = revGateway.snapshot(revGateway.headCommitId());

    for (const auto& rev : snapshot.objectRevisionsByFilter(filter)) {
        const auto& revData = rev.data();
        ASSERT(revData.geometry);
        ASSERT(revData.attributes);
        auto itName = revData.attributes->find(STR_AOI_NAME);
        REQUIRE(itName != revData.attributes->end(),
                "can not find " << STR_AOI_NAME <<
                " for revision " << rev.id());

        filteredAoi[rev.id().objectId()] =
            AoiData{itName->second, Geom(*revData.geometry)};
    }
    return filteredAoi;
}

} // namespace

std::string
SocialModerationRegions::Request::dump() const
{
    std::stringstream ss;
    ss << " uid: " << uid
       << " token: " << token
       << " aoi: " << aoiID;
    if (mode) {
        ss << " mode, acl role: " << moderation::toAclRoleName(*mode);
    }
    if (eventType) {
        ss << " event-type: " << *eventType;
    }
    if (createdBy) {
        ss << " created-by: " << *createdBy;
    }
    if (primaryObjectId) {
        ss << " primary-oid: " << *primaryObjectId;
    }
    if (aoiID) {
        ss << " countByCategoryGroup: " << countByCategoryGroup;
    }
    return ss.str();
}

SocialModerationRegions::SocialModerationRegions(const Request& request)
    : controller::BaseController<SocialModerationRegions>(BOOST_CURRENT_FUNCTION)
    , request_(request)
{}

std::string
SocialModerationRegions::printRequest() const
{
    return request_.dump();
}

void
SocialModerationRegions::control()
{
    loadRegions();

    if (request_.aoiID) {
        WIKI_REQUIRE(
            !result_->regions.empty(), ERR_FORBIDDEN,
            "Can not find allowed region " << request_.aoiID << " to moderate tasks");
    }
}

void
SocialModerationRegions::loadRegions()
{
    auto branchCtx = CheckedTrunkBranchContextFacade()
        .acquireReadCoreSocial(request_.token);

    auto user = acl::ACLGateway(branchCtx.txnCore()).user(request_.uid);
    user.checkActiveStatus();

    const auto regionPolicies =
        moderation::createRegionPolicies(user, request_.aoiID, request_.mode);
    if (regionPolicies.empty()) {
        return;
    }

    auto filteredAoi = loadFilteredAoi(branchCtx, moderation::collectAoiIds(regionPolicies));
    if (filteredAoi.empty()) {
        return;
    }

    std::map<social::ModerationMode, TOIds> mode2aoiIds;
    for (const auto& regionPolicy : regionPolicies) {
        if (filteredAoi.count(regionPolicy.aoiId)) {
            mode2aoiIds[regionPolicy.mode].insert(regionPolicy.aoiId);
        }
    }
    if (mode2aoiIds.empty()) {
        return;
    }

    social::Gateway gateway(branchCtx.txnSocial());
    auto modConsole = gateway.superModerationConsole(request_.uid);
    social::EventFilter eventFilter;
    eventFilter.deferred(social::Deferred::No);
    if (request_.eventType) {
        eventFilter.eventType(*request_.eventType);
    }
    if (request_.createdBy) {
        eventFilter.createdBy(*request_.createdBy);
    }
    if (request_.primaryObjectId) {
        eventFilter.objectId(*request_.primaryObjectId);
    }

    std::map<social::ModerationMode, social::CountsByAoiCategoryId> counts;
    for (const auto& [mode, aoiIds] : mode2aoiIds) {
        counts[mode] = modConsole.countsByAoiCategoryId(
            mode, eventFilter, aoiIds, cfg()->moderationTimeIntervals());
    }

    for (const auto& regionPolicy : regionPolicies) {
        if (!filteredAoi.count(regionPolicy.aoiId)) {
            continue;
        }
        const AoiData& aoiData = filteredAoi.at(regionPolicy.aoiId);

        ModerationRegion region {
            regionPolicy.aoiId, STR_AOI, aoiData.name,
            aoiData.geom, *aoiData.geom->getEnvelopeInternal(),
            regionPolicy.mode, social::TaskCounts(), {}, {}
        };

        fillCounts(
            region,
            regionPolicy,
            counts[region.mode].taskCounts[region.id],
            counts[region.mode].oldTaskAgeInHours,
            request_.countByCategoryGroup);

        result_->regions.push_back(std::move(region));
    }
}

} // namespace maps::wiki
