#include "tasks_defer.h"
#include "helpers.h"
#include <maps/wikimap/mapspro/services/editor/src/acl_utils.h>
#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/exception.h>
#include <yandex/maps/wiki/social/gateway.h>
#include <yandex/maps/wiki/common/string_utils.h>

namespace maps {
namespace wiki {

namespace {

const std::string TASK_METHOD_NAME = "SocialModerationTasksDefer";
const std::string EXPIRES_AT = "expiresAt";

} // namespace

SocialModerationTasksDefer::Request::Request(
        TUid uid,
        const social::TaskIds& taskIds,
        const std::string& expiresAt)
    : user(uid)
    , taskIds(taskIds)
    , expiresAt(expiresAt)
{
    CHECK_REQUEST_PARAM(uid);
    CHECK_NON_EMPTY_REQUEST_PARAM(expiresAt);
    moderation::checkRequestTaskIds(taskIds);
}

SocialModerationTasksDefer::Request::Request(
        TUid uid,
        const std::string& json)
    : user(uid)
{
    CHECK_REQUEST_PARAM(uid);

    auto jsonObject = moderation::requestJsonBody(json, TASK_METHOD_NAME);
    taskIds = moderation::requestTaskIds(jsonObject);

    expiresAt = jsonObject[EXPIRES_AT].toString();
    CHECK_NON_EMPTY_REQUEST_PARAM(expiresAt);
}

std::string
SocialModerationTasksDefer::Request::dump() const
{
    std::ostringstream os;
    os << " uid: " << user
       << " task ids: " << common::join(taskIds, ',')
       << " expires: " << expiresAt;
    return os.str();
}

SocialModerationTasksDefer::SocialModerationTasksDefer(
        const ObserverCollection&,
        const Request& request,
        taskutils::TaskID asyncTaskID)
    : controller::BaseController<SocialModerationTasksDefer>(BOOST_CURRENT_FUNCTION, asyncTaskID)
    , request_(request)
{}

std::string
SocialModerationTasksDefer::printRequest() const
{
    return request_.dump();
}

void
SocialModerationTasksDefer::control()
{
    auto branchCtx = CheckedTrunkBranchContextFacade().acquireWrite();

    checkUserActiveStatus(branchCtx, request_.user);

    auto modConsole = social::Gateway(branchCtx.txnSocial())
        .superModerationConsole(request_.user);
    result_->taskIds = modConsole.deferTasksByTaskIds(request_.taskIds, request_.expiresAt);

    for (auto taskId : request_.taskIds) {
        WIKI_REQUIRE(
            result_->taskIds.count(taskId), ERR_MODERATION_CONFLICT,
            "Can not defer moderation task:" << taskId);
    }

    result_->token = branchCtx.commit();
}

const std::string&
SocialModerationTasksDefer::taskName()
{
    return TASK_METHOD_NAME;
}

} // namespace wiki
} // namespace maps
