#include "user_acl_data.h"

#include "maps/wikimap/mapspro/services/editor/src/configs/config.h"
#include "maps/wikimap/mapspro/services/editor/src/check_permissions.h"
#include "maps/wikimap/mapspro/services/editor/src/objects/category_traits.h"

#include <yandex/maps/wiki/configs/editor/categories.h>

#include <maps/libs/common/include/profiletimer.h>

namespace maps::wiki {

namespace {

const std::chrono::milliseconds EXPIRATION_PERIOD = 600s;
const size_t USER_ACL_DATA_CACHE_SIZE = 10000;

} // namespace

StringSet
UserAclData::filterRequestCategories(const StringSet& requestCategories) const
{
    StringSet filtered;
    for (const auto& categoryId : requestCategories) {
        if (allowedCategories_.contains(categoryId) ||
            categoriesToAllowedAttrIdValues_.contains(categoryId))
        {
            filtered.insert(categoryId);
        }
    }
    return filtered;
}

UserAclData
UserAclData::create(TUid uid, Transaction& workCore)
{
    ProfileTimer timer;
    UserAclData data;
    const auto editorCfg = cfg()->editor();
    CheckPermissions permissionsChecker(uid, workCore, CheckPermissions::BannedPolicy::Allow);
    for (const auto& [id, category] : editorCfg->categories()) {
        if (!isSimpleGeomCategory(category)) {
            continue;
        }
        if (!permissionsChecker.isUserHasAccessToViewCategory(id)) {
            continue;
        }
        bool isControlledByAttr = false;
        for (const auto& attrDef : category.attrDefs()) {
            if (!attrDef->objectAccessControl()) {
                continue;
            }
            isControlledByAttr = true;
            for (const auto& valueDef : attrDef->values()) {
                if (permissionsChecker.isUserHasAccessToViewValue(
                    category.id(),
                    attrDef->id(),
                    valueDef.value))
                {
                    data.categoriesToAllowedAttrIdValues_[id].insert({attrDef->id(), valueDef.value});
                }
            }
        }
        if (!isControlledByAttr) {
            data.allowedCategories_.insert(id);
        }
    }
    INFO() << "UserAclData::create for uid: " << uid
        << " elapsed: " << timer.getElapsedTime()
        << " " << data.dump();

    return data;
}

bool
UserAclData::isAllowedObjectByAttributes(const StringMap& attributes) const
{
    const auto categoryId = categoryFromAttributes(attributes);
    if (allowedCategories_.contains(categoryId)) {
        return true;
    }
    const auto it = categoriesToAllowedAttrIdValues_.find(categoryId);
    if (it == categoriesToAllowedAttrIdValues_.end()) {
        return false;
    }
    const auto& allowingAttrsValues = it->second;
    for (const auto& attrValue : attributes) {
        if (allowingAttrsValues.contains(attrValue)) {
            return true;
        }
    }
    for (const auto& [id, value] : allowingAttrsValues) {
        if (value.empty() && !attributes.contains(id)) {
            return true;
        }
    }
    return false;
}

std::string
UserAclData::dump() const
{
    std::stringstream os;
    os << "categories: " << common::join(allowedCategories_, ",");
    os << " by attr value: ";
    for (const auto& [cat, attrValuePairs] : categoriesToAllowedAttrIdValues_) {
        for (const auto& [id, value] : attrValuePairs) {
            os << cat << "->[" << id << "," << value << "] ";
        }
    }
    return os.str();
}

UserAclDataCache::UserAclDataCache()
    : cache_(USER_ACL_DATA_CACHE_SIZE, EXPIRATION_PERIOD)
{
}

UserAclData
UserAclDataCache::get(TUid uid, const Token& dbToken)
{
    auto value = cache_.get(uid);
    if (value) {
        return *value;
    }
    auto createData = [&] {
        auto workCore = cfg()->poolCore().slaveTransaction(dbToken);
        return UserAclData::create(uid, *workCore);
    };
    auto data = createData();
    cache_.put(uid, data);
    return data;
}

} // namespace maps::wiki
