#include "task_points.h"
#include "buildings_for_tasks.h"
#include "load_objects.h"
#include "point_for_building.h"
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/exception.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/point.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>

#include <future>

namespace maps::wiki {

using namespace pedestrian;

namespace {

const int TASK_POINTS_LIMIT = 1000;

template<typename BldLinkedObject>
std::string limitExceedMessage()
{
    if constexpr(std::is_same_v<BldLinkedObject, Entrance>) {
        return ERR_ENTRANCES_LIMIT_EXCEEDED;
    } else if constexpr(std::is_same_v<BldLinkedObject, Address>) {
        return ERR_ADDRESSES_LIMIT_EXCEEDED;
    }
}


template<typename BldLinkedObject>
std::vector<Building> calculateResultBuildings(
    const geolib3::Polygon2& polygonMerc,
    pgpool3::Pool& viewTrunkPool,
    const std::string& txnToken)
{
    auto buildingsFut = std::async(std::launch::async, [&]() {
        auto viewTxnHandle = viewTrunkPool.slaveTransaction(txnToken);
        try {
            return loadBuildings<BldLinkedObject>(polygonMerc, viewTxnHandle.get());
        } catch (ObjectsLimitException&) {
            throw LogicException(ERR_BUILDINGS_LIMIT_EXCEEDED);
        }
    });

    auto entrancesFut = std::async(std::launch::async, [&]() {
        auto viewTxnHandle = viewTrunkPool.slaveTransaction(txnToken);
        try {
            return loadBldLinkedObjects<BldLinkedObject>(
                polygonMerc,
                viewTxnHandle.get()
            );
        } catch (ObjectsLimitException&) {
            throw LogicException(limitExceedMessage<BldLinkedObject>());
        }
    });

    auto bldForTasks = buildingsForTasks(
        buildingsFut.get(), entrancesFut.get()
    );

    if (bldForTasks.size() > TASK_POINTS_LIMIT) {
        throw LogicException(ERR_TASK_POINTS_LIMIT_EXCEEDED);
    }

    return bldForTasks;
}

} // unnamed namespace

ToolsPedestrianPoints::ToolsPedestrianPoints(const Request& request)
    : controller::BaseController<ToolsPedestrianPoints>(BOOST_CURRENT_FUNCTION)
    , request_(request)
{
}

std::string
ToolsPedestrianPoints::printRequest() const
{
    std::stringstream ss;
    ss << " token: " << request_.token;
    ss << " oid: " << request_.oid;
    return ss.str();
}

void
ToolsPedestrianPoints::control()
{
    auto& corePool = cfg()->poolCore();
    auto& viewPool = cfg()->poolViewTrunk();

    // Get task type and polygon
    //
    auto coreTxnHandle = corePool.slaveTransaction(request_.token);
    auto revisionGateway = revision::RevisionsGateway(coreTxnHandle.get());
    auto revisionSnapshot = revisionGateway.snapshot(revisionGateway.maxSnapshotId());

    auto layer = revisionSnapshot.objectRevision(request_.oid);
    ASSERT(layer);
    ASSERT(layer->data().attributes);
    ASSERT(layer->data().geometry);

    auto taskType = layer->data().attributes->at(ATTR_PEDESTRIAN_REGION_TASK_TYPE);
    auto polygonMerc = geolib3::WKB::read<geolib3::Polygon2>(*layer->data().geometry);

    // Calculate task points
    //
    const std::vector<Building> bldForTasks = [&](){
        if (taskType == "addresses") {
            return calculateResultBuildings<Address>(polygonMerc, viewPool, request_.token);
        } else if (taskType == "entrances") {
            return calculateResultBuildings<Entrance>(polygonMerc, viewPool, request_.token);
        } else {
            THROW_WIKI_INTERNAL_ERROR("unknown task type: " + taskType);
        }
    }();

    for (const auto& bld : bldForTasks) {
        result_->points.emplace_back(
            geolib3::convertMercatorToGeodetic(pointForBuilding(bld.polygonMerc)));
    }
}

} // namespace maps::wiki
