#include "stick_polygons.h"

#include "maps/wikimap/mapspro/services/editor/src/check_permissions.h"
#include "maps/wikimap/mapspro/services/editor/src/configs/config.h"
#include "maps/wikimap/mapspro/services/editor/src/collection.h"
#include "maps/wikimap/mapspro/services/editor/src/geom.h"
#include "maps/wikimap/mapspro/services/editor/src/objects_cache.h"
#include "maps/wikimap/mapspro/services/editor/src/objects/areal_object.h"
#include "maps/wikimap/mapspro/services/editor/src/objects/line_object.h"
#include "maps/wikimap/mapspro/services/editor/src/polyline_sticker/polyline_sticker.h"
#include "maps/wikimap/mapspro/services/editor/src/srv_attrs/registry.h"
#include "maps/wikimap/mapspro/services/editor/src/utils.h"
#include "maps/wikimap/mapspro/services/editor/src/views/objects_query.h"

#include <yandex/maps/wiki/configs/editor/categories.h>
#include <yandex/maps/wiki/configs/editor/category_groups.h>
#include <yandex/maps/wiki/configs/editor/category_template.h>
#include <yandex/maps/wiki/configs/editor/restrictions.h>

#include <maps/libs/tile/include/utils.h>
#include <geos/geom/Polygon.h>
#include <geos/geom/Coordinate.h>
#include <geos/geom/Envelope.h>

namespace maps::wiki {
namespace {
constexpr double TOLERANCE = 0.3;
const double VERTEX_TO_EDGE_ACCURACY_PIXELS = 2;
const double VERTEX_TO_VERTEX_ACCURACY_PIXELS = 2;
const double MAX_END_POINT_TO_ENDPOINT_METERS = 10;
const double ENDPOINT_TO_ENDPOINT_PIXELS = 12;

class GeomIPolyline final : public polyline_sticker::IPolyline
{
public:
    GeomIPolyline(const GeoObject* obj)
        : id_(obj->id())
        , geom_(obj->geom())
        , poly_(dynamic_cast<const geos::geom::Polygon*>(geom_.geosGeometryPtr()))
    {
        ASSERT(poly_);
    }

    polyline_sticker::ID id() const override { return id_; };

    size_t numLineStrings() const override { return poly_->getNumInteriorRing() + 1; }

    size_t numLineStringVertexes(size_t index) const override
    {
        const auto* lineString =
            index == 0
            ? poly_->getExteriorRing()
            : poly_->getInteriorRingN(index - 1);
        return lineString->getNumPoints();
    }

    geolib3::Point2 lineStringVertex(size_t index, size_t v) const override
    {
        const auto* lineString =
            index == 0
            ? poly_->getExteriorRing()
            : poly_->getInteriorRingN(index - 1);
        auto coord = lineString->getCoordinateN(v);
        return {coord.x, coord.y};
    }
    bool isClosed() const override { return true; }

private:

    polyline_sticker::ID id_;
    const Geom geom_;
    const geos::geom::Polygon* poly_;
};

class GeomILineString final : public polyline_sticker::IPolyline
{
public:
    GeomILineString(const GeoObject* obj)
        : id_(obj->id())
        , geom_(obj->geom())
        , line_(dynamic_cast<const geos::geom::LineString*>(geom_.geosGeometryPtr()))
    {
        ASSERT(line_);
    }

    polyline_sticker::ID id() const override { return id_; };

    size_t numLineStrings() const override { return 1; }

    size_t numLineStringVertexes(size_t index) const override
    {
        ASSERT(index == 0);

        return line_->getNumPoints();
    }

    geolib3::Point2 lineStringVertex(size_t index, size_t v) const override
    {
        ASSERT(index == 0);
        auto coord = line_->getCoordinateN(v);
        return {coord.x, coord.y};
    }

    bool isClosed() const override { return false; }

private:
    polyline_sticker::ID id_;
    const Geom geom_;
    const geos::geom::LineString* line_;
};

Geom polygonFromStickResult(polyline_sticker::ResultPolyline result)
{
    ASSERT(result.isClosed());
    std::unique_ptr<std::vector<geos::geom::Coordinate>>
        points(new std::vector<geos::geom::Coordinate>());
    const auto numPoints = result.numLineStringVertexes(0);
    points->reserve(result.numLineStringVertexes(0));
    for (size_t v = 0; v < numPoints; ++v) {
        const auto coord = result.lineStringVertex(0, v);
        points->emplace_back(geos::geom::Coordinate(coord.x(), coord.y()));
    }
    struct Deleter
    {
        void operator()(std::vector<GeosGeometryPtr>* geoms) const
        {
            if (geoms) {
                for (auto* geom : *geoms) {
                    delete geom;
                }
            }
            delete geoms;
        }
    };
    std::unique_ptr<std::vector<GeosGeometryPtr>, Deleter> holes;
    if (result.numLineStrings() > 1) {
        holes.reset(new std::vector<GeosGeometryPtr>());
        for (size_t iRing = 1; iRing < result.numLineStrings(); ++iRing) {
            std::unique_ptr<std::vector<geos::geom::Coordinate>>
                holeCoords(new std::vector<geos::geom::Coordinate>());;
            for (size_t v = 0; v < result.numLineStringVertexes(iRing); ++v) {
                const auto coord = result.lineStringVertex(iRing, v);
                holeCoords->emplace_back(geos::geom::Coordinate(coord.x(), coord.y()));
            }
            std::unique_ptr<geos::geom::Geometry> hole(createPolygon(
                holeCoords.release(), nullptr,
                SpatialRefSystem::Mercator,
                AntiMeridianAdjustPolicy::Ignore));
            holes->emplace_back(dynamic_cast<geos::geom::Polygon*>(
                hole.get())->getExteriorRing()->clone().release());
        }
    }
    return Geom(
        createPolygon(
            points.release(),
            holes.release(),
            SpatialRefSystem::Mercator,
            AntiMeridianAdjustPolicy::Ignore));
}

Geom lineFromStickResult(polyline_sticker::ResultPolyline result)
{
    ASSERT(!result.isClosed());
    std::unique_ptr<std::vector<geos::geom::Coordinate>>
        points(new std::vector<geos::geom::Coordinate>());
    const auto numPoints = result.numLineStringVertexes(0);
    points->reserve(result.numLineStringVertexes(0));
    for (size_t v = 0; v < numPoints; ++v) {
        const auto coord = result.lineStringVertex(0, v);
        points->emplace_back(geos::geom::Coordinate(coord.x(), coord.y()));
    }
    struct Deleter
    {
        void operator()(std::vector<GeosGeometryPtr>* geoms) const
        {
            if (geoms) {
                for (auto* geom : *geoms) {
                    delete geom;
                }
            }
            delete geoms;
        }
    };
    std::unique_ptr<std::vector<GeosGeometryPtr>, Deleter> holes;
    ASSERT(result.numLineStrings() == 1);
    return Geom(
        createPolyline(
            points.release(),
            SpatialRefSystem::Mercator,
            AntiMeridianAdjustPolicy::Ignore));
}

std::vector<const geos::geom::LineString*>
allPolylines(const Geom& geom)
{
    const auto* geosGeom = geom.geosGeometryPtr();
    std::vector<const geos::geom::LineString*> rings;
    const auto* poly = dynamic_cast<const geos::geom::Polygon*>(geosGeom);
    if (poly) {
        rings.reserve(1 + poly->getNumInteriorRing());
        rings.push_back(poly->getExteriorRing());
        for (size_t i = 0; i < poly->getNumInteriorRing(); ++i) {
            rings.push_back(poly->getInteriorRingN(i));
        }
    } else {
        const auto* line = dynamic_cast<const geos::geom::LineString*>(geosGeom);
        ASSERT(line);
        rings.push_back(line);
    }
    return rings;
}

double
minEdgeDistance(const Geom& geom1, const Geom& geom2)
{
    ASSERT(!geom1.isNull() && !geom2.isNull());
    double minDistance = geom1.distance(geom2);
    const auto polylines1 = allPolylines(geom1);
    const auto polylines2 = allPolylines(geom2);
    for (const auto* ring1 : polylines1) {
        for (const auto* ring2 : polylines2) {
            minDistance = std::min(minDistance, ring1->distance(ring2));
        }
    }
    return minDistance;
}

class StickGroup
{
public:
    explicit StickGroup(const std::string& categoryId)
    {
        const auto& categoryGroups = cfg()->editor()->categoryGroups();
        const auto& groupId = categoryGroups.findGroupByCategoryId(categoryId)->id();
        if (CATEGORY_GROUP_HD_MAP == groupId) {
            kind_ = Kind::HDMap;
        } else if (groupId == CATEGORY_GROUP_INDOOR) {
            kind_ = Kind::Indoor;
        } else {
            kind_ = Kind::Common;
            categoryId_ = categoryId;
        }
    }

    bool operator ==(const StickGroup& other) const {
        return std::tie(kind_, categoryId_) ==
            std::tie(other.kind_, other.categoryId_);
    }

    StringSet categories() const {
        if (kind_ == Kind::Common) {
            return {categoryId_};
        }
        StringSet categoryIds;
        if (kind_ == Kind::HDMap) {
            categoryIds =
                cfg()->editor()->categoryGroups().categoryIdsByGroup(CATEGORY_GROUP_HD_MAP);
        } else if (kind_ == Kind::Indoor) {
            categoryIds =
                cfg()->editor()->categoryGroups().categoryIdsByGroup(CATEGORY_GROUP_INDOOR);
        }
        StringSet result;
        const auto& editorCategories = cfg()->editor()->categories();
        for (const auto& categoryId : categoryIds) {
            const auto& category = editorCategories[categoryId];
            const auto& geomType = category.categoryTemplate().geometryType();
            if (!geomType.empty() && geomType != Geom::geomTypeNamePoint) {
                result.insert(categoryId);
            }
        }
        return result;
    }

private:
    enum class Kind {
        HDMap,
        Indoor,
        Common
    };
    Kind kind_;
    std::string categoryId_;
};

std::optional<TOid>
objectIndoorLevelId(const GeoObject* object)
{
    if (object->categoryId() == CATEGORY_INDOOR_LEVEL) {
        return object->id();
    }
    auto masters = object->masterRelations().range();
    for (const auto& master : masters) {
        if (master.categoryId() == CATEGORY_INDOOR_LEVEL) {
            return master.id();
        }
    }
    return {};
}

double
vertexGravity(TZoom zoom)
{
    const auto radius = VERTEX_TO_VERTEX_ACCURACY_PIXELS * tile::zoomToResolution(zoom);
    return std::min(radius, TOLERANCE);
}

double
vertexToEdgeGravity(TZoom zoom)
{
    const auto radius = VERTEX_TO_EDGE_ACCURACY_PIXELS * tile::zoomToResolution(zoom);
    return std::min(radius, TOLERANCE);
}

double
endpointGravity(TZoom zoom)
{
    return std::min(
        MAX_END_POINT_TO_ENDPOINT_METERS,
        ENDPOINT_TO_ENDPOINT_PIXELS * tile::zoomToResolution(zoom));
}

} // namespace

void
stickGeometries(
    GeoObjectCollection& moveableObjects,
    std::optional<GeoObjectCollection> fixedObjects,
    TUid userId,
    Transaction& workCore,
    TZoom workZooom)
{
    if (moveableObjects.empty() && (!fixedObjects || fixedObjects->empty())) {
        return;
    }
    polyline_sticker::PolylineSticker sticker(
        vertexGravity(workZooom),
        vertexToEdgeGravity(workZooom),
        endpointGravity(workZooom));
    auto addColleciton = [&](const GeoObjectCollection& collection, bool fixed) {
        for (const auto& object : collection) {
            if (is<ArealObject>(object)) {
                const GeomIPolyline polygon(object.get());
                sticker.addPolyline(&polygon, fixed);
            } else {
                const GeomILineString lineString(object.get());
                sticker.addPolyline(&lineString, fixed);
            }
            DEBUG() << "sticker: added " << object->id();
        }
    };
    addColleciton(moveableObjects, false);
    if (fixedObjects) {
        addColleciton(*fixedObjects, true);
    }

    if (!sticker.processMesh()) {
        DEBUG() << "sticker: No valid stick performed";
        return;
    }
    std::map<TOid, Geom> newGeoms;
    auto collectAndCheck = [&](TOid id, size_t maxVertexes) {
        auto resultGeom = sticker.resultPolyline(id);
        if (!resultGeom) {
            WARN() << "sticker: degenerated geom in result for: " << id;
            return false;
        }
        try {
            auto newGeom = resultGeom->isClosed()
                ? polygonFromStickResult(*resultGeom)
                : lineFromStickResult(*resultGeom);
            if (newGeom->isValid()) {
                newGeoms.emplace(id, newGeom);
            } else {
                WARN() << "sticker: invalid geom in result for: " << id;
                return false;
            }
            if (newGeom->getNumPoints() > maxVertexes) {
                WARN() << "stiker: " << newGeom->getNumPoints()
                    << " is too many vertexes in result for: " << id;
                return false;
            }
        } catch (const std::exception& ex) {
            std::ostringstream os;
            for (size_t i = 0; i < resultGeom->numLineStrings(); ++i) {
                os << " linestring " << i
                << "(" << resultGeom->numLineStringVertexes(i) << ")";
            }
            WARN() << "sticker: Exception for: " << id
                << " closed=" << resultGeom->isClosed()
                << os.str() << " " << ex.what();
            return false;
        }
        return true;
    };

    CheckPermissions checkPermissions(userId, workCore);
    auto updateGeom = [&](const ObjectPtr& obj) {
        auto it = newGeoms.find(obj->id());
        ASSERT(it != newGeoms.end());
        if (checkPermissions.hasPermissionsToEditObjectGeometry(obj.get())) {
            DEBUG() << "sticker: updaing " << obj->id();
            obj->setGeometry(it->second);
        }
    };

    for (auto& object : moveableObjects) {
        if (!collectAndCheck(
                object->id(),
                object->restrictions().maxVertexes()))
        {
            return;
        }
    }
    if (fixedObjects) {
        for (auto& object : *fixedObjects) {
            if (!collectAndCheck(
                    object->id(),
                    object->restrictions().maxVertexes()))
            {
                return;
            }
        }
    }

    for (auto& object : moveableObjects) {
        updateGeom(object);
    }
    if (fixedObjects) {
        for (auto& fixedObjects : *fixedObjects) {
            updateGeom(fixedObjects);
        }
    }
}

void stickGeometries(ObjectsCache& cache, TUid userId, TZoom workZooom)
{
    TOIds modifiedObjectIds;
    auto modifiedObjects = cache.find([&](const GeoObject* obj) {
        if (obj->isModifiedGeom() &&
                (is<ArealObject>(obj) || is<LineObject>(obj)))
        {
            modifiedObjectIds.insert(obj->id());
            return true;
        }
        return false;
    });
    if (modifiedObjectIds.empty()) {
        return;
    }
    std::optional<TOid> commonIndoorLevelId;
    bool hasObjectsWithoutIndoorLevel = false;
    std::optional<StickGroup> curStickGroup;
    geos::geom::Envelope envelope = *(*modifiedObjects.begin())->geom()->getEnvelopeInternal();
    for (const auto& modifiedObject : modifiedObjects) {
        StickGroup newStickGroup(modifiedObject->categoryId());
        if (!curStickGroup) {
            curStickGroup = newStickGroup;
        } else if (curStickGroup != newStickGroup) {
            return;
        }
        envelope.expandToInclude(modifiedObject->geom()->getEnvelopeInternal());
        std::optional<TOid> indoorLevelId = objectIndoorLevelId(modifiedObject.get());
        if (!indoorLevelId) {
            hasObjectsWithoutIndoorLevel = true;
        } else {
            if (commonIndoorLevelId && *commonIndoorLevelId != *indoorLevelId) {
                return;
            }
            commonIndoorLevelId = *indoorLevelId;
        }
    }
    if (hasObjectsWithoutIndoorLevel && commonIndoorLevelId) {
        return;
    }
    auto& workView = cache.workView();
    views::ObjectsQuery objectsQuery;
    objectsQuery.addCondition(views::CategoriesCondition(workView, curStickGroup->categories()));
    if (commonIndoorLevelId) {
        objectsQuery.addCondition(views::AllOfServiceAttributesCondition(
            workView,
            {{srv_attrs::SRV_INDOOR_LEVEL_ID, std::to_string(*commonIndoorLevelId)}}));
    } else {
        objectsQuery.addCondition(views::NoneOfServiceAttributesCondition(
            workView, {srv_attrs::SRV_INDOOR_LEVEL_ID}));
    }
    const auto searchRadius = endpointGravity(workZooom);
    envelope.expandBy(searchRadius);
    objectsQuery.addCondition(views::EnvelopeGeometryCondition(envelope));
    const auto neighbours = objectsQuery.exec(workView, cache.branchContext().branch.id());
    TOIds fixedObjectsIds;
    for (const auto& viewObject : neighbours) {
        if (modifiedObjectIds.contains(viewObject.id())) {
            continue;
        }
        for (const auto& modifiedObject : modifiedObjects) {
            if (minEdgeDistance(modifiedObject->geom(), viewObject.geom()) < searchRadius) {
                fixedObjectsIds.insert(viewObject.id());
                break;
            }
        }
    }
    if (fixedObjectsIds.empty() && modifiedObjectIds.size() == 1) {
        return;
    }
    GeoObjectCollection fixedObjects;
    if (!fixedObjectsIds.empty()) {
        fixedObjects = cache.get(fixedObjectsIds);
    }
    stickGeometries(modifiedObjects, fixedObjects, userId, cache.workCore(), workZooom);
}
} // namespace maps::wiki
