#include "handles.h"

#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>

#include <maps/wikimap/mapspro/services/editor/src/actions/get_rich_content.h>

#include <maps/wikimap/mapspro/services/editor/src/serialize/save_object_parser.h>
#include <maps/wikimap/mapspro/services/editor/src/serialize/create_intersections_parser.h>
#include <maps/wikimap/mapspro/services/editor/src/serialize/objects_query_route_diff_parser.h>
#include <maps/wikimap/mapspro/services/editor/src/serialize/objects_query_path_parser.h>
#include <maps/wikimap/mapspro/services/editor/src/serialize/get_suggest_parser.h>

#include <maps/wikimap/mapspro/services/editor/src/object_update_data.h>

#include <maps/wikimap/mapspro/libs/acl/include/deleted_users_cache.h>
#include <yandex/maps/wiki/common/robot.h>

#include <maps/infra/yacare/include/tvm.h>

namespace common = maps::wiki::common;
namespace chrono = maps::chrono;
namespace editor = maps::wiki;

namespace {

yacare::ThreadPool editorThreadPool("editor", 10, 100);

} // namespace

YCR_USE(editorThreadPool) {

YCR_RESPOND_TO("GET /objects/$", uid, branch = TRUNK_BRANCH_ID, token = "")
{
    editor::cfg()->deletedUsersCache().checkUser(uid);

    editor::GetObject::Request controllerRequest {
            getParam<editor::TOid>(argv, 0),
            uid,
            token,
            branch,
            getParam<bool>(request, "allow-substitution", false)
                ? editor::GetObject::SubstitutionPolicy::Allow
                : editor::GetObject::SubstitutionPolicy::Deny,
            getParam<bool>(request, "not-load-parts-of-complex-contour-objects", false)
                ? editor::GetObject::PartsOfComplexContourObjectsLoadPolicy::Skip
                : editor::GetObject::PartsOfComplexContourObjectsLoadPolicy::Load
            };
    handleController<editor::GetObject>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /objects/$/history", uid = NO_UID, branch = TRUNK_BRANCH_ID, token = "")
{
    editor::GetHistory::Request controllerRequest {
            getParam<editor::TOid>(argv, 0),
            uid,
            token,
            branch,
            getParam<size_t>(request, "page", 1),
            getParam<size_t>(request, "per-page", 10),
            getParam<bool>(request, "with-relations", true)
                ? editor::GetHistory::RelationsChangePolicy::Show
                : editor::GetHistory::RelationsChangePolicy::Hide,
            getParam<bool>(request, "primary-only", false)
                ? editor::GetHistory::IndirectChangePolicy::Hide
                : editor::GetHistory::IndirectChangePolicy::Show
        };
    handleController<editor::GetHistory>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /objects/query/lasso",
    branch = TRUNK_BRANCH_ID,
    token = "")
{
    editor::ObjectsQueryLasso::Request controllerRequest {
        getParam<std::string>(request, "categories"),
        getParam<std::string>(request, "geometry"),
        getStdOptionalParam<double>(request, "threshold"),
        token,
        branch,
        getParam<size_t>(request, "limit"),
        getParam<bool>(request, "intersects", false)
            ? editor::ObjectsQueryLasso::GeomPredicate::Intersects
            : editor::ObjectsQueryLasso::GeomPredicate::CoveredBy,
        getStdOptionalParam<editor::TOid>(request, "indoor-level-id"),
        getStdOptionalParam<maps::wiki::filters::TExpressionId>(request, "expression-id"),
        getParam<bool>(request, "allow-substitution", false)
                ? editor::ObjectsQueryLasso::SubstitutionPolicy::Allow
                : editor::ObjectsQueryLasso::SubstitutionPolicy::Deny,
    };
    handleController<editor::ObjectsQueryLasso>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /objects/query/lasso/meta",
    uid)
{
    editor::ObjectsQueryLassoMeta::Request controllerRequest {
        uid
    };
    handleController<editor::ObjectsQueryLassoMeta>(response, controllerRequest, requestedFormat(request));
}


YCR_RESPOND_TO("POST /objects/query/title",
    branch = TRUNK_BRANCH_ID,
    token = "")
{
    editor::GetSuggestParser parser;
    parser.parse(requestedFormat(request), request.body());
    editor::ObjectsQueryTitle::Request controllerRequest {
        getParam<std::string>(request, "categories"),
        getParam<std::string>(request, "text", ""),
        parser.geometry(),
        getParam<std::string>(request, "ll", ""),
        std::min(getParam<double>(request, "d", 2000.0), 2000.0),
        token,
        branch,
        std::min(getParam<size_t>(request, "limit", 10), 100ul),
        getStdOptionalParam<editor::TOid>(request, "indoor-level-id")
    };
    handleController<editor::ObjectsQueryTitle>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /objects/query/poi/conflicts",
    branch = TRUNK_BRANCH_ID,
    token = "",
    uid = NO_UID)
{
    editor::ObjectsQueryPoiConflicts::Request controllerRequest {
        getParam<std::string>(request, "geometry"),
        token,
        branch,
        getStdOptionalParam<editor::TOid>(request, "indoor-level-id"),
        getStdOptionalParam<editor::TOid>(request, "object-id"),
        getParam<bool>(request, "is-geoproduct")
            ? editor::ObjectsQueryPoiConflicts::Request::IsGeoproductParam::True
            : editor::ObjectsQueryPoiConflicts::Request::IsGeoproductParam::False,
        uid
    };
    handleController<editor::ObjectsQueryPoiConflicts>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /objects/query/pois/conflicts",
    branch = TRUNK_BRANCH_ID,
    token = "",
    uid = NO_UID)
{
    editor::ObjectsQueryPoisConflicts::Request controllerRequest {
        token,
        branch,
        getStdOptionalParam<editor::TOid>(request, "indoor-level-id"),
        request.body(),
        uid
    };
    handleController<editor::ObjectsQueryPoisConflicts>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /objects/query/pois/conflicts/meta",
    uid = NO_UID)
{
    editor::ObjectsQueryPoisConflictsMeta::Request controllerRequest {
        uid
    };
    handleController<editor::ObjectsQueryPoisConflictsMeta>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /objects/query/lasso",
    branch = TRUNK_BRANCH_ID,
    token = "")
{
    editor::ObjectsQueryLasso::Request controllerRequest {
        getParam<std::string>(request, "categories"),
        request.body(),
        getStdOptionalParam<double>(request, "threshold"),
        token,
        branch,
        getParam<size_t>(request, "limit"),
        getParam<bool>(request, "intersects", false)
            ? editor::ObjectsQueryLasso::GeomPredicate::Intersects
            : editor::ObjectsQueryLasso::GeomPredicate::CoveredBy,
        getStdOptionalParam<editor::TOid>(request, "indoor-level-id"),
        getStdOptionalParam<maps::wiki::filters::TExpressionId>(request, "expression-id"),
        getParam<bool>(request, "allow-substitution", false)
                ? editor::ObjectsQueryLasso::SubstitutionPolicy::Allow
                : editor::ObjectsQueryLasso::SubstitutionPolicy::Deny,
    };
    handleController<editor::ObjectsQueryLasso>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /objects/query/poi/business-id/$",
    branch = TRUNK_BRANCH_ID,
    token = "")
{
    editor::ObjectsQueryPoiBusinessId::Request controllerRequest {
        getParam<std::string>(argv, 0),
        getParam<std::string>(request, "geometry", ""),
        getStdOptionalParam<double>(request, "d"),
        token,
        branch
        };
    handleController<editor::ObjectsQueryPoiBusinessId>(response, controllerRequest, requestedFormat(request));
}

namespace {

editor::ObjectsQueryIds::SerializeDetailsFlags
queryIdsSerializeDetailsFlags(const yacare::Request& request)
{
    editor::ObjectsQueryIds::SerializeDetailsFlags flags = {editor::ObjectsQueryIds::SerializeDetails::Brief};
    if (getParam<bool>(request, "with-attrs", false)) {
        flags.set(editor::ObjectsQueryIds::SerializeDetails::Attrs);
    }
    if (getParam<bool>(request, "with-permissions", false)) {
        flags.set(editor::ObjectsQueryIds::SerializeDetails::Permissions);
    }
    if (getParam<bool>(request, "with-masters", false)) {
        flags.set(editor::ObjectsQueryIds::SerializeDetails::Masters);
    }
    return flags;
}

} // namespace

YCR_RESPOND_TO("GET /objects/query/ids",
    ids,
    uid = NO_UID,
    branch = TRUNK_BRANCH_ID,
    token = "")
{
    editor::ObjectsQueryIds::Request controllerRequest(
        editor::ObjectsQueryIds::IdsFormat::CSV,
        ids,
        queryIdsSerializeDetailsFlags(request),
        uid,
        token,
        branch);
    handleController<editor::ObjectsQueryIds>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /objects/query/ids",
    uid = NO_UID,
    branch = TRUNK_BRANCH_ID,
    token = "")
{

    editor::ObjectsQueryIds::Request controllerRequest(
        editor::ObjectsQueryIds::IdsFormat::JsonArray,
        request.body(),
        queryIdsSerializeDetailsFlags(request),
        uid,
        token,
        branch);
    handleController<editor::ObjectsQueryIds>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /objects/query/category/transport_operator",
    uid = NO_UID,
    branch = TRUNK_BRANCH_ID,
    token = "")
{
    editor::ObjectsQueryCategory::Request controllerRequest {
        editor::CATEGORY_TRANSPORT_OPERATOR,
        uid,
        token,
        branch};
    handleController<editor::ObjectsQueryCategory>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /objects/query/category/region",
    uid = NO_UID,
    branch = TRUNK_BRANCH_ID,
    token = "")
{
    editor::ObjectsQueryCategory::Request controllerRequest {
        editor::CATEGORY_REGION,
        uid,
        token,
        branch};
    handleController<editor::ObjectsQueryCategory>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /objects/query/category/vehicle_restriction",
    uid = NO_UID,
    branch = TRUNK_BRANCH_ID,
    token = "")
{
    editor::ObjectsQueryCategory::Request controllerRequest {
        editor::CATEGORY_VEHICLE_RESTRICTION,
        uid,
        token,
        branch};
    handleController<editor::ObjectsQueryCategory>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /objects/query/category/cond_annotation_phrase",
    uid = NO_UID,
    branch = TRUNK_BRANCH_ID,
    token = "")
{
    editor::ObjectsQueryCategory::Request controllerRequest {
        editor::CATEGORY_COND_ANNOTATION_PHRASE,
        uid,
        token,
        branch};
    handleController<editor::ObjectsQueryCategory>(response, controllerRequest, requestedFormat(request));
}


YCR_RESPOND_TO("GET /objects/query/filter",
    branch = TRUNK_BRANCH_ID,
    token = "")
{
    editor::ObjectsQueryFilter::Request controllerRequest {
        getParam<maps::wiki::filters::TExpressionId>(request, "expression-id"),
        getParam<std::string>(request, "geometry"),
        branch,
        token,
        getParam<size_t>(request, "limit", 100),
    };
    handleController<editor::ObjectsQueryFilter>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /objects/query/filter",
    branch = TRUNK_BRANCH_ID,
    token = "")
{
    editor::ObjectsQueryFilter::Request controllerRequest {
        getParam<maps::wiki::filters::TExpressionId>(request, "expression-id"),
        request.body(),
        branch,
        token,
        getParam<size_t>(request, "limit", 100),
    };
    handleController<editor::ObjectsQueryFilter>(response, controllerRequest, requestedFormat(request));
}


YCR_RESPOND_TO("POST /objects/query/route-diff",
    uid = NO_UID,
    branch = TRUNK_BRANCH_ID,
    token = "",
    sync = false)
{
    editor::ObjectsQueryRouteDiffParser parser;
    parser.parse(requestedFormat(request), request.body());

    editor::ObjectsQueryRouteDiff::Request controllerRequest{
        uid,
        token,
        branch,
        parser.revisionId(),
        parser.categoryId(),
        parser.addElementIds(),
        parser.removeElementIds(),
        parser.threadStopSequence(),
        parser.fromThreadStopIdx(),
        parser.toThreadStopIdx(),
        getParam<size_t>(request, "limit", 500)
    };

    handleController<editor::ObjectsQueryRouteDiff>(
        response,
        sync,
        controllerRequest,
        requestedFormat(request)
    );
}

YCR_RESPOND_TO("POST /objects/query/path",
    uid = NO_UID,
    branch = TRUNK_BRANCH_ID,
    token = "",
    sync = false)
{
    editor::ObjectsQueryPathParser parser;
    parser.parse(requestedFormat(request), request.body());

    editor::ObjectsQueryPath::Request controllerRequest{
        uid,
        token,
        branch,
        parser.elementIds(),
        getParam<size_t>(request, "limit", 500)
    };

    handleController<editor::ObjectsQueryPath>(
        response,
        sync,
        controllerRequest,
        requestedFormat(request)
    );
}

YCR_RESPOND_TO("POST /objects",
    uid, feedbackTaskId, branch = TRUNK_BRANCH_ID, sync = false)
{
    const auto format = requestedFormat(request);
    auto parser = editor::SaveObjectParser();
    parser.parse(format, request.body(),
        uid == common::WIKIMAPS_SPRAV_UID
            ? editor::AttributeErrorPolicy::EmptyResult
            : editor::AttributeErrorPolicy::Fail);
    const auto commitAttrs = gatherCommitAttributes(request);
    const auto userContext = makeUserContext(uid, request);
    const auto force = getParam<bool>(request, "force", true);
    const auto isLocalPolicy = getParam<bool>(request, "auto-is-local", false)
        ? editor::SaveObject::IsLocalPolicy::Auto
        : editor::SaveObject::IsLocalPolicy::Manual;
    const auto stickPolygonsPolicy = getParam<bool>(request, "stick-polygons", false)
        ? editor::SaveObject::StickPolygonsPolicy::On
        : editor::SaveObject::StickPolygonsPolicy::Off;
    if (parser.objects().isMultisave()) {
        WIKI_REQUIRE(!sync,
                editor::ERR_BAD_REQUEST,
                "Multiobject save should be performed in async mode.");
        for (const auto& object : parser.objects()) {
            const auto contextIt = parser.editContexts().find(object.id());
            WIKI_REQUIRE(contextIt != parser.editContexts().end(),
                editor::ERR_BAD_REQUEST,
                "Missing editContext for: " << object.id());
        }
    }
    editor::SaveObject::Request controllerRequest{
        userContext,
        force,
        branch,
        parser.objects(),
        parser.editContexts(),
        isLocalPolicy,
        feedbackTaskId,
        commitAttrs,
        request.body(),
        stickPolygonsPolicy
    };
    handleController<editor::SaveObject>(
        response, sync, controllerRequest, format);
}

YCR_RESPOND_TO("POST /objects/$/clone",
    uid, feedbackTaskId, branch = TRUNK_BRANCH_ID, sync = false)
{
    auto format = requestedFormat(request);
    auto parser = editor::SaveObjectParser();
    parser.parse(format, request.body(), editor::AttributeErrorPolicy::Fail);
    editor::CloneObject::Request controllerRequest{
            makeUserContext(uid, request),
            branch,
            getParam<editor::TOid>(argv, 0),
            parser.objects(),
            feedbackTaskId
        };
    handleController<editor::CloneObject>(
        response, sync, controllerRequest, format);
}

YCR_RESPOND_TO("POST /objects/update/attributes",
    uid,
    feedbackTaskId,
    branch = TRUNK_BRANCH_ID,
    sync = false)
{
    auto format = requestedFormat(request);
    editor::ObjectsUpdateAttributes::Request controllerRequest =
        {
            request.body(),
            makeUserContext(uid, request),
            branch,
            feedbackTaskId,
            format
        };
    handleController<editor::ObjectsUpdateAttributes>(
        response, sync, controllerRequest, format);
}

YCR_RESPOND_TO("GET /objects/export/meta",
    uid)
{
    auto formatType = requestedFormat(request);
    auto exportFormat = getParam<std::string>(request, "to");
    response[editor::HTTP_HEADER_CONTENT_TYPE] = httpContentType(formatType);
    WIKI_REQUIRE(
        formatType == common::FormatType::JSON &&
        exportFormat == "geojson",
        maps::wiki::ERR_UNSUPPORTED_FORMAT,
        "Unsupported formats requested.");
    response << R"({"categoryIds": ["mrc_region", "mrc_pedestrian_region"],  "maxObjectsCount": 1000})";
}

YCR_RESPOND_TO("GET /objects/update/update-attributes/meta",
    uid = NO_UID)
{
    handleController<editor::ObjectsUpdateAttributesMeta>(
        response, {uid}, requestedFormat(request));
}

YCR_RESPOND_TO("POST /objects/update/move",
    uid, feedbackTaskId, branch = TRUNK_BRANCH_ID, sync = false)
{
    auto format = requestedFormat(request);
    editor::ObjectsUpdateMove::Request controllerRequest =
        {
            request.body(),
            makeUserContext(uid, request),
            branch,
            feedbackTaskId,
            format
        };
    handleController<editor::ObjectsUpdateMove>(
        response, sync, controllerRequest, format);
}

YCR_RESPOND_TO("GET /objects/update/move/meta",
    uid = NO_UID)
{
    handleController<editor::ObjectsUpdateMoveMeta>(
        response, {uid}, requestedFormat(request));
}

YCR_RESPOND_TO("POST /objects/update/sync-geometry",
    uid, feedbackTaskId, branch = TRUNK_BRANCH_ID, sync = false)
{
    auto format = requestedFormat(request);
    editor::ObjectsUpdateSyncGeometry::Request controllerRequest =
        {
            request.body(),
            makeUserContext(uid, request),
            branch,
            feedbackTaskId,
            format
        };
    handleController<editor::ObjectsUpdateSyncGeometry>(
        response, sync, controllerRequest, format);
}

YCR_RESPOND_TO("GET /objects/update/sync-geometry/meta",
    uid = NO_UID)
{
    handleController<editor::ObjectsUpdateSyncGeometryMeta>(
        response, {uid}, requestedFormat(request));
}

YCR_RESPOND_TO("POST /objects/update/union",
    uid,
    branch = TRUNK_BRANCH_ID,
    sync = false)
{
    auto format = requestedFormat(request);
    editor::ObjectsUpdateUnion::Request controllerRequest =
        {
            request.body(),
            makeUserContext(uid, request),
            branch,
            format
        };
    handleController<editor::ObjectsUpdateUnion>(
        response, sync, controllerRequest, format);
}

YCR_RESPOND_TO("GET /objects/update/union/meta",
    uid = NO_UID)
{
    handleController<editor::ObjectsUpdateUnionMeta>(
        response, {uid}, requestedFormat(request));
}

YCR_RESPOND_TO("POST /objects/update/relation",
    uid,
    branch = TRUNK_BRANCH_ID,
    sync = false)
{
    auto format = requestedFormat(request);
    editor::ObjectsUpdateRelation::Request controllerRequest =
        {
            request.body(),
            makeUserContext(uid, request),
            branch,
            format
        };
    handleController<editor::ObjectsUpdateRelation>(
        response, sync, controllerRequest, format);
}

YCR_RESPOND_TO("GET /objects/update/relations/meta",
    uid = NO_UID)
{
    handleController<editor::ObjectsUpdateRelationsMeta>(
        response, {uid}, requestedFormat(request));
}

YCR_RESPOND_TO("POST /objects/update/relations",
    uid,
    feedbackTaskId,
    branch = TRUNK_BRANCH_ID,
    sync = false)
{
    auto format = requestedFormat(request);
    editor::ObjectsUpdateRelations::Request controllerRequest =
        {
            request.body(),
            makeUserContext(uid, request),
            branch,
            feedbackTaskId,
            format
        };
    handleController<editor::ObjectsUpdateRelations>(
        response, sync, controllerRequest, format);
}

YCR_RESPOND_TO("POST /objects/update/merge",
    uid,
    branch = TRUNK_BRANCH_ID,
    sync = false)
{
    auto format = requestedFormat(request);
    editor::ObjectsUpdateMerge::Request controllerRequest =
        {
            request.body(),
            makeUserContext(uid, request),
            branch,
            format
        };
    handleController<editor::ObjectsUpdateMerge>(
            response, sync, controllerRequest, format);
}

YCR_RESPOND_TO("GET /objects/update/snap/meta",
    uid = NO_UID)
{
    handleController<editor::ObjectsUpdateSnapMeta>(
        response, {uid}, requestedFormat(request));
}

YCR_RESPOND_TO("POST /objects/update/snap",
    uid, branch = TRUNK_BRANCH_ID, sync = false)
{
    auto format = requestedFormat(request);
    editor::ObjectsUpdateSnap::Request controllerRequest = {
        request.body(),
        makeUserContext(uid, request),
        branch,
        format
    };
    handleController<editor::ObjectsUpdateSnap>(
        response, sync, controllerRequest, format);
}

YCR_RESPOND_TO("POST /create-intersections",
    uid, feedbackTaskId, branch = TRUNK_BRANCH_ID)
{
    auto format = requestedFormat(request);
    auto parser = editor::CreateIntersectionsParser();
    parser.parse(format, request.body());
    editor::CreateIntersections::Request controllerRequest{
            makeUserContext(uid, request),
            branch,
            parser.objects(),
            parser.editContext(),
            feedbackTaskId
        };
    handleController<editor::CreateIntersections>(
        response, /* bool sync = */ false, controllerRequest, format);
}

YCR_RESPOND_TO("GET /results/$", dummyParam = 0)
{
    editor::GetAsyncResult::Request controllerRequest(
        getParam<std::string>(argv, 0));
    handleController<editor::GetAsyncResult>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /objects/$/state/$",
    uid, feedbackTaskId, branch = TRUNK_BRANCH_ID, sync = false)
{
    auto format = requestedFormat(request);
    editor::ObjectsUpdateState::Request controllerRequest(
        makeUserContext(uid, request),
        getParam<editor::TOid>(argv, 0),
        getParam<std::string>(argv, 1),
        branch,
        editor::s_emptyString,
        format,
        feedbackTaskId
    );
    handleController<editor::ObjectsUpdateState>(
        response, sync, controllerRequest, format);
}

YCR_RESPOND_TO("POST /objects/update/state",
    uid, feedbackTaskId, branch = TRUNK_BRANCH_ID, sync = false)
{
    auto format = requestedFormat(request);
    editor::ObjectsUpdateState::Request controllerRequest(
        makeUserContext(uid, request),
        0,
        "",
        branch,
        request.body(),
        format,
        feedbackTaskId
    );
    handleController<editor::ObjectsUpdateState>(
        response, sync, controllerRequest, format);
}

YCR_RESPOND_TO("GET /objects/update/state/meta",
    uid = NO_UID)
{
    handleController<editor::ObjectsUpdateStateMeta>(
        response, {uid}, requestedFormat(request));
}

YCR_RESPOND_TO("GET /objects/$/topology", uid = NO_UID, branch = TRUNK_BRANCH_ID, token = "")
{
    editor::GetTopology::Request controllerRequest {
        getParam<editor::TOid>(argv, 0),
        token,
        getParam<std::string>(request, "bb"),
        branch
    };
    handleController<editor::GetTopology>(response, controllerRequest, requestedFormat(request));
}

const size_t MAXIMUM_SLAVES_PER_REQUEST = 1000;

YCR_RESPOND_TO("GET /objects/$/slaves",
    startId,
    beforeAfter,
    limit,
    branch = TRUNK_BRANCH_ID,
    token = "")
{
    const auto& input = request.input();
    std::string bb;
    size_t offset = 0;
    size_t limit = std::min(getParam<size_t>(request, "limit", 0), MAXIMUM_SLAVES_PER_REQUEST);
    if (input.has("bb")) {
        bb = getParam<std::string>(request, "bb", "");
    } else if (input.has("limit")) {
        offset = getParam<size_t>(request, "offset", 0);
    } else {
        THROW_WIKI_LOGIC_ERROR(editor::ERR_BAD_REQUEST,
            "Missing required parameters: bb or limit");
    }
    const auto format = requestedFormat(request);
    editor::GetSlaveInfos::Request controllerRequest {
            getParam<editor::TOid>(argv, 0),
            token,
            getParam<std::string>(request, "role-id"),
            getParam<std::string>(request, "category-id"),
            offset,// TODO: Drop with xml
            limit,
            bb,
            branch,
            beforeAfter,
            startId,
            format
        };
    handleController<editor::GetSlaveInfos>(response, controllerRequest, format);
}

YCR_RESPOND_TO("GET /objects/$/$/rich-content", uid = NO_UID, branch = TRUNK_BRANCH_ID, token = "")
{
    auto objectId = getParam<editor::TOid>(argv, 0);
    auto revisionId = getParam<editor::TRevisionId>(argv, 1);
    editor::GetRichContent::Request controllerRequest{
        uid,
        objectId,
        revisionId,
        branch,
        token
    };

    handleBinaryController<editor::GetRichContent>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /objects/$/branch-diff", uid, token = "")
{
    auto objectId = getParam<editor::TOid>(argv, 0);
    auto fromBranchId = getParam<editor::TBranchId>(request, "from-branch");
    auto toBranchId = getParam<editor::TBranchId>(request, "to-branch");
    editor::GetBranchDiff::Request controllerRequest {
        objectId,
        fromBranchId,
        toBranchId,
        uid,
        token,
    };

    handleController<editor::GetBranchDiff>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /objects/$/relations/cleanup",
    uid, branch = TRUNK_BRANCH_ID, sync = false)
{
    editor::ObjectsCleanupRelations::Request controllerRequest{
        makeUserContext(uid, request),
        getParam<editor::TOid>(argv, 0),
        branch
    };
    handleController<editor::ObjectsCleanupRelations>(
        response, sync, std::move(controllerRequest), requestedFormat(request));
}

YCR_RESPOND_TO("GET /renderer/sublayers", uid = NO_UID, token = "")
{
    editor::GetLayers::Request controllerRequest {
        uid,
        getParam<std::string>(request, "map-type", editor::s_emptyString),
        getParam<std::string>(request, "project", editor::s_emptyString),
        token
    };
    handleController<editor::GetLayers>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /renderer/sublayers2", uid = NO_UID, token = "")
{
    editor::GetLayers2::Request controllerRequest {
        uid,
        getParam<std::string>(request, "map-type", editor::s_emptyString),
        getParam<std::string>(request, "project", editor::s_emptyString),
        token
    };
    handleController<editor::GetLayers2>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO(
    "GET /renderer/sublayers2/cartograph", token = "",
    YCR_USING(yacare::Tvm2ServiceRequire("nmaps-editor")))
{
    // maps-front-cartograph-admin tvm ids:
    // https://abc.yandex-team.ru/services/maps-front-cartograph-admin/resources/?view=consuming&layout=table&supplier=14&type=47
    std::unordered_set<unsigned int> CARTOGRAPH_TVM_IDS = {2011524, 2011526, 2011580};
    const std::string SERVICE_TICKET_HEADER = "HTTP_X_YA_SERVICE_TICKET";

    const auto tvmServiceTicket = request.env(SERVICE_TICKET_HEADER);
    const auto tvmSrcId = editor::cfg()->getTvmSrcId(tvmServiceTicket);

    editor::GetLayers2::Request controllerRequest {
        NO_UID,
        getParam<std::string>(request, "map-type", editor::s_emptyString),
        getParam<std::string>(request, "project", editor::s_emptyString),
        token
    };
    if (tvmSrcId && CARTOGRAPH_TVM_IDS.contains(tvmSrcId.value())) {
        controllerRequest.filterByAcl = editor::GetLayers2::FilterPolicy::DoNotFilter;
    }
    handleController<editor::GetLayers2>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /commits/$", uid = NO_UID, branch = TRUNK_BRANCH_ID, token = "")
{
    editor::GetCommit::Request controllerRequest {
        getParam<editor::TCommitId>(argv, 0),
        uid,
        branch,
        token};
    handleController<editor::GetCommit>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /commits/$/dependent",
    uid, token = "")
{
    editor::GetDependentCommits::Request controllerRequest {
        uid,
        getParam<editor::TCommitId>(argv, 0),
        getParam<size_t>(request, "page", 1),
        getParam<size_t>(request, "per-page", 10),
        token};
    handleController<editor::GetDependentCommits>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /commits/$/diff",
    uid = NO_UID, branch = TRUNK_BRANCH_ID, token = "")
{
    editor::GetCommitDiff::Request controllerRequest {
        getParam<editor::TCommitId>(argv, 0),
        getParam<editor::TOid>(request, "oid", 0),
        uid,
        token,
        branch};
    handleController<editor::GetCommitDiff>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /commits/$/diff/geometry",
    uid = NO_UID,
    branch = TRUNK_BRANCH_ID, token = "")
{
    editor::GetCommitGeomDiff::Request controllerRequest {
        getParam<editor::TCommitId>(argv, 0),
        getParam<editor::TOid>(request, "oid", 0),
        uid,
        token,
        branch,
        getParam<std::string>(request, "bb"),
        getParam<editor::TZoom>(request, "z")
    };
    handleController<editor::GetCommitGeomDiff>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /commits/$/approve", uid, sync = false)
{
    editor::CommitsApprove::Request controllerRequest(
        uid,
        getParam<editor::TCommitId>(argv, 0));
    handleController<editor::CommitsApprove>(response, sync, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /commits/$/revert",
    uid, feedbackTaskId, sync = false)
{
    editor::CommitsRevert::Request controllerRequest(
        makeUserContext(uid, request),
        getParam<editor::TCommitId>(argv, 0),
        getOptionalParam<editor::RevertReason>(request, "revertReason"),
        feedbackTaskId
    );
    handleController<editor::CommitsRevert>(response, sync, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /token", branch = TRUNK_BRANCH_ID)
{
    editor::CreateToken::Request controllerRequest{
            branch
    };
    handleController<editor::CreateToken>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /objects/query/junction",
    uid, feedbackTaskId, branch = TRUNK_BRANCH_ID, sync = false)
{
    editor::ObjectsQueryJunction::Request controllerRequest{
        request.body(),
        getParam<editor::TZoom>(request, "z"),
        getParam<editor::TOid>(request, "oid"),
        makeUserContext(uid, request),
        branch,
        feedbackTaskId
    };
    handleController<editor::ObjectsQueryJunction>(
        response, sync, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /sprav-feedback/v1/tasks", uid)
{
    editor::CreateSpravTask::Request controllerRequest {
        request.body(),
        uid,
        getParam<editor::TOid>(request, "object-id"),
        };
    handleController<editor::CreateSpravTask>(response, controllerRequest, common::FormatType::JSON);
}

YCR_RESPOND_TO("GET /sprav-feedback/v1/tasks",
    uid,
    token = "")
{
    editor::GetSpravTasks::Request controllerRequest {
        uid,
        getParam<editor::TUid>(request, "created-by", 0),
        getParam<editor::TOid>(request, "object-id", 0),
        request.input().has("statuses") && !getParam<std::string>(request, "statuses").empty()
            ? editor::splitCast<std::vector<editor::SpravTask::Status>>(
                getParam<std::string>(request, "statuses"), ',')
            : std::vector<editor::SpravTask::Status>(),
        token,
        getParam<size_t>(request, "offset", 0),
        getParam<size_t>(request, "limit", 0)
        };
    handleController<editor::GetSpravTasks>(response, controllerRequest, common::FormatType::JSON);
}

} // YCR_USE
