#include "handles.h"

namespace chrono = maps::chrono;
namespace editor = maps::wiki;
namespace mws = maps::wiki::social;

namespace {

yacare::ThreadPool socialThreadPool("social", 2, 20);
yacare::ThreadPool socialFeedsThreadPool("social-feeds", 3, 30);
yacare::ThreadPool socialModerationThreadPool("social-moderation", 3, 30);
yacare::ThreadPool socialCommentsThreadPool("social-comments", 2, 20);

} // namespace

YCR_USE(socialFeedsThreadPool) {

YCR_RESPOND_TO("GET /social/feeds/$/$",
   uid = NO_UID,
   branch = TRUNK_BRANCH_ID,
   token = "",
   skipCreatedBy,
   since,
   till)
{
    std::vector<maps::wiki::acl_utils::UserKind> skippedUserKinds;
    for (const auto& str : skipCreatedBy) {
        try {
            auto enumValue = maps::enum_io::fromString<maps::wiki::acl_utils::UserKind>(str);
            skippedUserKinds.push_back(enumValue);
        } catch (maps::RuntimeError& e) {
            THROW_WIKI_LOGIC_ERROR(editor::ERR_BAD_REQUEST,
                "Invalid skip-created-by parameter value: '" << str << "'\n" << e.what());
        }
    }

    editor::GetRegularSocialFeed::Request controllerRequest{
        uid,
        branch,
        getParam<editor::social::FeedType>(argv, 0),
        getParam<editor::social::TId>(argv, 1),
        getOptionalParam<std::string>(request, "category-group"),
        skippedUserKinds,
        getParam<bool>(request, "preapproved-only", false)
            ? editor::GetRegularSocialFeed::PreApprovedOnly::Yes
            : editor::GetRegularSocialFeed::PreApprovedOnly::No,
        getParam<size_t>(request, "page", 0),
        getParam<size_t>(request, "per-page", 10),
        getParam<editor::social::TId>(request, "before", 0),
        getParam<editor::social::TId>(request, "after", 0),
        token,
        getParam<bool>(request, "with-total", false)
            ? editor::GetRegularSocialFeed::WithTotal::Yes
            : editor::GetRegularSocialFeed::WithTotal::No,
        since,
        till
    };
    handleController<editor::GetRegularSocialFeed>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/feeds/meta",
               uid = NO_UID, token = "")
{
    editor::GetRegularSocialFeedMeta::Request controllerRequest{ uid, token };
    handleController<editor::GetRegularSocialFeedMeta>
        (response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/feeds/suspicious",
   uid = NO_UID,
   branch = TRUNK_BRANCH_ID,
   token = "",
   since,
   till)
{
    auto actions = editor::splitCast<std::vector<editor::social::FeedAction>>(
        getParam<std::string>(request, "action", "commit-reverted,object-deleted"), ',');
    for (const auto& action: actions) {
        WIKI_REQUIRE(
            action == editor::social::FeedAction::CommitReverted ||
            action == editor::social::FeedAction::ObjectDeleted,
            editor::ERR_BAD_REQUEST,
            "action is not allowed.");
    }

    editor::GetSuspiciousSocialFeed::Request controllerRequest{
            uid,
            branch,
            editor::splitCast<std::vector<editor::social::TUid>>(
                getParam<std::string>(request, "target-uid", ""), ','),
            getOptionalParam<editor::social::ModerationMode>(request, "target-mode"),
            since,
            till,
            actions,
            getParam<size_t>(request, "per-page", 10),
            getParam<editor::social::TId>(request, "before", 0),
            getParam<editor::social::TId>(request, "after", 0),
            token};
    handleController<editor::GetSuspiciousSocialFeed>(response, controllerRequest, requestedFormat(request));
}

} // YCR_USE(socialFeedsThreadPool)

YCR_USE(socialModerationThreadPool) {

YCR_RESPOND_TO("GET /social/moderation/regions/$",
               uid, token = "")
{
    auto mode = getOptionalParam<editor::social::ModerationMode>(request, "mode");
    auto countByGroup = getParam<bool>(request, "count-by-category-group", false);
    auto aoi = getParam<editor::TOid>(argv, 0);

    editor::SocialModerationRegions::Request controllerRequest{
        uid, token, mode,
        getEventType(request),
        getOptionalParam<editor::TUid>(request, "created-by"),
        getOptionalParam<editor::TOid>(request, "primary-oid"),
        aoi, countByGroup};
    handleController<editor::SocialModerationRegions>(
        response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/moderation/regions",
               uid, token = "")
{
    auto mode = getOptionalParam<editor::social::ModerationMode>(request, "mode");
    auto countByGroup = getParam<bool>(request, "count-by-category-group", false);

    editor::SocialModerationRegions::Request controllerRequest{
        uid, token, mode,
        /*eventType*/boost::none,
        /*createdBy*/boost::none,
        /*primaryObjectId*/boost::none,
        /*aoiId*/0, countByGroup};
    handleController<editor::SocialModerationRegions>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/moderation/stat",
               uid, token = "")
{
    editor::GetSocialModerationStat::Request controllerRequest{
        uid,
        token};
    handleController<editor::GetSocialModerationStat>(
        response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/moderation/tasks/$",
               uid, token = "")
{
    editor::GetSocialModerationTask::Request controllerRequest{
        uid,
        getParam<editor::social::TId>(argv, 0),
        token};
    handleController<editor::GetSocialModerationTask>(
        response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/moderation/task",
               uid, token = "")
{
    editor::GetSocialModerationTaskByCommit::Request controllerRequest{
        uid,
        getParam<editor::TCommitId>(request, "commit-id", 0),
        token};
    handleController<editor::GetSocialModerationTaskByCommit>(
        response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /social/moderation/tasks/acquire",
               uid, aoi, sync = false)
{
    editor::SocialModerationTasksAcquire::Request::SuspiciousOnly suspiciousOnly =
        getParam<bool>(request, "suspicious-users", false)
        ? editor::SocialModerationTasksAcquire::Request::SuspiciousOnly::Yes
        : editor::SocialModerationTasksAcquire::Request::SuspiciousOnly::No;
    editor::SocialModerationTasksAcquire::Request::NovicesOnly novicesOnly =
        getParam<bool>(request, "novice-users", false)
        ? editor::SocialModerationTasksAcquire::Request::NovicesOnly::Yes
        : editor::SocialModerationTasksAcquire::Request::NovicesOnly::No;

    editor::SocialModerationTasksAcquire::Request controllerRequest{
        uid,
        aoi,
        getParam<editor::social::ModerationMode>(request, "mode"),
        getOptionalParam<std::string>(request, "category-group"),
        getEventType(request),
        getOptionalParam<editor::TUid>(request, "created-by"),
        getOptionalParam<editor::TUid>(request, "resolved-by"),
        getOptionalParam<editor::TOid>(request, "primary-oid"),
        getParam<size_t>(request, "limit"),
        getParam<bool>(request, "reversed", false)
            ? mws::TasksOrder::OldestFirst
            : mws::TasksOrder::NewestFirst,
        suspiciousOnly,
        novicesOnly
    };
    handleController<editor::SocialModerationTasksAcquire>(
        response, sync, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /social/moderation/tasks/acquire/blocking",
               uid, sync = false)
{
    editor::SocialModerationTasksAcquireBlocking::Request controllerRequest{
        uid,
        getParam<editor::TCommitId>(request, "commit-id")
    };
    handleController<editor::SocialModerationTasksAcquireBlocking>(
        response, sync, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /social/moderation/tasks/release",
               uid, sync = false)
{
    editor::SocialModerationTasksRelease::Request controllerRequest{uid};
    handleController<editor::SocialModerationTasksRelease>(
        response, sync, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /social/moderation/tasks/resolve",
               uid, sync = false)
{
    editor::SocialModerationTasksResolve::Request controllerRequest{
        makeUserContext(uid, request),
        getOptionalParam<editor::TOid>(request, "aoi"),
        request.body()};
    handleController<editor::SocialModerationTasksResolve>(
        response, sync, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /social/moderation/tasks/close",
               uid, sync = false)
{
    const auto NO_AOI = boost::none;

    editor::SocialModerationTasksClose::Request controllerRequest{
        makeUserContext(uid, request),
        NO_AOI,
        request.body()};
    handleController<editor::SocialModerationTasksClose>(
        response, sync, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /social/moderation/tasks/defer",
               uid, sync = false)
{
    editor::SocialModerationTasksDefer::Request controllerRequest{
        uid,
        request.body()};
    handleController<editor::SocialModerationTasksDefer>(
        response, sync, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/moderation/tasks/stat",
               uid, token = "")
{
    editor::SocialModerationTasksStat::Request controllerRequest{
        uid,
        getParam<int>(request, "tz"),
        token};
    handleController<editor::SocialModerationTasksStat>(
        response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/moderation/tasks",
               uid, token = "")
{
    editor::GetSocialModerationTasks::Request controllerRequest{
        uid,
        token,
        getParam<mws::TId>(request, "before", 0),
        getParam<mws::TId>(request, "after", 0),
        getParam<size_t>(request, "per-page", mws::TaskFeedParams::perPageDefault),
        getStdOptionalParam<std::string>(request, "category-group"),
        getStdOptionalParam<mws::TUid>(request, "resolved-by"),
        mws::DateTimeCondition::makeOptional(
            getStdOptionalParam<chrono::TimePoint>(request, "resolved-after"),
            getStdOptionalParam<chrono::TimePoint>(request, "resolved-before")),
        getStdOptionalParam<mws::ResolveResolution>(request, "resolve-resolution"),
        getStdOptionalParam<mws::TUid>(request, "closed-by"),
        mws::DateTimeCondition::makeOptional(
            getStdOptionalParam<chrono::TimePoint>(request, "closed-after"),
            getStdOptionalParam<chrono::TimePoint>(request, "closed-before")),
        getStdOptionalParam<mws::CloseResolution>(request, "close-resolution")};
    handleController<editor::GetSocialModerationTasks>(
        response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/moderation/dashboard",
               uid, token = "")
{
    editor::SocialModerationDashboard::Request controllerRequest{
        uid,
        getParam<editor::social::ModerationMode>(
            request, "mode", editor::social::ModerationMode::Moderator),
        getParam<bool>(request, "tree-output", false)
            ? editor::SocialModerationDashboard::OutputType::Tree
            : editor::SocialModerationDashboard::OutputType::Flat,
        token,
        getOptionalParam<std::string>(request, "category-group"),
        getOptionalParam<std::string>(request, "event-type")};
    handleController<editor::SocialModerationDashboard>(
        response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/moderation/dashboard/meta",
               uid = NO_UID)
{
    handleController<editor::SocialModerationDashboardMeta>
        (response, {uid}, maps::wiki::common::FormatType::JSON);
}

} // YCR_USE(socialModerationThreadPool)

YCR_USE(socialCommentsThreadPool) {

YCR_RESPOND_TO("POST /social/comments",
               uid, sync = false)
{
    auto controllerRequest =
        editor::CommentsCreate::Request::parseJson(
            makeUserContext(uid, request),
            request.body());
    handleController<editor::CommentsCreate>(
        response, sync, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("POST /social/comments/$/delete",
               uid, sync = false)
{
    editor::CommentsDelete::Request controllerRequest{
        makeUserContext(uid, request),
        getParam<editor::TId>(argv, 0)};
    handleController<editor::CommentsDelete>(
        response, sync, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("DELETE /social/comments",
               uid, sync = false)
{
    editor::ClearUserComments::Request controllerRequest{
        makeUserContext(uid, request),
        getParam<editor::TUid>(request, "created-by", 0)
    };
    handleController<editor::ClearUserComments>(
        response, sync, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/comments",
               aoi = 0,
               uid = NO_UID, token = "")
{
    editor::GetCommentsFeed::Request controllerRequest{
            editor::FeedParams::create(
                uid,
                getParam<editor::TOid>(request, "oid", 0),
                getParam<editor::TCommitId>(request, "commit-id", 0),
                getOptionalParam<editor::TId, std::optional<editor::TId>>(request, "feedback-task-id"),
                getParam<editor::TOid>(request, "aoi", 0),
                getParam<editor::TUid>(request, "created-by", 0),
                getParam<mws::CommentTypes>(request, "types", mws::comments::ANY_TYPE),
                getParam<bool>(request, "with-deleted", false)
                    ? mws::comments::DeletedPolicy::Show
                    : mws::comments::DeletedPolicy::Hide,
                token
            ),
            getParam<size_t>(request, "per-page", 10),
            getParam<editor::social::TId>(request, "after", 0),//older
            getParam<editor::social::TId>(request, "before", 0),//newer
            token};
    handleController<editor::GetCommentsFeed>(response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/comments/stat",
               aoi = 0,
               uid = NO_UID, token = "")
{
    editor::GetCommentsStat::Request controllerRequest{
            editor::FeedParams::create(
                uid,
                getParam<editor::TOid>(request, "oid", 0),
                getParam<editor::TCommitId>(request, "commit-id", 0),
                getOptionalParam<editor::TId, std::optional<editor::TId>>(request, "feedback-task-id"),
                aoi,
                getParam<editor::TUid>(request, "created-by", 0),
                getParam<mws::CommentTypes>(request, "types", mws::comments::ANY_TYPE),
                mws::comments::DeletedPolicy::Hide,
                token
            ),
            token};
    handleController<editor::GetCommentsStat>(response, controllerRequest, requestedFormat(request));
}

} // YCR_USE(socialCommentsThreadPool)

YCR_USE(socialThreadPool) {

YCR_RESPOND_TO("GET /social/subscriptions",
               uid, token = {})
{
    editor::GetSocialSubscriptions::Request controllerRequest{
        uid,
        token
    };
    handleController<editor::GetSocialSubscriptions>(
            response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/events/$",
               uid, token = {})
{
    editor::GetSocialEvent::Request controllerRequest{
        uid,
        getParam<editor::TId>(argv, 0),
        token
    };
    handleController<editor::GetSocialEvent>(
        response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/feedback/tasks/$/history",
               uid, token = {})
{
    editor::GetSocialFeedbackTaskHistory::Request controllerRequest{
        uid,
        getParam<editor::TId>(argv, 0),
        token,
        getParam<size_t>(request, "per-page", 10),
        getOptionalParam<editor::social::TId>(request, "before"),
        getOptionalParam<editor::social::TId>(request, "after")
    };
    handleController<editor::GetSocialFeedbackTaskHistory>(
        response, controllerRequest, requestedFormat(request));
}

YCR_RESPOND_TO("GET /social/feedback/tasks/$/stat",
               uid, token = {})
{
    editor::GetSocialFeedbackTaskStat::Request controllerRequest{
        uid,
        getParam<editor::TId>(argv, 0),
        token
    };
    handleController<editor::GetSocialFeedbackTaskStat>(
        response, controllerRequest, requestedFormat(request));
}

} // YCR_USE(socialThreadPool)
