#pragma once

#include <maps/wikimap/mapspro/services/editor/src/exception.h>
#include <maps/wikimap/mapspro/services/editor/src/moderation.h>
#include <maps/wikimap/mapspro/services/editor/src/user_context.h>

#include <maps/wikimap/mapspro/libs/controller/include/exception_info.h>
#include <maps/infra/yacare/include/yacare.h>
#include <yandex/maps/wiki/common/before_after.h>
#include <yandex/maps/wiki/common/compound_token.h>
#include <yandex/maps/wiki/common/format_type.h>
#include <yandex/maps/wiki/revision/common.h>
#include <yandex/maps/wiki/social/comments_feed.h>
#include <yandex/maps/wiki/validator/storage/stored_message_data.h>
#include <maps/libs/chrono/include/time_point.h>

class UidParamType
{
public:
    UidParamType() = default;

    void assign(maps::wiki::TUid uid);

    operator maps::wiki::TUid() const { return uid_; }

private:
    maps::wiki::TUid uid_ = 0;
};

const UidParamType NO_UID;
const auto TRUNK_BRANCH_ID = maps::wiki::revision::TRUNK_BRANCH_ID;

std::istream& operator >> (std::istream& is, UidParamType& uidParam);

maps::wiki::common::FormatType
requestedFormat(const yacare::Request& r);

std::string
httpContentType(maps::wiki::common::FormatType formatType);

boost::optional<maps::chrono::TimePoint>
getOptionalTimePointParam(const yacare::Request& request, const std::string& name);


maps::wiki::StringMap
gatherCommitAttributes(const yacare::Request& request);

boost::optional<maps::wiki::social::EventType>
getEventType(const yacare::Request& r);


void wrapError(yacare::Response& resp, const maps::wiki::controller::ExceptionInfo& exInfo);
void yacareErrorReporter(const yacare::Request& req, yacare::Response& resp);


inline maps::wiki::UserContext makeUserContext(maps::wiki::TUid uid, const yacare::Request& r)
{
    return {uid, r.getClientIpAddress(), r.getClientPort()};
}

template <typename T, typename Optional = std::optional<T>>
Optional
getStdOptionalParam(const yacare::Request& r, const std::string& name)
{
    std::string strParam = r.input()[name];
    if (strParam.empty()) {
        return {};
    }
    try {
        return boost::lexical_cast<T>(strParam);
    } catch (const boost::bad_lexical_cast&) {
        THROW_WIKI_LOGIC_ERROR(maps::wiki::ERR_BAD_REQUEST,
            " Invalid parameter: " << name << " : " << strParam);
    }
}

template<>
inline std::optional<maps::chrono::TimePoint>
getStdOptionalParam<maps::chrono::TimePoint>(const yacare::Request& r, const std::string& name)
{
    const auto strParam = r.input()[name];
    if (strParam.empty()) {
        return {};
    }
    try {
        return maps::chrono::parseIsoDateTime(strParam);
    } catch (const maps::Exception&) {
        THROW_WIKI_LOGIC_ERROR(maps::wiki::ERR_BAD_REQUEST,
            " Invalid parameter: " << name << " : " << strParam);
    }
}

template <typename T, typename Optional = boost::optional<T>>
Optional
getOptionalParam(const yacare::Request& r, const std::string& name)
{
    std::string strParam = r.input()[name];
    if (strParam.empty()) {
        return {};
    }
    try {
        return boost::lexical_cast<T>(strParam);
    } catch (const boost::bad_lexical_cast&) {
        THROW_WIKI_LOGIC_ERROR(maps::wiki::ERR_BAD_REQUEST,
            " Invalid parameter: " << name << " : " << strParam);
    }
}

template <>
inline boost::optional<bool>
getOptionalParam<bool>(const yacare::Request& r, const std::string& name)
{
    std::string strParam = r.input()[name];
    if (strParam.empty()) {
        return boost::none;
    }
    if (strParam == "true") {
        return true;
    }
    if (strParam == "false") {
        return false;
    }

    THROW_WIKI_LOGIC_ERROR(
            maps::wiki::ERR_BAD_REQUEST,
            " Invalid parameter: " << name << " : " << strParam);
}

template <>
inline boost::optional<maps::wiki::social::CommentTypes>
getOptionalParam<maps::wiki::social::CommentTypes>(const yacare::Request& r, const std::string& name)
{
    std::string strParam = r.input()[name];
    if (strParam.empty()) {
        return boost::none;
    }
    return  maps::wiki::splitCast<maps::wiki::social::CommentTypes>(strParam,',');
};

template <>
inline boost::optional<maps::wiki::social::ModerationMode>
getOptionalParam<maps::wiki::social::ModerationMode>(const yacare::Request& r, const std::string& name)
{
    std::string strParam = r.input()[name];
    if (strParam.empty()) {
        return boost::none;
    }
    const auto& str2modeMap = maps::wiki::moderation::getModerationModeResolver();
    auto it = str2modeMap.find(strParam);
    WIKI_REQUIRE(
        it != str2modeMap.end(),
        maps::wiki::ERR_BAD_REQUEST,
        " Invalid parameter: " << name << " : " << strParam);
    return it->second;
}

template <class T>
T
getParam(const yacare::Request& r, const std::string& name, const T& defaultVal)
{
    auto param = getOptionalParam<T>(r, name);
    return param ? *param : defaultVal;
}

template <class T>
T
getParam(const yacare::Request& r, const std::string& name)
{
    auto value = getOptionalParam<T>(r, name);
    if (!value) {
        THROW_WIKI_LOGIC_ERROR(maps::wiki::ERR_BAD_REQUEST, " Missing parameter: " << name);
    }
    return *value;
}

template <class T>
T
getParam(const std::vector<std::string>& argv, size_t index)
{
    if (index >= argv.size()) {
        THROW_WIKI_LOGIC_ERROR(maps::wiki::ERR_BAD_REQUEST,
            " Bad parameters count. " << index << ">=" << argv.size() << "!");
    }
    try {
        return boost::lexical_cast<T>(argv[index]);
    } catch (const boost::bad_lexical_cast&) {
        THROW_WIKI_LOGIC_ERROR(maps::wiki::ERR_BAD_REQUEST,
            " Invalid parameter: $" << index + 1 << " : " << argv[index]);
    }
}

YCR_QUERY_PARAM(uid, UidParamType);
YCR_QUERY_PARAM(aoi, maps::wiki::TOid);
YCR_QUERY_PARAM(branch, maps::wiki::TBranchId);
YCR_QUERY_PARAM(sync, bool);
YCR_QUERY_PARAM(ids, std::string);
YCR_QUERY_PARAM(dummyParam, uint64_t);

YCR_QUERY_CUSTOM_PARAM((), skipCreatedBy, std::vector<std::string>, YCR_DEFAULT(std::vector<std::string>()))
{
    return yacare::impl::parseArg(dest, request, "skip-created-by", true);
}

YCR_QUERY_CUSTOM_PARAM((), feedbackTaskId, boost::optional<maps::wiki::TId>)
{
    dest = getOptionalParam<maps::wiki::TId>(request, "feedback-task-id");
    return true;
}

YCR_QUERY_CUSTOM_PARAM((), token, std::string)
{
    auto token = request.input()["token"];
    WIKI_REQUIRE(
        maps::wiki::common::CompoundToken::isValid(token),
        maps::wiki::ERR_BAD_REQUEST,
        "Invalid token");
    dest = token;
    return true;
}

YCR_QUERY_CUSTOM_PARAM((), startId, maps::wiki::TId)
{
    if (request.input().has("after")) {
        return yacare::impl::parseArg(dest, request, "after");
    } else if (request.input().has("before")) {
        return yacare::impl::parseArg(dest, request, "before");
    } else {
        dest = 0;
    }
    return true;
}

YCR_QUERY_CUSTOM_PARAM((), startMessageId, maps::wiki::validator::storage::MessageId)
{
    if (request.input().has("after")) {
        return yacare::impl::parseArg(dest, request, "after");
    } else if (request.input().has("before")) {
        return yacare::impl::parseArg(dest, request, "before");
    } else {
        dest = maps::wiki::validator::storage::MessageId();
    }
    return true;
}

YCR_QUERY_CUSTOM_PARAM((), beforeAfter, maps::wiki::common::BeforeAfter)
{
    return maps::wiki::common::queryBeforeAfterParam(request, dest);
}

YCR_QUERY_PARAM(limit, size_t, YCR_DEFAULT(0));

YCR_QUERY_CUSTOM_PARAM((), namePart, std::string, YCR_DEFAULT(""))
{ return yacare::impl::parseArg(dest, request, "name-part"); }

YCR_QUERY_CUSTOM_PARAM((), since, boost::optional<maps::chrono::TimePoint>)
{
    dest = getOptionalTimePointParam(request, "since");
    return true;
}

YCR_QUERY_CUSTOM_PARAM((), till, boost::optional<maps::chrono::TimePoint>)
{
    dest = getOptionalTimePointParam(request, "till");
    return true;
}
