#include "approve_status.h"

#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/social/gateway.h>
#include <maps/wikimap/mapspro/libs/revision_meta/include/commit_regions.h>

namespace maps {
namespace wiki {
namespace approve_status {

namespace {

const std::map<ApproveStatus, std::string> APPROVE_STATUS_TO_STR = {
    {ApproveStatus::Pending, "pending"},
    {ApproveStatus::Approved, "approved"},
    {ApproveStatus::Rejected, "rejected"},
    {ApproveStatus::Published, "published"}
};

struct TaskStatus
{
    TaskStatus() = default;

    explicit TaskStatus(const social::Task& task)
    {
        if (task.isResolved()) {
            resolveResolution = task.resolved().resolution();
        }
        if (task.isClosed()) {
            closeResolution = task.closed().resolution();
        }
    }

    boost::optional<social::ResolveResolution> resolveResolution;
    boost::optional<social::CloseResolution> closeResolution;
};

ApproveStatus
calculateApproveStatus(
    bool published,
    const boost::optional<TaskStatus>& taskStatus)
{
    if (taskStatus &&
        (taskStatus->resolveResolution == social::ResolveResolution::Revert ||
         taskStatus->closeResolution   == social::CloseResolution::Revert))
    {
        return ApproveStatus::Rejected;
    }

    // A commit could possibly has a non-closed task (for example, if
    // `revisionapi` tool has been used), however if it is already published,
    // the status must reflect this fact.
    if (published) {
        return ApproveStatus::Published;
    }

    if (taskStatus && !taskStatus->closeResolution) {
        return ApproveStatus::Pending;
    }

    return ApproveStatus::Approved;
}

void
assertEventsMatchCommits(
    const social::Events& events,
    const std::list<revision::Commit>& commits)
{
    ASSERT(events.size() == commits.size());

    TCommitIds commitIdsFromEvents;
    for (const auto& event: events) {
        if (event.commitData()) {
            commitIdsFromEvents.insert(event.commitData()->commitId());
        }
    }

    TCommitIds commitIds;
    for (const auto& commit: commits) {
        commitIds.insert(commit.id());
    }
    ASSERT(commitIds == commitIdsFromEvents);
}

std::map<social::TId, TaskStatus>
loadTaskStatusesByEventIds(const BranchContext& branchCtx, const social::TIds& eventIds)
{
    social::Gateway sgw(branchCtx.txnSocial());
    auto tasksList = sgw.loadTasksByTaskIds(eventIds);

    std::map<social::TId, TaskStatus> idToStatus;
    for (const auto& task: tasksList) {
        idToStatus.emplace(task.id(), task);
    }
    return idToStatus;
}

std::map<TCommitId, TaskStatus>
loadTaskStatusesByCommitIds(const BranchContext& branchCtx, const TCommitIds& commitIds)
{
    social::Gateway sgw(branchCtx.txnSocial());
    auto tasksList = sgw.loadEditTasksByCommitIds(commitIds);

    std::map<TCommitId, TaskStatus> commitIdToStatus;
    for (const auto& task: tasksList) {
        // commit id is unique among edit tasks
        commitIdToStatus.emplace(task.commitId(), task);
    }
    return commitIdToStatus;
}

} // namespace

std::ostream&
operator << (std::ostream& os, const ApproveStatus& status)
{
    os << APPROVE_STATUS_TO_STR.at(status);
    return os;
}

std::istream&
operator >> (std::istream& is, ApproveStatus& status)
{
    std::string s;
    is >> s;
    for (const auto& p: APPROVE_STATUS_TO_STR) {
        if (p.second == s) {
            status = p.first;
            return is;
        }
    }
    throw Exception() << "Unrecognized approve status: " << status;
}

std::map<TCommitId, ApproveStatus>
evalApproveStatuses(const BranchContext& branchCtx, const std::list<revision::Commit>& commits)
{
    if (commits.empty()) {
        return {};
    }

    if (branchCtx.branch.id() != revision::TRUNK_BRANCH_ID) {
        return {};
    }

    TCommitIds commitIds;
    for (const auto& commit : commits) {
        commitIds.insert(commit.id());
    }

    revision_meta::CommitRegions commitRegions(branchCtx.txnCore());
    auto notPublishedCommitIds = commitRegions.loadNotPublished(commitIds);
    auto commitIdToTaskStatus = loadTaskStatusesByCommitIds(branchCtx, commitIds);

    std::map<TCommitId, ApproveStatus> result;
    for (const auto& commit: commits) {
        auto commitId = commit.id();

        boost::optional<TaskStatus> taskStatus;
        auto taskIt = commitIdToTaskStatus.find(commitId);
        if (taskIt != commitIdToTaskStatus.end()) {
            taskStatus = taskIt->second;
        }

        auto published =
            notPublishedCommitIds.count(commitId) == 0 &&
            commit.stableBranchId() > 0;

        result.emplace(commitId, calculateApproveStatus(published, taskStatus));
    }
    return result;
}

ApproveStatus
evalApproveStatus(const BranchContext& branchCtx, const revision::Commit& commit)
{
    auto statusesMap = evalApproveStatuses(branchCtx, {commit});
    auto it = statusesMap.find(commit.id());
    ASSERT(it != statusesMap.end());
    return it->second;
}

std::map<social::TId, ApproveStatus>
evalEventsApproveStatuses(
    const BranchContext& branchCtx,
    const social::Events& events,
    const std::list<revision::Commit>& commits)
{
    assertEventsMatchCommits(events, commits);
    if (events.empty()) {
        return {};
    }

    std::map<social::TId, ApproveStatus> result;
    if (branchCtx.branch.id() != revision::TRUNK_BRANCH_ID) {
        for (const auto& event: events) {
            result.emplace(event.id(), ApproveStatus::Approved);
        }
        return result;
    }

    TCommitIds commitIds;
    TCommitIds commitIdsInStableBranch;
    for (const auto& commit : commits) {
        auto commitId = commit.id();
        commitIds.insert(commitId);
        if (commit.stableBranchId() > 0) {
            commitIdsInStableBranch.insert(commitId);
        }
    }

    revision_meta::CommitRegions commitRegions(branchCtx.txnCore());
    auto notPublishedCommitIds = commitRegions.loadNotPublished(commitIds);

    social::TIds eventIds;
    for (const auto& event: events) {
        eventIds.insert(event.id());
    }
    auto eventIdToTaskStatuses = loadTaskStatusesByEventIds(branchCtx, eventIds);

    for (const auto& event: events) {
        boost::optional<TaskStatus> taskStatus;
        auto eventId = event.id();
        auto taskIt = eventIdToTaskStatuses.find(eventId);
        if (taskIt != eventIdToTaskStatuses.end()) {
            taskStatus = taskIt->second;
        }

        ASSERT(event.commitData());
        auto commitId = event.commitData()->commitId();
        auto published =
            notPublishedCommitIds.count(commitId) == 0 &&
            commitIdsInStableBranch.count(commitId) > 0;

        result.emplace(eventId, calculateApproveStatus(published, taskStatus));
    }
    return result;
}

} // namespace approve_status
} // namespace wiki
} // namespace maps
