#include "approver.h"

#include "updater.h"
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/moderation.h>
#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>

#include <maps/wikimap/mapspro/libs/revision_meta/include/utils.h>
#include <yandex/maps/wiki/common/robot.h>
#include <yandex/maps/wiki/revision/commit_manager.h>
#include <yandex/maps/wiki/social/gateway.h>

namespace maps {
namespace wiki {
namespace approved_commits {

namespace {

void checkResolution(social::CloseResolution resolution)
{
    REQUIRE(resolution != social::CloseResolution::Revert,
            "can not close task with " << resolution << " resolution");
}

social::ResolveResolution
resolveResolutionBy(social::CloseResolution closeResolution)
{
    return closeResolution == social::CloseResolution::Approve
        ? social::ResolveResolution::Accept
        : social::ResolveResolution::Edit;
}

} // namespace

CommitsApprover::CommitsApprover(
        const BranchContext& branchCtx,
        const acl::User& user,
        TCommitIds commitIds)
    : branchCtx_(branchCtx)
    , uid_(user.uid())
    , commitIds_(std::move(commitIds))
{
    user.checkActiveStatus();
    revision_meta::enqueueCommitsForApprovedBranch(branchCtx_.txnCore(), commitIds_);
}

social::TaskIds
CommitsApprover::closeTasks(social::CloseResolution closeResolution) const
{
    checkResolution(closeResolution);
    const auto resolveResolution = resolveResolutionBy(closeResolution);

    auto socialGw = social::Gateway(branchCtx_.txnSocial());
    socialGw
        .superModerationConsole(common::ROBOT_UID)
        .resolveEditTasksByCommitIds(resolveResolution, commitIds_);

    return socialGw
        .superModerationConsole(uid_)
        .closeEditTasksByCommitIds(commitIds_, resolveResolution, closeResolution);
}

social::TaskIds
CommitsApprover::closeTasksWithContributing(social::CloseResolution closeResolution) const
{
    return closeTasksWithContributing(uid_, closeResolution);
}

social::TaskIds
CommitsApprover::closeTasksWithContributing(TUid closeBy, social::CloseResolution closeResolution) const
{
    if (commitIds_.empty()) {
        return {};
    }

    checkResolution(closeResolution);
    const auto resolveResolution = resolveResolutionBy(closeResolution);

    auto socialGw = social::Gateway(branchCtx_.txnSocial());
    auto userConsole  = socialGw.superModerationConsole(closeBy);
    auto robotConsole = socialGw.superModerationConsole(common::ROBOT_UID);

    auto result =
        robotConsole.resolveEditTasksByCommitIds(
            resolveResolution, commitIds_
        );
    result.merge(
        userConsole.closeEditTasksByCommitIds(
            commitIds_, resolveResolution, closeResolution
        )
    );

    const auto contributingCommitIds = revision::CommitManager(branchCtx_.txnCore())
        .findDraftContributingCommits(commitIds_, {});
    result.merge(
        robotConsole.closeEditTasksByCommitIds(
            contributingCommitIds, resolveResolution, closeResolution
        )
    );

    return result;
}

void
CommitsApprover::awakeUpdater() const
{
    if (!commitIds_.empty()) {
        auto updater = cfg()->approvedCommitsUpdater();
        if (updater) {
            updater->awake();
        }
    }
}

} // namespace approved_commits
} // namespace wiki
} // namespace maps
