#include "processor.h"

#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/sync/sync_objects.h>
#include <maps/wikimap/mapspro/libs/revision_meta/include/branch_lock_ids.h>

#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/revision/commit_manager.h>

namespace maps {
namespace wiki {
namespace approved_commits {

namespace {

const std::string NAME = "Processing approved commits queue";

bool
checkBranchStateIsValid(const revision::Branch& branch)
{
    auto state = branch.state();
    if (state == revision::BranchState::Normal) {
        return true;
    }
    INFO() << NAME << " skipped, branch not normal, state: " << state;
    return false;
}

TCommitIds
approveCommits(const TCommitIds& commitIds)
{
    auto work = sync::masterCoreTransaction(AccessMode::ReadWrite);
    auto approvedCommitIds =
        revision::CommitManager(*work).approve(commitIds);
    work->commit();
    approvedCommitIds.insert(commitIds.begin(), commitIds.end());
    return approvedCommitIds;
}

revision::Branch::LockId
lockIdByMode(revision_meta::ApprovedQueueMode mode)
{
    switch (mode) {
    case revision_meta::ApprovedQueueMode::PreApprove:
        throw RuntimeError() << "PreApprove queue should not be used here";
    case revision_meta::ApprovedQueueMode::ViewAttrs:
        return revision_meta::TO_APPROVE_LOCK_ID;
    case revision_meta::ApprovedQueueMode::Labels:
        return revision_meta::TO_SYNC_LABELS_LOCK_ID;
    case revision_meta::ApprovedQueueMode::Bboxes:
        return revision_meta::TO_SYNC_BBOXES_LOCK_ID;
    }
    throw RuntimeError() << "Unreachable code";
}

void
finalizeCommits(revision_meta::ApprovedQueue& queue, const TCommitIds& commitIds)
{
    queue.deleteCommits(commitIds);
    if (queue.mode() != revision_meta::ApprovedQueueMode::ViewAttrs) {
        return;
    }

    auto& txn = queue.txn();
    revision_meta::ApprovedQueue(txn, revision_meta::ApprovedQueueMode::Labels)
        .push(commitIds);
    revision_meta::ApprovedQueue(txn, revision_meta::ApprovedQueueMode::Bboxes)
        .push(commitIds);
}

} // namespace

Processor::Processor(
        sync::BranchLocker& branchLocker,
        const revision::Branch& approvedBranch,
        size_t threads,
        size_t commitRangeBatchSize)
    : branchLocker_(branchLocker)
    , approvedBranch_(approvedBranch)
    , threads_(threads)
    , commitRangeBatchSize_(commitRangeBatchSize)
{
}

bool Processor::tryProcess(
    revision_meta::ApprovedQueueMode mode,
    const ExecutionStatePtr& executionState) const
{
    INFO() << NAME << " try process " << mode;

    if (!executionState->isOk()) {
        INFO() << NAME << " execution state is failed";
        return false;
    }

    if (!checkBranchStateIsValid(approvedBranch_)) {
        return false; // branch not normal
    }

    if (hasStabeBranchCreationAttribute(approvedBranch_)) {
        INFO() << NAME << " skipped, stable branch creation is on";
        return false;
    }

    auto& work = branchLocker_.work();
    if (!approvedBranch_.tryLock(
            work, lockIdByMode(mode),
            revision::Branch::LockType::Exclusive)) {
        INFO() << NAME << " skipped, branch already locked";
        return false;
    }

    revision_meta::ApprovedQueue queue(work, mode);
    auto commitIds = queue.readAllCommitIds();
    if (commitIds.empty()) {
        INFO() << NAME << " skipped, empty queue";
        return false;
    }

    auto params = prepareSyncParams(mode, executionState);
    params.setCommitRangeBatchSize(commitRangeBatchSize_);
    sync::SyncObjects syncController(std::move(params), branchLocker_);

    // check after lock branch
    if (!checkBranchStateIsValid(syncController.branch())) {
        return false;
    }

    if (hasStabeBranchCreationAttribute(approvedBranch_)) {
        INFO() << NAME << " skipped, stable branch creation is on";
        return false;
    }

    if (mode == revision_meta::ApprovedQueueMode::ViewAttrs) {
        commitIds = approveCommits(commitIds);
    }

    syncController.run(commitIds);

    if (executionState->isOk()) {
        finalizeCommits(queue, commitIds);
        return true;
    }
    INFO() << NAME << " state, "
        << " CANCELED: " << executionState->cancel
        << " FAILED: " << executionState->fail;
    return false;
}

void Processor::waitAndProcess(revision_meta::ApprovedQueueMode mode) const
{
    ASSERT(approvedBranch_.state() == revision::BranchState::Progress);

    INFO() << NAME << " waiting lock";

    auto& work = branchLocker_.work();
    approvedBranch_.lock(
        work, lockIdByMode(mode),
        revision::Branch::LockMode::Wait,
        revision::Branch::LockType::Exclusive);

    auto commitIds = revision_meta::ApprovedQueue(work, mode).readAllCommitIds();
    if (commitIds.empty()) {
        INFO() << NAME << "skipped, empty queue";
        return;
    }

    INFO() << NAME << " start approving and sync commit ids, size: " << commitIds.size();

    auto executionState = std::make_shared<ExecutionState>();

    sync::SyncObjects syncController(prepareSyncParams(mode, executionState), branchLocker_);

    if (mode == revision_meta::ApprovedQueueMode::ViewAttrs) {
        commitIds = approveCommits(commitIds);
    }

    syncController.run(commitIds);

    REQUIRE(executionState->isOk(),
        NAME << " state, "
        << " CANCELED: " << executionState->cancel
        << " FAILED: " << executionState->fail);

    INFO() << NAME << " approved and sync commit ids, size: " << commitIds.size();
    auto workCore = sync::masterCoreTransaction(AccessMode::ReadWrite);
    revision_meta::ApprovedQueue queue(*workCore, mode);
    finalizeCommits(queue, commitIds);
    workCore->commit();
    INFO() << NAME << " done";
}

sync::SyncParams Processor::prepareSyncParams(
    revision_meta::ApprovedQueueMode mode,
    const ExecutionStatePtr& executionState) const
{
    auto syncFlags = [&] {
        switch (mode) {
        case revision_meta::ApprovedQueueMode::PreApprove:
            throw RuntimeError() << "PreApprove queue should not be used here";
        case revision_meta::ApprovedQueueMode::ViewAttrs:
            return sync::SyncFlags(StringSet{sync::STAGES_VIEW, sync::STAGES_ATTRS});
        case revision_meta::ApprovedQueueMode::Labels:
            return sync::SyncFlags(StringSet{sync::STAGES_LABELS});
        case revision_meta::ApprovedQueueMode::Bboxes:
            return sync::SyncFlags(StringSet{sync::STAGES_BBOX});
        }
    };

    return sync::SyncParams(
        executionState,
        syncFlags(),
        approvedBranch_.id(),
        threads_);
}

} // namespace approved_commits
} // namespace wiki
} // namespace maps
