#include "attributes.h"
#include "shadow_attributes.h"
#include "exception.h"
#include "utils.h"
#include "configs/config.h"
#include "configs/categories.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/libs/dbutils/include/parser.h>
#include <yandex/maps/wiki/common/pg_utils.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/configs/editor/categories.h>

#include <algorithm>

namespace maps
{
namespace wiki
{

const std::string Attribute::novalue ="";
const Attribute::Values Attribute::novalues = Attribute::Values();

//---------------------------------Attributes---------------------------------
Attribute::Attribute( const AttributeDef& attrDef )
    :attrDef_(attrDef)
{
    if (!attrDef.anyValue()) {
        values_.push_back(attrDef.defaultValue());
    }
}

namespace
{

template <typename Container> Container
unpackAll(const Container& packedStrings)
{
    Container unpackedAll;
    for (const auto& packed : packedStrings) {
        const auto unpacked = AttributeDef::unpackMultiValue(packed);
        unpackedAll.insert(unpackedAll.end(), unpacked.begin(), unpacked.end());
    }
    return unpackedAll;
}

} // namespace

Attribute::Attribute(const AttributeDef& attrDef, const StringVec& values)
    :attrDef_(attrDef)
{
    const StringVec& unpackedValues = attrDef_.multiValue()
        ? unpackAll(values)
        : values;
    if ( attrDef_.multiValue() ) {
        for (const auto& value : unpackedValues) {
            values_.push_back(value);
        }
    } else if (!unpackedValues.empty()) {
        WIKI_REQUIRE(unpackedValues.size() < 2, ERR_ATTRIBUTE_VALUE_INVALID,
                "Attribute " << attrDef_.id() << " doesn't support multiple values");
        values_.push_back(unpackedValues[0]);
    }
}


Attribute& Attribute::operator = (const Attribute& rhs)
{
    values_ = rhs.values_;
    return *this;
}

void
Attribute::setValue(const std::string& newValue)
{
    values_.clear();
    addValue(newValue);
}

void
Attribute::addValue(const std::string& newValue)
{
    values_.push_back(newValue);
}

void
Attribute::clear()
{
    values_.clear();
}


bool
Attribute::operator == (const Attribute& a) const
{
    if(values_.empty() || a.values_.empty()){
        return static_cast<size_t>(std::count(values_.begin(), values_.end(), "")) == values_.size()
            && static_cast<size_t>(std::count(a.values_.begin(), a.values_.end(), "")) == a.values_.size();
    }
    return values_ == a.values_;
}

const std::string&
Attribute::value()const
{
    return values_.empty() ? novalue : *values_.begin();
}

std::string
Attribute::packedValues() const
{
    return common::join(values_, AttributeDef::MULTIVALUE_DELIMITER);
}

const std::string&
Attribute::name()const
{
    return attrDef_.id();
}

//---------------------------------Attributes---------------------------------

Attributes&
Attributes::operator=(const Attributes& rhs)
{
    attr_ = rhs.attr_;
    defs_ = rhs.defs_;
    return *this;
}

void
Attributes::load(const StringMap& contents)
{
    for ( const auto& def: defs_){
        if (def->table())
            continue;
        if (contents.count(def->id())) {
            addAttribute(Attribute(*def, {contents.at(def->id())}));
        } else {
            addAttribute(Attribute(*def, {def->autoValue()}));
        }
    }
}



void
Attributes::load(const StringMultiMap& attrValues)
{
    for ( const auto& def: defs_){
        if (def->table())
            continue;
        std::pair<StringMultiMap::const_iterator, StringMultiMap::const_iterator> valuesRange =
            attrValues.equal_range(def->id());
        bool hasIncomingValue = (valuesRange.first != valuesRange.second);
        if (hasIncomingValue && isShadowAttributeId(def->id())) {
            const auto useShadowId = useShadowAttributeId(def->id());
            const auto newUseValueIt = attrValues.find(useShadowId);
            if ((newUseValueIt == attrValues.end() || newUseValueIt->second.empty()) && value(useShadowId).empty()) {
                hasIncomingValue = false;
            }
        }

        if (!hasIncomingValue) {
            if (!isSet(def->id())) {
                addAttribute(Attribute(*def));
            }
        } else {
            StringVec values;
            std::transform(valuesRange.first
                , valuesRange.second
                , std::back_inserter(values)
                , std::bind(&StringMultiMap::value_type::second, std::placeholders::_1)
                );
            addAttribute(Attribute(*def, values));
        }
    }
}

void
Attributes::addAttribute(const Attribute& attribute)
{
    auto res = attr_.findByKey(attribute.id());
    if (res == attr_.end()) {
        attr_.push_back(attribute);
    } else {
        *res = attribute;
    }
}

bool
Attributes::areAllSystem() const
{
    for (const auto& def : defs_ ){
        if(!def->system()){
            return false;
        }
    }
    return true;
}

bool
Attributes::empty() const
{
    return defs_.empty() || attr_.empty();
}

std::string
Attributes::asHstore(Transaction& work) const
{
    StringMap definedAttributes;
    for(const auto& def : defs_) {
        if (def->table()) {
            continue;
        }
        auto it = attr_.findByKey(def->id());
        if (it != attr_.end()) {
            definedAttributes.emplace(it->def().id(), it->packedValues());
        }
    }
    return common::attributesToHstore(work, definedAttributes);
}


bool
Attributes::isDefined(const std::string& name) const
{
    return defs_.findByKey(name) != defs_.end();
}

bool
Attributes::isSet(const std::string& name) const
{
    return attr_.findByKey(name) != attr_.end();
}

const std::string&
Attributes::value(const std::string& name)const
{
    auto it = attr_.findByKey(name);
    if(it != attr_.end()){
        return it->value();
    }
    return Attribute::novalue;
}

const Attribute::Values&
Attributes::values(const std::string& name) const
{
    auto it = attr_.findByKey(name);
    if(it != attr_.end()){
        return it->values();
    }
    return Attribute::novalues;
}

void
Attributes::setValue(const std::string& attrName, const std::string& attrValue)
{
    auto it = attr_.findByKey(attrName);
    if(it != attr_.end()){
        it->setValue(attrValue);
    } else {
        auto defIt = defs_.findByKey(attrName);
        if(defIt == defs_.end()){
            THROW_WIKI_INTERNAL_ERROR( "Attribute " << attrName << " not defined");
        }
        addAttribute(Attribute(**defIt, {attrValue}));
    }
}

void
Attributes::addValue(const std::string& attrName, const std::string& attrValue)
{
    auto it = attr_.findByKey(attrName);
    if(it != attr_.end()){
        it->addValue(attrValue);
    } else {
        auto defIt = defs_.findByKey(attrName);
        if(defIt == defs_.end()){
            THROW_WIKI_INTERNAL_ERROR( "Attribute " << attrName << " not defined");
        }
        addAttribute(Attribute(**defIt, {attrValue}));
    }
}

void
Attributes::clear(const std::string& attrName)
{
    auto it = attr_.findByKey(attrName);
    if(it != attr_.end()){
        it->clear();
    }
}

size_t
Attributes::size() const
{
    return attr_.size();
}

bool
Attributes::operator ==(const Attributes& a) const
{
    return areEqual(a, false);
}

bool
Attributes::areSystemEqual(const Attributes& a) const
{
    return areEqual(a, true);
}

bool
Attributes::areEqual(const Attributes& a, bool system) const
{
    std::map<std::string, std::string> actualShadowAttributeValue;
    for (auto it = attr_.begin(); it != attr_.end(); ++it) {
        if (!isShadowAttributeId(it->id())) {
            continue;
        }
        auto useIt = attr_.findByKey(useShadowAttributeId(it->id()));
        if (useIt == attr_.end() || useIt->value().empty()) {
            actualShadowAttributeValue.emplace(it->id(), s_emptyString);
        } else {
            actualShadowAttributeValue.emplace(it->id(), it->def().autoValue());
        }
    }
    bool ret = true;
    for (auto it = attr_.begin(); ret && it != attr_.end(); ++it) {
        if (it->def().table())
            continue;
        const auto isShadow = actualShadowAttributeValue.count(it->id());
        const auto thisValue = isShadow
            ? actualShadowAttributeValue.at(it->id())
            : it->value();
        auto other = a.attr_.findByKey(it->id());
        if (other == a.attr_.end()) {
            ret = thisValue.empty();
        } else {
            if (system) {
                if (it->def().system() && other->def().system()) {
                    ret = (*other == *it);
                }
            } else {
                ret =
                    (it->def().system() && other->def().system() ) ||
                    (*other == *it) ||
                    (isShadow && other->value() == thisValue);
            }
        }
    }
    return ret;
}

namespace {

template <class Keys>
std::string
categoryByExistingAttributesNames(const Keys& keys)
{
    const auto& categories = cfg()->editor()->categories();
    for (const std::string& key : keys) {
        std::string catId = canonicalCategoryIdToPlain(key);
        if (catId.empty()) {
            continue;
        }
        if (categories.defined(catId)) {
            return catId;
        }
    }
    return s_emptyString;
}

} // namespace

std::string
categoryFromAttributes(const std::string& attributesIds)
{
    return categoryByExistingAttributesNames(dbutils::parsePGArray(attributesIds));
}

std::string
categoryFromAttributes(const StringMap& attributes)
{
    return categoryByExistingAttributesNames(readKeys(attributes));
}

std::string
categoryFromAttributes(const Attributes& attributes)
{
    std::set<std::string> attrIds;
    for (const auto& attr : attributes) {
        attrIds.insert(attr.id());
    }
    return categoryByExistingAttributesNames(attrIds);
}

}//namespace wiki
}//namespace maps

