#pragma once

#include "common.h"
#include "configs/categories.h"

#include <yandex/maps/wiki/configs/editor/unique_list.h>
#include <yandex/maps/wiki/configs/editor/attrdef.h>
#include <yandex/maps/wiki/common/natural_sort.h>

#include <string>
#include <set>

namespace maps::wiki {

/**
 * Represents object`s attribute and its possible values
 */
class Attribute
{
public:
    friend class Attributes;
    typedef UniqueList<std::string> Values;
    //!empty string for any value
    static const std::string novalue;
    static const Values novalues;

    Attribute(const AttributeDef& attrDef);
    Attribute(const AttributeDef& attrDef, const StringVec& values);
    Attribute(const Attribute&) = default;
    Attribute& operator = (const Attribute&);

    const std::string& name() const;
    const std::string& id() const { return name();}
    const std::string& value() const;
    const Values& values() const
    { return values_; }
    //!pack values in string like value|value
    std::string packedValues() const;

    bool operator == (const Attribute& a) const;

    const AttributeDef& def() const
    { return attrDef_;};

private:
    void setValue(const std::string& newValue);
    void addValue(const std::string& newValue);
    void clear();

    const AttributeDef& attrDef_; //!attribute definition
    Values values_; //!current values of attribute
};

class Attributes
{
private:
    template <typename IdIndexedClass> class IdKey
    {
    public:
        IdKey(const std::shared_ptr<IdIndexedClass>& obj):id_(obj->id()){};
        IdKey(const IdIndexedClass& obj):id_(obj.id()){};
        IdKey(const std::string& attrId):id_(attrId){};
        bool operator <(const IdKey& other) const {return id_ < other.id_;}
    private:
        std::string id_;
    };
public:
    typedef UniqueList<std::shared_ptr<AttributeDef>, IdKey<AttributeDef>> DefsList;
    typedef UniqueList<Attribute, IdKey<Attribute>> AttributesList;

    /**
    * Chage defintions keeping values
    */
    template <typename AttrDefIter>
    void redefine(AttrDefIter first, AttrDefIter last) {
        defs_.clear();
        storeDefs(first, last);
    }

    Attributes() = default;
    Attributes(const Attributes&) = default;

    Attributes& operator=(const Attributes&);

    /**
    * Load values from multimap (attrId, [values])
    * @param attrValues -source multimap
    */
    void load(const StringMultiMap& attrValues);

    /**
    * Load values from packed contents attrId=value|value|...
    * @param contents -source
    */
    void load(const StringMap& contents);
    bool isDefined(const std::string& name) const;
    bool isSet(const std::string& name) const;
    const std::string& value(const std::string& name) const;

    const Attribute::Values& values(const std::string& name) const;

    void setValue(const std::string& attrName, const std::string& attrValue);
    void addValue(const std::string& attrName, const std::string& attrValue);
    void clear(const std::string& attrName);

    size_t size() const;

    bool operator == (const Attributes& a) const;

    bool areSystemEqual(const Attributes& a) const;

    bool areAllSystem() const;

    bool empty() const;


    /**
    * Format attributes as hstore constructor
    * @param work - used for values escaping
    */
    std::string asHstore(Transaction& work) const;

    const DefsList& definitions() const
    { return defs_;};

    AttributesList::const_iterator begin() const { return attr_.begin();}
    AttributesList::const_iterator end() const { return attr_.end();}
    AttributesList::const_iterator find(const std::string& id) const
    { return attr_.findByKey(id);}

    void clearValues() { attr_.clear(); }

private:
    template <typename AttrDefIter>
    void storeDefs(AttrDefIter first, AttrDefIter last)
    {
        for ( ;first!=last; ++first){
            defs_.push_back(*first);
        }
    }
    bool areEqual(const Attributes& a, bool system) const;
    void addAttribute(const Attribute& attribute);

    AttributesList attr_;
    DefsList defs_;
};

std::string categoryFromAttributes(const std::string& attributesIds);
std::string categoryFromAttributes(const StringMap& attributes);
std::string categoryFromAttributes(const Attributes& attributes);

} // namespace maps::wiki
