#include "db_monitor.h"
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/pool_sections.h>

#include <yandex/maps/wiki/common/string_utils.h>

#include <maps/libs/common/include/profiletimer.h>
#include <maps/libs/pgpool/include/monitoring.h>

#include <maps/infra/yacare/include/yacare.h>

using namespace std::chrono_literals;

namespace maps::wiki {

namespace {

const std::string WORKER_NAME = "HangingQueriesMonitor";
const std::string EMPTY;

constexpr auto REFRESH_TIME = 300s;

constexpr std::chrono::seconds WARN_DURATION = 30min;
constexpr std::chrono::seconds CRIT_DURATION = 6h;

const StringVec POOL_SECTIONS {
    POOL_SECTION_NAME_CORE,
    POOL_SECTION_NAME_SOCIAL,
    POOL_SECTION_NAME_VIEW_TRUNK,
    POOL_SECTION_NAME_VIEW_STABLE,
    POOL_SECTION_NAME_LABELS_TRUNK,
    POOL_SECTION_NAME_LABELS_STABLE,
    POOL_SECTION_NAME_VALIDATION,
};

} // namespace

bool HangingQueriesMonitoringState::isWarning(const std::string& poolName) const
{
    std::lock_guard<std::mutex> lock(mutex_);

    auto it = poolInfos_.find(poolName);
    if (it != poolInfos_.end()) {
        return it->second.state != MonitoringState::Ok;
    }
    return false;
}

bool HangingQueriesMonitoringState::isCritical(const std::string& poolName) const
{
    std::lock_guard<std::mutex> lock(mutex_);

    auto it = poolInfos_.find(poolName);
    if (it != poolInfos_.end()) {
        return it->second.state == MonitoringState::Crit;
    }
    return false;
}

const std::string& HangingQueriesMonitoringState::message(const std::string& poolName) const
{
    std::lock_guard<std::mutex> lock(mutex_);

    auto it = poolInfos_.find(poolName);
    if (it != poolInfos_.end()) {
        return it->second.message;
    }
    return EMPTY;
}

void HangingQueriesMonitoringState::setPoolState(
    const std::string& poolName,
    MonitoringState state,
    std::string message)
{
    std::lock_guard<std::mutex> lock(mutex_);
    poolInfos_[poolName] = PoolInfo{ state, std::move(message) };
}

HangingQueriesMonitoringState& getMonitoringState()
{
    static HangingQueriesMonitoringState state;
    return state;
}

//============== HangingQueriesMonitor ==============

HangingQueriesMonitor::HangingQueriesMonitor()
    : ThreadObserver<HangingQueriesMonitor>(REFRESH_TIME)
{
    start();
}

HangingQueriesMonitor::~HangingQueriesMonitor() = default;

const std::string& HangingQueriesMonitor::name() const
{
    return WORKER_NAME;
}

void HangingQueriesMonitor::onStart()
{
    INFO() << WORKER_NAME << " started";
}

void HangingQueriesMonitor::onStop()
{
    INFO() << WORKER_NAME << " stopped";
}

void HangingQueriesMonitor::doWork()
try {
    ProfileTimer pt;

    for (auto poolName : POOL_SECTIONS) {
        checkPool(poolName, cfg()->pool(poolName));
    }

    INFO() << WORKER_NAME << " elapsed time " << pt.getElapsedTime();
} catch (const maps::Exception& ex) {
    ERROR() << WORKER_NAME << " failed: " << ex;
} catch (const std::exception& ex) {
    ERROR() << WORKER_NAME << " failed: " << ex.what();
}

void HangingQueriesMonitor::checkPool(const std::string& poolName, pgpool3::Pool& pool)
{
    auto result = pgpool3::queryPgStatActivity(pool);

    auto state = MonitoringState::Ok;

    std::vector<std::pair<std::string, std::chrono::seconds>> hostDurations;

    for (const auto& [host, activity] : result.hostToActivity) {
        auto longestDuration = std::max(
            activity.longestRunningQuery,
            activity.longestRunningTransaction);

        if (longestDuration > CRIT_DURATION) {
            state = std::max(state, MonitoringState::Crit);
            hostDurations.emplace_back(host, longestDuration);
        } else if (longestDuration > WARN_DURATION) {
            state = std::max(state, MonitoringState::Warn);
            hostDurations.emplace_back(host, longestDuration);
        }
    }

    //Longest durations first
    std::sort(hostDurations.begin(), hostDurations.end(),
        [](const auto& lhs, const auto& rhs) {
            return lhs.second > rhs.second;
        });

    auto message = common::join(
        hostDurations,
        [](const auto& pair) {
            return "There is a transaction longer than " + std::to_string(pair.second.count())
                + " seconds on host " + pair.first;
        },
        "; ");

    INFO() << WORKER_NAME << " pool " << poolName << " result "
        << (state == MonitoringState::Ok ? "OK" : message);

    getMonitoringState().setPoolState(poolName, state, message);
}

std::unique_ptr<HangingQueriesMonitor> createDbMonitor()
{
    return std::make_unique<HangingQueriesMonitor>();
}

} // namespace maps::wiki
