#include "check_permissions.h"

#include "configs/config.h"
#include "exception.h"
#include "object_update_data.h"
#include "objects/attr_object.h"
#include "objects/category_traits.h"
#include "objects/helpers.h"
#include "objects/linear_element.h"
#include "objects/object.h"
#include "objects/relation_object.h"
#include "objects/relation_object.h"
#include "relation_infos.h"

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/tile/include/tile.h>
#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <maps/wikimap/mapspro/libs/acl/include/exception.h>
#include <yandex/maps/wiki/configs/editor/categories.h>
#include <yandex/maps/wiki/diffalert/message.h>
#include <yandex/maps/wiki/revision/commit.h>
#include <yandex/maps/wiki/social/event_alert.h>

#include <boost/algorithm/string/trim.hpp>

#include <geos/io/WKBReader.h>

namespace maps {
namespace wiki {

namespace {

const std::string EMPTY_VALUE_ID = "empty_value_id";

const acl::SubjectPath PERMISSION_TO_MANAGE_BRANCHES("mpro/vcs/manage-branches");
const acl::SubjectPath PERMISSION_TO_EDIT_STABLE("mpro/vcs/edit-stable");
const acl::SubjectPath PERMISSION_TO_VIEW_BRANCHES("mpro/vcs/view-branches");

const acl::SubjectPath PERMISSION_TO_CLONE("mpro/tools/clone");
const acl::SubjectPath PERMISSION_TO_FILTERS("mpro/tools/attrs-filters");
const acl::SubjectPath PERMISSION_TOOLS_POIS_CONFLICTS("mpro/tools/pois-conflicts");
const acl::SubjectPath PERMISSION_TOOLS_BUSINESS_PHOTOS("mpro/tools/business-photos");
const acl::SubjectPath PERMISSION_TOOLS_EDITS_FEED("mpro/tools/edits-feed");
const acl::SubjectPath PERMISSION_SETTINGS_AUTO_IS_LOCAL("mpro/settings/auto-is-local");
const acl::SubjectPath PERMISSION_SETTINGS_EDITOR_MODE("mpro/settings/editor-mode");
const acl::SubjectPath PERMISSION_SETTINGS_STICK_POLYGONS("mpro/settings/stick-polygons");

const acl::SubjectPath PERMISSION_TO_GROUP_MOVE("mpro/tools/group-operations/move");
const acl::SubjectPath PERMISSION_TO_GROUP_SYNC_GEOMETRY("mpro/tools/group-operations/sync-geometry");
const acl::SubjectPath PERMISSION_TO_GROUP_UPDATE_ATTRIBUTES("mpro/tools/group-operations/update-attributes");
const acl::SubjectPath PERMISSION_TO_GROUP_UPDATE_RELATIONS("mpro/tools/group-operations/update-relations");
const acl::SubjectPath PERMISSION_TO_GROUP_UPDATE_RELATION("mpro/tools/group-operations/update-relation");
const acl::SubjectPath PERMISSION_TO_GROUP_DELETE("mpro/tools/group-operations/delete");
const acl::SubjectPath PERMISSION_TO_GROUP_DELETE_COMMENTS("mpro/tools/group-operations/delete-comments");

const acl::SubjectPath PERMISSION_FOR_COMPLETED_BY_OTHERS_TASK_FEED("mpro/social/completed-by-others-moderation-task-feed");

const acl::SubjectPath PERMISSION_TO_IGNORE_RESTRICTIONS_ACL_BASE("mpro/tools/ignore-restrictions");

const acl::SubjectPath PERMISSION_TO_BLOCKING_TASKS("mpro/social/blocking-moderation-tasks");

const acl::SubjectPath PERMISSION_TO_ANNOTATE_COMMITS("mpro/social/comments/annotate-other-user-commit");
const acl::SubjectPath PERMISSION_TO_DELETE_COMMENTS("mpro/social/comments/delete-other-user-comment");

const acl::SubjectPath PERMISSION_TASKS_VALIDATOR("mpro/tasks/validator");

const std::vector<std::string> PRECISION_ATTRIBUTES = {
    ATTR_POI_POSITION_QUALITY,
    ATTR_INDOOR_POI_POSITION_QUALITY
};

const std::unordered_map<std::string, std::vector<std::string>> REQUIRED_ATTRIBUTES_VIEW_ACCESS = {
    {diffalert::message::POI_WITH_PRECISE_LOCATION_DISPLACED, PRECISION_ATTRIBUTES},
    {diffalert::message::POI_WITH_PRECISE_LOCATION_OFFICIAL_NAME_CHANGED, PRECISION_ATTRIBUTES},
    {diffalert::message::POI_WITH_PRECISE_LOCATION_BUSINESS_ID_CHANGED, PRECISION_ATTRIBUTES},
    {diffalert::message::POI_WITH_PRECISE_LOCATION_RUBRIC_ID_CHANGED, PRECISION_ATTRIBUTES}
};

const GeoObject*
findBlockingObject(const GeoObject* obj);

bool isNonBlockingRelation(const GeoObject* master, const std::string& roleId)
{
    return master->category().slaveRole(roleId).nonBlocking();
}

const GeoObject*
findBlockingMasterObject(const GeoObject* obj)
{

    for (const auto& master : obj->masterRelations().range()) {
        if (isNonBlockingRelation(master.relative(), master.roleId())) {
            continue;
        }
        auto masterBlock = findBlockingObject(master.relative());
        if (masterBlock) {
            return masterBlock;
        }
    }
    for (const auto& master : obj->masterRelations().diff().deleted) {
        if (isNonBlockingRelation(master.relative(), master.roleId())) {
            continue;
        }
        auto masterBlock = findBlockingObject(master.relative());
        if (masterBlock) {
            return masterBlock;
        }
    }
    return nullptr;
}

const GeoObject*
findBlockingObject(const GeoObject* obj)
{
    if (obj->isBlocked()) {
        return obj;
    }
    const auto& contourDefs = cfg()->editor()->contourObjectsDefs();
    const auto contourPartType = contourDefs.partType(obj->categoryId());
    if (contourPartType != ContourObjectsDefs::PartType::None) {
        switch (contourPartType) {
            case ContourObjectsDefs::PartType::LinearElement :
            case ContourObjectsDefs::PartType::Contour :
            case ContourObjectsDefs::PartType::Center : {
                auto blockingObject = findBlockingMasterObject(obj);
                if (blockingObject) {
                    return blockingObject;
                }
                break;
            }
            case ContourObjectsDefs::PartType::Object :
            default: {
                return nullptr;
            }
        }
    } else if (is<LinearElement>(obj)) {
        auto blockingObject = findBlockingMasterObject(obj);
        if (blockingObject) {
            return blockingObject;
        }
    }
    return nullptr;
}

void
checkAttributePermissionsPerValues(
    const acl::SubjectPath& attrAclPath,
    const acl::CheckContext& aclContext,
    const Attribute::Values& oldVals,
    const Attribute::Values& newVals)
{
    StringVec alteredVals;
    std::set_symmetric_difference(
        oldVals.begin(), oldVals.end(),
        newVals.begin(), newVals.end(),
        std::back_inserter(alteredVals));
    for (const std::string& value : alteredVals) {
        if (!value.empty()) {
            attrAclPath(value).check(aclContext);
        }
    }
}

bool
isModifiedContents(const GeoObject* obj)
{
     return obj->isModifiedGeom()
        || obj->isModifiedAttr()
        || obj->isModifiedTableAttrs()
        || obj->isModifiedRichContent()
        || obj->isModifiedState();
}

bool
cantHaveExternalRelations(const GeoObject* obj)
{
    const auto& contourDefs = cfg()->editor()->contourObjectsDefs();
    const auto contourPartType = contourDefs.partType(obj->categoryId());
    return contourPartType == ContourObjectsDefs::PartType::Contour
        || contourPartType == ContourObjectsDefs::PartType::LinearElement
        || contourPartType == ContourObjectsDefs::PartType::Center
        || is<LinearElement>(obj);
}

bool
isModifiedBlockedGeomParts(const GeoObject* obj)
{
    auto blockedGeomPartsFilter = [](const SlaveRole& role)
    {
        return role.geomPart() && !role.nonBlocking();
    };
    return !obj->slaveRelations().diff(
            obj->category().slaveRoleIds(blockedGeomPartsFilter)).empty();
}

bool
isModifiedExternalRelationsOnly(const GeoObject* obj)
{
    return !(isModifiedContents(obj)
            || cantHaveExternalRelations(obj)
            || isModifiedBlockedGeomParts(obj));
}

bool
isLinearElementAffectedByJunctionDelete(const GeoObject* obj)
{
    if (!is<LinearElement>(obj) || obj->primaryEdit()) {
        return false;
    }
    for (const auto& rel : obj->slaveRelations().diff().deleted) {
        auto relative = rel.relative();

        if (is<Junction>(relative) &&
            relative->isDeleted() &&
            relative->primaryEdit()) {
            return true;
        }
    }
    return false;
}

bool
allLinearRelatives(const MixedRolesInfosContainer& relativeInfos,
    std::function<bool(const LinearElement* el)> predicate)
{
    for (const auto& rel : relativeInfos) {
        if (!is<LinearElement>(rel.relative())) {
            return false;
        }
        if (!predicate(as<LinearElement>(rel.relative()))) {
            return false;
        }
    }
    return true;
}

bool
areRelationsTheEffectOfLinearSplit(const RelationInfosDiff& diffSlaves)
{
    return
        allLinearRelatives(diffSlaves.deleted,
            [](const LinearElement* el) {
                return !el->isDeleted() && 0 != el->affectedBySplitId();
            }) &&
        allLinearRelatives(diffSlaves.added,
            [](const LinearElement* el) {
                return 0 != el->affectedBySplitId();
            });
}

bool
areRelationsTheEffectOfLinearMerge(const RelationInfosDiff& diffSlaves)
{
    return
        allLinearRelatives(diffSlaves.added,
            [](const LinearElement* el) {
                return isLinearElementAffectedByJunctionDelete(el);
            }) &&
        allLinearRelatives(diffSlaves.deleted,
            [](const LinearElement* el) {
                return isLinearElementAffectedByJunctionDelete(el);
            });
}

bool permissionExists(Transaction& work, const acl::SubjectPath& permissionPath)
{
    acl::ACLGateway gw(work);
    try {
        gw.permission(permissionPath);
        return true;
    } catch (const acl::PermissionNotExists& ex) {
    } catch (const std::exception& ex) {
        WARN() << "permissionExists exception: " << ex.what();
    }
    return false;
}
} // namespace

CheckPermissions::CheckPermissions(
    TUid userId,
    Transaction& work,
    BannedPolicy bannedPolicy)
    : userId_(userId)
    , work_(work)
    , objectsUpdateData_(nullptr)
    , bannedPolicy_(bannedPolicy)
{
}

CheckPermissions::CheckPermissions(
    TUid userId,
    Transaction& work,
    const ObjectsDataMap& objectsUpdateData)
    : userId_(userId)
    , work_(work)
    , objectsUpdateData_(&objectsUpdateData)
    , bannedPolicy_(BannedPolicy::Deny)
{
}

const acl::CheckContext&
CheckPermissions::globalContext()
{
    if (!globalContext_) {
        globalContext_ = make_unique<acl::CheckContext>(
                userId_,
                std::vector<std::string>{},
                work_,
                bannedPolicy_ == BannedPolicy::Deny
                    ? std::set<acl::User::Status>{acl::User::Status::Active}
                    : std::set<acl::User::Status>{acl::User::Status::Active, acl::User::Status::Banned});
    }
    return *globalContext_;
}

void
CheckPermissions::checkPermissionsToDeleteObject(
    const acl::CheckContext& aclContext, const GeoObject* obj)
{
    if (!obj->isDeleted()) {
        return;
    }
    acl::SubjectPath aclPath(SERVICE_ACL_BASE);
    aclPath(obj->category().aclPermissionName())(STR_DELETE_PERMISSION).check(aclContext);
}

void
CheckPermissions::checkPermissionsToObjectGeometry(
    const acl::CheckContext& aclContext, const GeoObject* obj)
{
    if (!obj->isModifiedGeom() || obj->geom().isNull()) {
        return;
    }
    acl::SubjectPath aclPath(SERVICE_ACL_BASE);
    aclPath(obj->category().aclPermissionName())
        (STR_EDIT_PERMISSION)(STR_GEOMETRY_PERMISSION).check(aclContext);
}

bool
CheckPermissions::hasPermissionsToEditObjectGeometry(const GeoObject* obj)
{
    ASSERT(!obj->geom().isNull());
    auto aclContext = globalContext();
    aclContext = aclContext.narrow({obj->geom().wkb()}, work_);
    acl::SubjectPath aclPath(SERVICE_ACL_BASE);
    return aclPath(obj->category().aclPermissionName())
        (STR_EDIT_PERMISSION)(STR_GEOMETRY_PERMISSION).isAllowed(aclContext);
}

void
CheckPermissions::checkPermissionsToObjectRelations(
    const acl::CheckContext& aclContext,
    const GeoObject* obj,
    const RelationInfosDiff& diffSlaves)
{
    if (!needCheckRelationsPermissions(obj->categoryId())) {
        return;
    }
    const auto slaveRelationsDiff = obj->slaveRelations().diff(
        obj->category().slaveRoleIds(roles::filters::IsNotTable));
    if (!slaveRelationsDiff.empty())
    {
        bool isSplitOnly = areRelationsTheEffectOfLinearSplit(diffSlaves);
        bool isMergeOnly = !isSplitOnly &&
            areRelationsTheEffectOfLinearMerge(diffSlaves);
        if (!isSplitOnly && !isMergeOnly) {
            const auto relationsAclPath = SERVICE_ACL_BASE
                (obj->category().aclPermissionName())
                (STR_EDIT_PERMISSION)
                (STR_RELATIONS_PERMISSION);
            for (const auto& addedRel : slaveRelationsDiff.added) {
                relationsAclPath(addedRel.roleId()).check(aclContext);
            }
            for (const auto& deletedRel : slaveRelationsDiff.deleted) {
                relationsAclPath(deletedRel.roleId()).check(aclContext);
            }
        }
    }
    if (obj->primaryEdit() && !obj->masterRelations().diff().empty()) {
        const auto masterRelationsDiff = obj->masterRelations().diff();
        auto checkRelRange = [&] (const maps::wiki::MixedRolesInfosContainer& relInfos) {
            for (const auto& relInfo : relInfos) {
                const auto roleInMasterAclPath =
                    SERVICE_ACL_BASE(relInfo.relative()->category().aclPermissionName())
                        (STR_EDIT_PERMISSION)
                        (STR_RELATIONS_PERMISSION)(relInfo.roleId());
                roleInMasterAclPath.check(aclContext);
            }
        };
        checkRelRange(masterRelationsDiff.added);
        checkRelRange(masterRelationsDiff.deleted);
    }
}

bool
CheckPermissions::isUserHasAccessViewCategoryAttribute(
    const std::string& categoryId,
    const std::string& attribute)
{
    const auto& category = cfg()->editor()->categories()[categoryId];
    return acl::SubjectPath(SERVICE_ACL_BASE)
        (category.aclPermissionName())
        (STR_VIEW_PERMISSION)
        (STR_ATTRIBUTES_PERMISSION)(attribute).isAllowed(globalContext());
}

void
CheckPermissions::checkPermissionsToObjectAttributes(
    const acl::CheckContext& aclContext, const GeoObject* obj)
{
    if ((!obj->primaryEdit() && !obj->category().complex())
        || (!obj->isModifiedAttr() && !obj->isModifiedTableAttrs())) {
        return;
    }
    if (!obj->category().complex()) {
        checkPermissionsToObjectAttributesWithContext(aclContext, obj);
        return;
    }
    try {
        checkPermissionsToObjectAttributesWithContext(globalContext(), obj);
        return;
    } catch (const wiki::acl::AccessDenied& ex) {
        // no global access
    }
    std::vector<std::string> slavesWKB;
    for (const auto& slaveInfo : obj->slaveRelations().range()) {
        const GeoObject* slave = slaveInfo.relative();
        if (!slave->geom().isNull()) {
            slavesWKB.push_back(slave->geom().wkb());
        }
    }
    auto geomContext = globalContext();
    geomContext = geomContext.narrow(slavesWKB, work_);
    checkPermissionsToObjectAttributesWithContext(geomContext, obj);
}

namespace {
bool isValueSetByDefaults(const GeoObject* obj, const Attribute& attr)
{
    const auto& def = attr.def();
    if (obj->isCreated() && attr.value() == def.defaultValue())
    {
        return true;
    }
    const auto& attrs = obj->attributes();

    if (def.id().ends_with(ATTR_DISP_CLASS_SUFFIX) &&
        attrs.isDefined(ATTR_POI_BUSINESS_RUBRIC_ID))
    {
        const auto& rubricsCfg = cfg()->editor()->externals().rubrics();
        const auto& rubricId = attrs.value(ATTR_POI_BUSINESS_RUBRIC_ID);
        if (rubricId.empty() || attr.value().empty()) {
            return false;
        }
        const auto defaultDispClass =
            rubricsCfg.defaultDispClass(boost::lexical_cast<RubricId>(rubricId));
        if (defaultDispClass == boost::lexical_cast<DispClass>(attr.value())) {
            return true;
        }
    }
    if (isPermalinkReset(obj) &&
        attr.value().empty() &&
        (def.id() == ATTR_SYS_IMPORT_SOURCE ||
            def.id() == ATTR_SYS_IMPORT_SOURCE_ID))
    {
        return true;
    }

    return false;
}
} // namespace
void
CheckPermissions::checkPermissionsToObjectAttributesWithContext(
    const acl::CheckContext& aclContext, const GeoObject* obj)
{
    const ObjectUpdateData* updateData = nullptr;
    if (objectsUpdateData_) {
        auto updateDataIt = objectsUpdateData_->find(obj->id());
        if (updateDataIt != objectsUpdateData_->end()) {
            updateData = updateDataIt->second;
        }
    }
    acl::SubjectPath aclPath(SERVICE_ACL_BASE);
    if (obj->hasExistingOriginal() && obj->categoryId() != obj->original()->categoryId()) {
        aclPath(obj->category().aclPermissionName()).checkPartialAccess(aclContext);
        aclPath(obj->original()->category().aclPermissionName()).checkPartialAccess(aclContext);
    }
    for (const auto& attr : obj->attributes()) {
        if (attr.def().system() || attr.def().table()) {
            continue;
        }
        const auto& oldAttrId =
            obj->hasExistingOriginal()
            ? boost::replace_all_copy(attr.id(), obj->categoryId(), obj->original()->categoryId())
            : attr.id();
        if (updateData) {
            const auto updatedValues = updateData->attributes().equal_range(attr.id());
            if (updatedValues.first == updatedValues.second) {
                continue;
            } else if (obj->hasExistingOriginal()) {
                auto oldVals = obj->original()->attributes().values(oldAttrId);
                if (std::all_of(updatedValues.first, updatedValues.second,
                        [&](const ObjectUpdateData::AttributesValues::value_type& idAndValue)
                        {
                            return oldVals.contains(idAndValue.second);
                        })
                    ) {
                    continue;
                }
            }
        }
        if (isValueSetByDefaults(obj, attr)) {
            continue;
        }
        if (obj->hasExistingOriginal()
              && attr.value() == obj->original()->attributes().value(oldAttrId)) {
            continue;
        }
        auto newVals = obj->attributes().values(attr.name());
        Attribute::Values oldVals;
        if (obj->hasExistingOriginal()) {
            oldVals = obj->original()->attributes().values(oldAttrId);
        }
        checkAttributePermissionsPerValues(
                aclPath(obj->category().aclPermissionName())
                (STR_EDIT_PERMISSION)(STR_ATTRIBUTES_PERMISSION)(attr.name()),
                aclContext,
                oldVals,
                newVals);
    }
    for (const auto& attrName : obj->tableAttributes().attrNames()) {
        const auto& vals = obj->tableAttributes().find(attrName);
        if (obj->isCreated() && vals.empty()) {
            continue;
        }
        if (obj->hasExistingOriginal()
              && obj->original()->tableAttributes().find(attrName).equal(vals)) {
            continue;
        }
        aclPath(obj->category().aclPermissionName())
            (STR_EDIT_PERMISSION)(STR_ATTRIBUTES_PERMISSION)(attrName).checkPartialAccess(aclContext);
    }
}

void
CheckPermissions::checkPermissionsToBlockedObjects(
    const acl::CheckContext& aclContext,
    const GeoObject* obj)
{
    if (isModifiedExternalRelationsOnly(obj)) {
        return;
    }
    acl::SubjectPath aclPath(SERVICE_ACL_BASE);
    auto blockingObj = findBlockingObject(obj);
    if (!blockingObj &&
        obj->hasExistingOriginal() && obj->original()->isBlocked()) {
        blockingObj = obj;
    }
    if (blockingObj) {
        aclPath(blockingObj->category().aclPermissionName())
                (STR_EDIT_PERMISSION)
                (STR_ATTRIBUTES_PERMISSION)
                (ATTR_SYS_BLOCKED).check(aclContext);
    }
}

void
CheckPermissions::checkPermissionsForPrivateObjects(
    const GeoObject* obj)
{
    if (!obj->hasExistingOriginal()) {
        return;
    }
    auto commit = revision::Commit::load(work_, obj->revision().commitId());
    WIKI_REQUIRE(commit.createdBy() == userId_,
                 ERR_FORBIDDEN,
                 "Attempt to modify private object id: " << obj->id()
                 << " by uid: " << userId_);
}

void
CheckPermissions::operator()(const GeoObject* obj)
{
    if (is<AttrObject>(obj) ||
        is<RelationObject>(obj)) {
        return;
    }

    DEBUG()
        << "CheckPermissions for object " << obj->id()
        << " (" << obj->categoryId() << ").";
    if (obj->isPrivate()) {
        checkPermissionsForPrivateObjects(obj);
    }

    std::vector<std::string> wkbs;
    if (obj->categoryId() != CATEGORY_AOI) {
        if (!obj->geom().isNull()) {
            wkbs.push_back(obj->geom().wkb());
        }
        if (obj->hasExistingOriginal() && !obj->original()->geom().isNull()) {
            wkbs.push_back(obj->original()->geom().wkb());
        }
    }
    RelationInfosDiff slavesDiff;
    if (needCheckRelationsPermissions(obj->categoryId())) {
        slavesDiff = obj->slaveRelations().diff();
        for (const auto& slaveInfo : slavesDiff.added) {
            const GeoObject* slave = slaveInfo.relative();
            if (!slave->geom().isNull()) {
                wkbs.push_back(slave->geom().wkb());
            }
        }
        for (const auto& slaveInfo : slavesDiff.deleted) {
            const GeoObject* slave = slaveInfo.relative();
            if (!slave->geom().isNull()) {
                wkbs.push_back(slave->geom().wkb());
            }
        }
    }
    auto aclContext = globalContext();
    aclContext = aclContext.narrow(wkbs, work_);
    if (obj->isCreated()) {
        checkPermissionsToCreateObject(aclContext, obj);
    }
    checkPermissionsToBlockedObjects(aclContext, obj);
    if (obj->isDeleted() && !isLinearElementAffectedByJunctionDelete(obj)) {
        checkPermissionsToDeleteObject(aclContext, obj);
        return;
    }
    checkPermissionsToObjectRelations(aclContext, obj, slavesDiff);
    checkPermissionsToObjectGeometry(aclContext, obj);
    checkPermissionsToObjectAttributes(aclContext, obj);
    checkPermissionsToEditObject(obj, CheckPolicy::ModifiedPrimary);
}

void
CheckPermissions::checkPermissionsToManageBranches()
{
    PERMISSION_TO_MANAGE_BRANCHES.check(globalContext());
}

void
CheckPermissions::checkPermissionsToEditBranch(TBranchId branchId)
{
    if (branchId != revision::TRUNK_BRANCH_ID) {
        PERMISSION_TO_EDIT_STABLE.check(globalContext());
    }
}

bool
CheckPermissions::isUserHasAccessToViewBranches()
{
    try {
        acl::CheckContext globalViewContext(
            userId_,
            std::vector<std::string>{},
            work_,
            {acl::User::Status::Active});
        return PERMISSION_TO_VIEW_BRANCHES.isAllowed(globalViewContext);
    } catch (const wiki::acl::AccessDenied&) {
        // banned user
        return false;
    }
}

void
CheckPermissions::checkPermissionsForCloneOperation(const GeoObject* object)
{
    auto checkContext = globalContext();
    if (!object->geom().isNull()) {
        checkContext = checkContext.narrow({object->geom().wkb()}, work_);
    } else {
        StringSet geomPartsRoles = object->category().slaveRoleIds(roles::filters::IsGeom);
        StringVec wkbs;
        const auto geomParts = object->slaveRelations().range(geomPartsRoles);
        wkbs.reserve(geomParts.size());
        for (const auto& geomPart : geomParts) {
            ASSERT(!geomPart.relative()->geom().isNull());
            wkbs.emplace_back(geomPart.relative()->geom().wkb());
        }
        checkContext = checkContext.narrow(wkbs, work_);
    }
    PERMISSION_TO_CLONE(object->categoryId()).check(checkContext);
}

void
CheckPermissions::checkPermissionsForGroupMove(
    const StringSet& categoryIds)
{
    for (const auto& categoryId : categoryIds) {
        PERMISSION_TO_GROUP_MOVE(categoryId).check(globalContext());
    }
}

void
CheckPermissions::checkPermissionsForGroupSyncGeometry(
    const std::string& categoryId)
{
   PERMISSION_TO_GROUP_SYNC_GEOMETRY(categoryId).check(globalContext());
}

void
CheckPermissions::checkPermissionsForGroupUpdateAttributes(
    const std::string& categoryId,
    const StringMultiMap& attributes)
{
    auto categoryAclPath = PERMISSION_TO_GROUP_UPDATE_ATTRIBUTES(categoryId);
    for (const auto& attr : attributes) {
        categoryAclPath
        (STR_ATTRIBUTES_PERMISSION)(attr.first).check(globalContext());
    }
}

void
CheckPermissions::checkPermissionsForGroupUpdateRelations(
    const StringSet& categoryIds)
{
    for (const auto& categoryId : categoryIds) {
        PERMISSION_TO_GROUP_UPDATE_RELATIONS(categoryId).check(globalContext());
    }
}

void
CheckPermissions::checkPermissionsForGroupUpdateRelation()
{
    PERMISSION_TO_GROUP_UPDATE_RELATION.check(globalContext());
}

void
CheckPermissions::checkPermissionsForGroupDelete()
{
    PERMISSION_TO_GROUP_DELETE.check(globalContext());
}

void
CheckPermissions::checkPermissionsForGroupDeleteComments()
{
    PERMISSION_TO_GROUP_DELETE_COMMENTS.check(globalContext());
}

void
CheckPermissions::checkPermissionForCompletedByOthersTaskFeed()
{
    PERMISSION_FOR_COMPLETED_BY_OTHERS_TASK_FEED.check(globalContext());
}

namespace {
std::unordered_set<std::string>
collectObjectAccessControlAttributesValuesPaths(const GeoObject* object)
{
    std::unordered_set<std::string> attrValueSubPath;
    auto add = [&](const std::string& attrId, const std::string& value) {
        if (!value.empty()) {
            attrValueSubPath.emplace(attrId + "/" + value);
        } else {
            attrValueSubPath.emplace(attrId + "/" + EMPTY_VALUE_ID);
        }
    };
    for (const auto& attrDef : object->category().attrDefs()) {
        if (!attrDef->objectAccessControl() || attrDef->table()) {
            continue;
        }
        const auto& attrId = attrDef->id();
        if (attrDef->multiValue()) {
            for (const auto& value : object->attributes().values(attrId)) {
                add(attrId, value);
            }
        } else {
            const auto& value = object->attributes().value(attrId);
            add(attrId, value);
        }
    }
    return attrValueSubPath;
}
} // namespace


void
CheckPermissions::checkPermissionsToViewObject(const GeoObject* object)
{
    checkPermissionsToPerformActionOnObject(object, STR_VIEW_PERMISSION);
}

void
CheckPermissions::checkPermissionsToEditObject(const GeoObject* object, CheckPolicy checkPolicy)
{
    if (checkPolicy == CheckPolicy::All ||
        (object->primaryEdit() && object->isModified()))
    {
        checkPermissionsToPerformActionOnObject(object, STR_EDIT_PERMISSION);
    }
}

bool
CheckPermissions::isUserHasAccessToViewCategory(const std::string& categoryId)
{
    acl::CheckContext globalViewContext = globalContext();
    globalViewContext = globalViewContext.inflate();
    acl::SubjectPath aclPath(SERVICE_ACL_BASE);
    return aclPath(categoryId)(STR_VIEW_PERMISSION).isAllowedPartially(globalViewContext);
}

bool
CheckPermissions::isUserHasAccessToViewValue(
    const std::string& categoryId,
    const std::string& attrId,
    const std::string& value)
{
    acl::CheckContext globalViewContext = globalContext();
    globalViewContext = globalViewContext.inflate();
    acl::SubjectPath aclPath(SERVICE_ACL_BASE);
    const auto checkValue = value.empty()
        ? EMPTY_VALUE_ID
        : value;
    return aclPath(categoryId)
        (STR_VIEW_PERMISSION)(STR_ATTRIBUTES_PERMISSION)
            (attrId)(checkValue).isAllowedPartially(globalViewContext);
}

void
CheckPermissions::checkPermissionsToPerformActionOnObject(
    const GeoObject* object,
    const std::string& action)
{
    acl::SubjectPath aclPath(SERVICE_ACL_BASE);
    aclPath = aclPath(object->category().aclPermissionName())
        (action);
    acl::CheckContext globalViewContext = globalContext();
    auto objectAccessControlAttributesValuesPaths =
        collectObjectAccessControlAttributesValuesPaths(object);
    if (object->original()) {
        auto oldObjectAccessControlAttributesValuesPaths =
            collectObjectAccessControlAttributesValuesPaths(object->original().get());
        objectAccessControlAttributesValuesPaths.insert(
            oldObjectAccessControlAttributesValuesPaths.begin(),
            oldObjectAccessControlAttributesValuesPaths.end());
    }
    auto check = [&](const auto& checkContext) {
        aclPath.checkPartialAccess(checkContext);
        for (const auto& subPath : objectAccessControlAttributesValuesPaths) {
            aclPath(STR_ATTRIBUTES_PERMISSION)(subPath).check(checkContext);
        }
    };
    if (!object->category().complex()) {
        ASSERT(!object->geom().isNull());
        auto context = globalViewContext.narrow({object->geom().wkb()}, work_);
        check(context);
        return;
    }
    try {
        check(globalViewContext);
        return;
    } catch (const wiki::acl::AccessDenied& ex) {
        // no global access
        auto slaveInfos = object->slaveRelations().range(
                RelativesLimit {
                    cfg()->editor()->system().slavesPerRoleLimit(),
                    RelativesLimit::PerRole
                });
        if (!slaveInfos || !slaveInfos->rolesWithExceededLimit().empty()) {
            throw;
        }
        std::vector<std::string> slavesWKB;
        for (const auto& slaveInfo : *slaveInfos) {
            const GeoObject* slave = slaveInfo.relative();
            if (!slave->geom().isNull()) {
                slavesWKB.push_back(slave->geom().wkb());
            }
        }
        auto geomContext = globalViewContext.narrow(slavesWKB, work_);
        check(geomContext);
    }
}

void
CheckPermissions::checkPermissionsToBlockingTasks()
{
    PERMISSION_TO_BLOCKING_TASKS.check(globalContext());
}

bool
CheckPermissions::allowToIgnoreRestrictions(const Category& category)
{
    acl::SubjectPath aclPath(PERMISSION_TO_IGNORE_RESTRICTIONS_ACL_BASE);
    aclPath = aclPath(category.aclPermissionName());
    return permissionExists(work_, aclPath) && aclPath.isAllowed(globalContext());
}

bool
CheckPermissions::isUserHasAccessToIgnoreOverlay(const StringSet& baseCategories)
{
    return std::any_of(baseCategories.begin(), baseCategories.end(),
        [&](const auto& categoryId) {
            const auto& category = cfg()->editor()->categories()[categoryId];
            acl::SubjectPath aclPath(SERVICE_ACL_BASE);
            return aclPath(category.aclPermissionName())
                (STR_EDIT_PERMISSION)
                (STR_ATTRIBUTES_PERMISSION)
                (ATTR_SYS_IGNORE_FORBIDDEN_OVERLAY).isAllowed(globalContext());
        });
}

bool
CheckPermissions::isUserHasAccessToStickPolygons()
{
    return PERMISSION_SETTINGS_STICK_POLYGONS.isAllowed(globalContext());
}

void
CheckPermissions::checkUserHasAccessToIsLocalManualPolicy()
{
    PERMISSION_SETTINGS_AUTO_IS_LOCAL.check(globalContext());
}

void
CheckPermissions::checkUserHasAccessToValidatorTasks()
{
    PERMISSION_TASKS_VALIDATOR.check(globalContext());
}

void
CheckPermissions:: checkAccessToEditorMode()
{
    PERMISSION_SETTINGS_EDITOR_MODE.check(globalContext());
}

void
CheckPermissions::checkPermissionsToCreateObject(
    const acl::CheckContext& aclContext,
    const GeoObject* object)
{
    acl::SubjectPath aclPath(SERVICE_ACL_BASE);
    aclPath = aclPath(object->category().aclPermissionName())
        (STR_CREATE_PERMISSION);
    aclPath.check(aclContext);
}

void
CheckPermissions::checkPermissionsToEditFilters()
{
    acl::SubjectPath aclPath(PERMISSION_TO_FILTERS);
    aclPath(STR_EDIT_PERMISSION).check(globalContext());
}

void
CheckPermissions::checkPermissionsToViewFilters()
{
    acl::SubjectPath aclPath(PERMISSION_TO_FILTERS);
    aclPath(STR_VIEW_PERMISSION).check(globalContext());
}

bool
CheckPermissions::isUserHasAccessToViewFilters()
{
    acl::SubjectPath aclPath(PERMISSION_TO_FILTERS);
    return aclPath(STR_VIEW_PERMISSION).isAllowed(globalContext());
}

bool
CheckPermissions::isUserHasAccessToEditPublicFilters()
{
    acl::SubjectPath aclPath(PERMISSION_TO_FILTERS);
    return aclPath(STR_EDIT_PUBLIC_PERMISSION).isAllowed(globalContext());
}

void
CheckPermissions::checkAccessToPoisConflictsTool()
{
    acl::SubjectPath(PERMISSION_TOOLS_POIS_CONFLICTS).check(globalContext());
}

void CheckPermissions::checkAccessToBusinessPhotosTool()
{
    acl::SubjectPath(PERMISSION_TOOLS_BUSINESS_PHOTOS).check(globalContext());
}

void
CheckPermissions::checkAccessToEditsFeed()
{
    acl::SubjectPath(PERMISSION_TOOLS_EDITS_FEED).check(globalContext());
}

bool
CheckPermissions::isUserHasAccessToEventAlert(
    const social::EventAlert& alert,
    const std::string& categoryId)
{
    auto iter = REQUIRED_ATTRIBUTES_VIEW_ACCESS.find(alert.description());
    if (iter == REQUIRED_ATTRIBUTES_VIEW_ACCESS.end()) {
        return true;
    }
    for (const auto& attrName : iter->second) {
        if (!isUserHasAccessViewCategoryAttribute(categoryId, attrName)) {
            return false;
        }
    }
    return true;
}

bool
CheckPermissions::canUserAnnotateOtherUserCommit()
{
    return PERMISSION_TO_ANNOTATE_COMMITS.isAllowed(globalContext());
}

bool
CheckPermissions::canUserDeleteOtherUserComment()
{
    return PERMISSION_TO_DELETE_COMMENTS.isAllowed(globalContext());
}

} // namespace wiki
} // namespace maps
