#include "collection.h"

namespace maps {
namespace wiki {

ObjectPtr
GeoObjectCollection::getById(TOid id) const
{
    std::map<TOid, TObjectsIt>::const_iterator it = objectsMap_.find(id);
    if (it != objectsMap_.end()) {
        return *(it->second);
    }
    return ObjectPtr();
}


GeoObjectCollection::TObjectsCIt
GeoObjectCollection::begin() const
{
    return objects_.begin();
}

GeoObjectCollection::TObjectsCIt
GeoObjectCollection::end() const
{
    return objects_.end();
}

ObjectPtr
GeoObjectCollection::front() const
{
    return objects_.front();
}

ObjectPtr GeoObjectCollection::back() const
{
    return objects_.back();
}

bool
GeoObjectCollection::empty() const
{
    return objects_.empty();
}

size_t
GeoObjectCollection::size() const
{
    return objects_.size();
}

void
GeoObjectCollection::clear()
{
    objects_.clear();
    objectsMap_.clear();
}

ObjectPtr
GeoObjectCollection::primaryObject() const
{
    ObjectPtr primaryObj;
    for (const auto& obj : *this) {
        if (obj->primaryEdit()) {
            if (primaryObj) {
                primaryObj.reset();
                break;
            }
            primaryObj = obj;
        }
    }
    return primaryObj;
}

ObjectPtr
GeoObjectCollection::add(ObjectPtr object)
{
    std::map<TOid, TObjectsIt>::iterator mapIt = objectsMap_.find(object->id());
    if (mapIt == objectsMap_.end()) {
        objects_.push_back(object);
        TObjectsIt it = objects_.end();
        --it;
        objectsMap_.insert(std::make_pair(object->id(), it));
        return *it;
    }
    return *(mapIt->second);
}

void
GeoObjectCollection::remove(TOid id)
{
    std::map<TOid, TObjectsIt>::iterator it = objectsMap_.find(id);
    if (it != objectsMap_.end()) {
        objects_.erase(it->second);
        objectsMap_.erase(it);
    }
}

void
GeoObjectCollection::append(const GeoObjectCollection& collection)
{
    for (const auto& object : collection) {
        add(object);
    }
}

GeoObjectCollection
GeoObjectCollection::find(const ObjectPredicate& predicate) const
{
    GeoObjectCollection res;
    for (const auto& pair : objectsMap_) { //output sorted collection
        if (predicate((*pair.second).get())) {
            res.add(*pair.second);
        }
    }
    return res;
}

void
GeoObjectCollection::dump(std::ostream& os) const
{
    os << "***Collection contents.***" << std::endl;
    for (const auto& objectPair : objectsMap_) {
        const GeoObject* object = objectPair.second->get();
        os << "DUMP ID:" << object->id();
        os << " CATEGORY:" << object->categoryId();
        if (object->revision().valid()) {
            os << " REVISION:" << object->revision();
        }
        os << " DELETED:" << object->isDeleted();
        if (!object->geom().isNull()) {
            std::stringstream outc;
            outc.precision(DOUBLE_FORMAT_PRECISION);
            object->geom().geoJson(outc, SpatialRefSystem::Mercator);
            os << " GEOM:" << outc.str();
        }
        os << std::endl;
    }
}

GeoObjectCollection::GeoObjectCollection(std::initializer_list<ObjectPtr> objects)
{
    for (const auto& obj : objects) {
        add(obj);
    }
}

GeoObjectCollection::GeoObjectCollection(const GeoObjectCollection& col)
{
    append(col);
}

GeoObjectCollection&
GeoObjectCollection::operator = (GeoObjectCollection&& col)
{
    if (&col != this) {
        clear();

        objects_.swap(col.objects_);
        objectsMap_.swap(col.objectsMap_);
    }
    return *this;
}

} // namespace wiki
} // namespace maps
