#pragma once

#include "common.h"
#include "approve_status.h"
#include "edit_notes.h"

#include <yandex/maps/wiki/common/commit_properties.h>

#include <boost/optional.hpp>

#include <iostream>
#include <vector>

namespace maps::wiki {

namespace revision {
class Commit;
}

class GeoObjectCollection;
class ObjectsCache;

namespace commit_state {
const std::string DRAFT = "draft";
const std::string PREAPPROVING = "preapproving";
const std::string APPROVING = "approving";
const std::string APPROVED = "approved";
} // commit_state

enum class RevertReason
{
    DuplicateCreation,
    UnrealObjectCreation,
    UnfinishedObjectCreation,
    ForbiddenObjectCreation,
    IncorrectObjectCategory,
    IncorrectObjectAttr,
    IncorrectObjectGeometry,
    UnchartedObjectCreation,
    Vandalism,
    Self,
    Other
};

std::ostream& operator << (std::ostream& os, const RevertReason& reason);
std::istream& operator >> (std::istream& is, RevertReason& reason);

//!Create attributes for commit notification
//!http://wiki.yandex-team.ru/JandeksKarty/development/fordevelopers/wikimap/mapspro/feeds
StringMap
createCommitNotes(const std::string& action, const StringMap& actionNotes,
    const GeoObjectCollection& objects, ObjectsCache& cache);

StringMap
actionNotes(const revision::Commit& commit);

boost::optional<TOid>
primaryObjectId(const revision::Commit& commit);

TOIds
affectedObjectIds(const revision::Commit& commit);

std::string
objectEditNotesKey(TOid objectId);

std::string
primaryObjectKey(TOid objectId);

edit_notes::EditNotesTree
objectEditNotesTree(const revision::Commit& commit, TOid objectId);

edit_notes::EditNotesTree
objectEditNotesTree(const std::string& notesString);

bool
hasEditNotes(const revision::Commit& commit);

TOIds
oidsWithNotes(const revision::Commit& commit);

boost::optional<RevertReason>
revertReason(const revision::Commit& commit);

std::string
commitAttributesObjectEditNotes(const StringMap& attributes, TOid objectId);

revision::Commit recentAffectingCommit(ObjectsCache& cache, const TRevisionId& objRevId);

revision::Commit firstCommit(ObjectsCache& cache, TOid objectId);

TBranchId sourceBranchId(const revision::Commit& commit);


class CommitModel
{
public:
    CommitModel(revision::Commit commit);

    const revision::Commit& commit() const;
    const boost::optional<TOid>& contextPrimaryObjectId() const;
    bool primary() const;
    boost::optional<edit_notes::EditNotesTree> editNotesTree() const;
    const boost::optional<std::string>& state() const;
    const boost::optional<approve_status::ApproveStatus>& approveStatus() const;
    const boost::optional<bool>& last() const;
    const boost::optional<TId>& feedbackTaskId() const;
    const boost::optional<bool>& isRevertible() const;

    void setCustomContextObjectId(TOid customContextObjectId);
    void setupState(const BranchContext& branchCtx);
    void setApproveStatus(approve_status::ApproveStatus approveStatus);
    void setupApproveStatus(const BranchContext& branchCtx);
    void setupFeedbackTaskId(const BranchContext& branchCtx);
    void setIsRevertible(bool);

private:
    friend class BatchCommitPreparedFields;

    revision::Commit commit_;
    boost::optional<TOid> primaryObjectId_;
    boost::optional<TOid> contextPrimaryObjectId_;
    boost::optional<std::string> state_;
    boost::optional<approve_status::ApproveStatus> approveStatus_;
    boost::optional<bool> last_;
    boost::optional<TId> feedbackTaskId_;
    boost::optional<bool> isRevertible_;
};

typedef std::vector<CommitModel> CommitModels;


class BatchCommitPreparedFields
{
public:
    void prepareStates(
        const BranchContext& branchCtx,
        const std::list<revision::Commit>& commits);
    void prepareApproveStatuses(
        const BranchContext& branchCtx,
        const std::list<revision::Commit>& commits);
    void prepareLastFlags(
        const BranchContext& branchCtx,
        const std::list<revision::Commit>& commits,
        TBranchId branchId);

    void fillCommitModel(CommitModel& commitModel);
private:
    std::map<TId, std::string> commitIdToState_;
    std::map<TId, approve_status::ApproveStatus> commitIdToApproveStatus_;
    bool lastCommitsWasPrepared_ = false;
    TCommitIds lastCommitIds_;
};

} // namespace maps::wiki
