#include "config.h"
#include "rate_limiter.h"

#include "pool_sections.h"

#include "maps/wikimap/mapspro/services/editor/src/common.h"
#include "maps/wikimap/mapspro/services/editor/src/exception.h"
#include "maps/wikimap/mapspro/services/editor/src/moderation_log.h"
#include "maps/wikimap/mapspro/services/editor/src/rendererpool.h"

#include "maps/wikimap/mapspro/services/editor/src/approved_commits/updater.h"
#include "maps/wikimap/mapspro/services/editor/src/background_tasks/deleted_users_cache_updater.h"
#include "maps/wikimap/mapspro/services/editor/src/background_tasks/monitoring.h"
#include "maps/wikimap/mapspro/services/editor/src/background_tasks/suspicious_users_table_cleaner.h"
#include "maps/wikimap/mapspro/services/editor/src/background_tasks/sprav_tasks_updater.h"
#include "maps/wikimap/mapspro/services/editor/src/revision_meta/updater.h"

#include <maps/libs/chrono/include/days.h>
#include <maps/libs/common/include/environment.h>
#include <maps/libs/geolib/include/common.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/tile/include/utils.h>

#include <maps/wikimap/mapspro/libs/acl_utils/include/cached_moderation_statuses.h>
#include <yandex/maps/wiki/social/moderation_time_intervals.h>
#include <yandex/maps/wiki/threadutils/executor.h>
#include <yandex/maps/wiki/threadutils/threadpool.h>
#include <yandex/maps/wiki/threadutils/thread_pools.h>
#include <yandex/maps/wiki/validator/validator.h>

#include <boost/lexical_cast.hpp>

using namespace std::chrono_literals;
namespace maps {
namespace wiki {

//------------------- M A G I C   Z O N E -------------------

namespace {

const size_t REVISION_META_UPDATER_THREADS = 3;

const size_t DEFAULT_MONITORING_TIMEOUT_SEC = 60;
const size_t DEFAULT_SUSPICIOUS_USERS_CLEANER_TIMEOUT_DAYS = 14;

constexpr std::chrono::seconds DEFAULT_SPRAV_TASK_UPDATE_TIMEOUT = 300s;
constexpr std::chrono::seconds NOT_SUBMITTED_SPRAV_TASK_UPDATE_TIMEOUT = 30s;
constexpr std::chrono::minutes USERGROUPS_UIDS_REFRESH_PERIOD = 60min;


const size_t DEFAULT_SUGGEST_COUNT = 10;

const std::string EDITOR_CONFIG_XPATH = "/config/services/editor/config";
const std::string BRANCH_MASK_CLEANER_XPATH = "/config/services/editor/branch-mask-cleaner";
const std::string MODERATION_TIME_INTERVALS_XPATH = "/config/services/social/moderation-time-intervals";

const std::string DEFAULT_EDITOR_LAYERS_PATH = "/usr/share/yandex/maps/wiki/renderer/layers";

const std::string DESIGNSTAND = "designstand";

const std::string TVM_SERVICE = "nmaps-editor";

const std::string TASKUTILS_CONFIG_PATH = "/config/services/editor/tasks/";

const std::set<maps::common::Environment> ENVIRONMENTS_WITH_TVM = {
    maps::common::Environment::Stable,
    maps::common::Environment::Testing,
    maps::common::Environment::Unstable,
};

size_t masterPoolMaxSize(pgpool3::Pool& pool)
{
    return pool.state().constants.masterMaxSize;
}

const StringVec EXTRA_POOL_SECTIONS {
    POOL_SECTION_NAME_SOCIAL,
    POOL_SECTION_NAME_VIEW_TRUNK,
    POOL_SECTION_NAME_VIEW_STABLE,
    POOL_SECTION_NAME_LABELS_TRUNK,
    POOL_SECTION_NAME_LABELS_STABLE,
    POOL_SECTION_NAME_VALIDATION,
};

} // namespace

//------------------Config------------------

Config::Config(
        const std::string& filename,
        const std::string& coreSectionName,
        const std::string& poolName,
        const StringSet& allowedExtraPoolSections,
        LongReadAccess lrAccess)
    : doc_(filename)
    , lrAccess_(lrAccess)
    , deletedUsersCache_(make_unique<acl::DeletedUsersCache>())
    , threadPools_(make_unique<ThreadPools>())
{
    projectName_ = get<std::string>("/config/common/project");
    logLevel_ = get<log8::Level>("/config/log/level", log8::Level::INFO);

    tmpDir_ = get<std::string>("/config/common/tmp-dir");
    feedbackApiUrl_ = get<std::string>("/config/common/fbapi/base-url");
    suggestCount_ = get<size_t>("/config/suggest/count", DEFAULT_SUGGEST_COUNT);
    viewPartsEnabled_ = getAttr<bool>(
        "/config/services/editor/view-parts", "enabled", true);

    branchMaskCleanEmptyEnabled_ = getAttr<bool>(
        BRANCH_MASK_CLEANER_XPATH, "clean-empty", true);
    branchMaskMergeDuplicateEnabled_ = getAttr<bool>(
        BRANCH_MASK_CLEANER_XPATH, "merge-duplicate", true);

    moderationTimeIntervals_ = make_unique<social::ModerationTimeIntervals>();
    moderationTimeIntervals_->supervisorDelay = std::chrono::seconds(get<int>(
        MODERATION_TIME_INTERVALS_XPATH + "/supervisor-delay"));
    moderationTimeIntervals_->superModeratorDelay = std::chrono::seconds(get<int>(
        MODERATION_TIME_INTERVALS_XPATH + "/supermoderator-delay"));
    moderationTimeIntervals_->moderatorOldTaskAge = std::chrono::seconds(get<int>(
        MODERATION_TIME_INTERVALS_XPATH + "/moderator-old-task-age"));
    moderationTimeIntervals_->superModeratorOldTaskAge = std::chrono::seconds(get<int>(
        MODERATION_TIME_INTERVALS_XPATH + "/supermoderator-old-task-age"));

    std::mutex poolMutex;
    auto setPoolHolder = [&](const auto& sectionName, auto holder) {
        INFO() << "Init pool section: " << sectionName << " end";
        std::lock_guard<std::mutex> lock(poolMutex);
        poolHolders_[sectionName] = std::move(holder);
    };

    auto createConnMgrHolder = [&] (const std::string& sectionName) {
        INFO() << "Init pool section: " << sectionName << " start";
        try {
            setPoolHolder(
                sectionName,
                std::make_shared<common::PoolHolder>(
                    doc_, sectionName, poolName, "pgpool." + sectionName));
        } catch (const maps::xml3::NodeNotFound&) {
            WARN() << "section " << sectionName
                   << " not found in config, initializing pool " << poolName
                   << " from section " << coreSectionName;
            setPoolHolder(
                sectionName,
                std::make_shared<common::PoolHolder>(
                    doc_, coreSectionName, poolName, "pgpool." + sectionName));
        }
    };

    const auto extraPoolSections =
        allowedExtraPoolSections.empty()
        ? EXTRA_POOL_SECTIONS
        : StringVec{ allowedExtraPoolSections.begin(), allowedExtraPoolSections.end() };

    Executor executor;
    for (const auto& sectionName : extraPoolSections) {
        executor.addTask([&, sectionName] { createConnMgrHolder(sectionName); });
    }

    executor.addTask([&] {
        INFO() << "Init pool section: " << POOL_SECTION_NAME_CORE << " (" << coreSectionName << ") start";
        setPoolHolder(
            POOL_SECTION_NAME_CORE,
            std::make_shared<common::PoolHolder>(
                doc_, coreSectionName, poolName, "pgpool." + coreSectionName));
    });

    executor.addTask([&] {
        rateLimiter_ = make_unique<RateLimiter>(doc_.node("/config/common/rate-limiter", true));

        initRenderer();
        initEditorCfg();
    });

    auto poolHolder = threadPools().pool(extraPoolSections.size() + 1);
    executor.executeAll(*poolHolder);

    userGroupsUidsCache_ = std::make_unique<acl_utils::UserGroupsUidsCache>();
}

Config::~Config()
{
    asyncTasksSupport_.reset();
    afterCommitThreadPools_.clear();
    poolHolders_.clear();
    threadPools_.reset();
}

void
Config::onLoad()
{
    editorCfg_->onLoad();
}

void
Config::initAfterCommitThreadPools()
{
    auto& core = poolCore();
    const auto corePoolSize = masterPoolMaxSize(core);
    const auto viewTrunkPoolSize = masterPoolMaxSize(poolViewTrunk());
    const auto viewStablePoolSize = masterPoolMaxSize(poolViewStable());
    const auto labelsTrunkPoolSize = masterPoolMaxSize(poolLabelsTrunk());
    const auto labelsStablePoolSize = masterPoolMaxSize(poolLabelsStable());

    auto minViewPoolSize = std::min(viewTrunkPoolSize, viewStablePoolSize);
    auto minLabelsPoolSize = std::min(labelsTrunkPoolSize, labelsStablePoolSize);

    afterCommitThreadPools_.insert(std::make_pair(
        AfterCommitPoolType::View,
        ThreadPool(std::min(minViewPoolSize, minLabelsPoolSize) / 2 + 1)));

    afterCommitThreadPools_.insert(std::make_pair(
        AfterCommitPoolType::Core, ThreadPool(corePoolSize / 2 + 1)));
}

void
Config::initThreadPools(const std::string& programName)
{
    // unstable, testing, stable, designstand, load
    auto& core = poolCore();
    deletedUsersCacheUpdater_ = make_unique<DeletedUsersCacheUpdater>(core, *deletedUsersCache_);

    initMonitoring();
    initUidsByModerationStatusCache();

    if (maps::common::getYandexEnvironment() == maps::common::Environment::Load) {
        return;
    }

    // unstable, testing, stable, designstand
    initAfterCommitThreadPools();
    asyncTasksSupport_ =
        make_unique<controller::AsyncTasksSupport>(
            core,
            doc_,
            TASKUTILS_CONFIG_PATH);

    if (programName.find(DESIGNSTAND) != std::string::npos) {
        initTvmClient();
        return;
    }

    // unstable, testing, stable
    initCommentsLogger();
    initSuspiciousUsersTableCleaner();
    initSpravTaskUpdaters();

    revisionMetaUpdater_ =
        make_unique<RevisionMetaUpdatersManager>(REVISION_META_UPDATER_THREADS);

    auto enabled = getAttr<bool>(
        "/config/services/editor/approved-commits-updater", "enabled", true);
    if (enabled) {
        approvedCommitsUpdater_ = make_unique<approved_commits::Updater>();
    }

    auto logPath = get<std::string>(
        "/config/services/editor/moderation-log", {});
    if (!logPath.empty()) {
        moderationLogger_ = make_unique<ModerationLogger>(logPath);
    }
}

void
Config::initMonitoring()
{
    const auto monitoringTimeout = std::chrono::seconds(
        get<size_t>("/config/misc/monitoringTimeoutSec", DEFAULT_MONITORING_TIMEOUT_SEC)
    );

    for (const auto& name2poolHolder : poolHolders_) {
        const auto& poolHolderPtr = name2poolHolder.second;
        ASSERT(poolHolderPtr);
        monitoring_.emplace_back(
            poolHolderPtr->pool(), name2poolHolder.first, monitoringTimeout);
    }
}

void
Config::initCommentsLogger()
{
    auto commentsLog = get<std::string>("/config/services/editor/comments-log", s_emptyString);
    if (!commentsLog.empty()) {
        commentsLogger_ = make_unique<common::TskvLogger>(commentsLog);
    }
}

std::string
Config::getTvmTicket(const std::string& clientAlias) const
{
    if (!tvmClient_) {
        return {};
    }
    return tvmClient_->GetServiceTicketFor(clientAlias.c_str());
}

std::optional<NTvmAuth::TTvmId>
Config::getTvmSrcId(const std::string& ticket) const
{
    if (!tvmClient_ || ticket.empty()) {
        return std::nullopt;
    }
    auto parsedTicket = tvmClient_->CheckServiceTicket(ticket);
    if (parsedTicket) {
        return parsedTicket.GetSrc();
    }
    return std::nullopt;
}

void
Config::initTvmClient()
{
    if (ENVIRONMENTS_WITH_TVM.count(maps::common::getYandexEnvironment())) {
        tvmClient_ = std::make_unique<NTvmAuth::TTvmClient>(
            maps::auth::TvmtoolSettings().selectClientAlias(TVM_SERVICE).makeTvmClient());
    }
}

void
Config::initSpravTaskUpdaters()
{
    auto poolHolderIt = poolHolders_.find(POOL_SECTION_NAME_SOCIAL);
    if (poolHolderIt == poolHolders_.end()) {
        return;
    }

    initTvmClient();
    if (!tvmClient_) {
        WARN() << "Sprav task updaters not started without TVM-client";
        return;
    }

    auto& pgpool = poolHolderIt->second->pool();
    notSubmittedTasksUpdater_.reset(
        std::make_unique<SpravTasksUpdater>(
            pgpool,
            NOT_SUBMITTED_SPRAV_TASK_UPDATE_TIMEOUT,
            SpravTask::Status::NotSubmitted,
            common::AdvisoryLockIds::SPRAV_TASK_NOT_SUBMITTED
        ).release());
    newTasksUpdater_.reset(
        make_unique<SpravTasksUpdater>(
            pgpool,
            DEFAULT_SPRAV_TASK_UPDATE_TIMEOUT,
            SpravTask::Status::New,
            common::AdvisoryLockIds::SPRAV_TASK_NEW
        ).release());
    inProgressTasksUpdater_.reset(
        make_unique<SpravTasksUpdater>(
            pgpool,
            DEFAULT_SPRAV_TASK_UPDATE_TIMEOUT,
            SpravTask::Status::InProgress,
            common::AdvisoryLockIds::SPRAV_TASK_IN_PROGRESS
        ).release());
    acceptedTasksUpdater_.reset(
        make_unique<SpravTasksUpdater>(
            pgpool,
            DEFAULT_SPRAV_TASK_UPDATE_TIMEOUT,
            SpravTask::Status::Accepted,
            common::AdvisoryLockIds::SPRAV_TASK_ACCEPTED
        ).release());
}

void
Config::initUidsByModerationStatusCache()
{
    const std::vector<acl_utils::UserGroup> groupsToCache {
         acl_utils::UserGroup::Robot,
         acl_utils::UserGroup::Cartographer,
         acl_utils::UserGroup::YandexModerator,
         acl_utils::UserGroup::Outsourcer
    };

    Executor executor;
    for (auto group: groupsToCache) {
        (*userGroupsUidsCache_).try_emplace(group);

        executor.addTask([this, group] {
            userGroupsUidsCache_->at(group) =
                std::make_unique<acl_utils::UserGroupUidsCache>(
                    group, poolCore(), USERGROUPS_UIDS_REFRESH_PERIOD);
        });
    }
    auto poolHolder = threadPools().pool(
        groupsToCache.size() - 1 /* for own thread used by executeAll */);
    executor.executeAll(*poolHolder);
}

void
Config::stopSpravTaskUpdaters()
{
    acceptedTasksUpdater_.reset();
    inProgressTasksUpdater_.reset();
    newTasksUpdater_.reset();
    notSubmittedTasksUpdater_.reset();
    tvmClient_.reset();
}

void
Config::initSuspiciousUsersTableCleaner()
{
    auto poolHolderIt = poolHolders_.find(POOL_SECTION_NAME_SOCIAL);
    if (poolHolderIt == poolHolders_.end()) {
        return;
    }

    const auto cleanerTimeout = chrono::Days(
        get<size_t>(
            "/config/misc/suspiciousUsersTableCleanerTimeoutDays",
            DEFAULT_SUSPICIOUS_USERS_CLEANER_TIMEOUT_DAYS
        )
    );
    suspiciousUsersTableCleaner_ = make_unique<SuspiciousUsersTableCleaner>(
        poolHolderIt->second->pool(),
        cleanerTimeout
    );
}

void
Config::stopAfterCommitThreadPools()
{
    for (auto& pool : afterCommitThreadPools_) {
        pool.second.shutdown();
    }
}

void
Config::onUnload()
{
    if (asyncTasksSupport_) {
        asyncTasksSupport_->shutdown();
    }
    stopAfterCommitThreadPools();

    approvedCommitsUpdater_.reset();
    revisionMetaUpdater_.reset();
    stopSpravTaskUpdaters();
    suspiciousUsersTableCleaner_.reset();
    userGroupsUidsCache_.reset();
    monitoring_.clear();
    deletedUsersCacheUpdater_.reset();

    if (asyncTasksSupport_) {
        asyncTasksSupport_->terminateTasks();
    }
}

void
Config::initEditorCfg()
{
    editorCfg_ = make_unique<EditorCfg>(
        get<std::string>(EDITOR_CONFIG_XPATH));
}

void
Config::initRenderer()
{
    rendererParams_ = make_unique<RendererParams>(doc_);
    rendererPool_ = make_unique<RendererPool>(rendererParams_->mapPath());
}

log8::Level
Config::logLevel() const
{
    return logLevel_;
}

void
Config::initValidatorModules()
{
    validator::ValidatorConfig validatorConfig(get<std::string>(EDITOR_CONFIG_XPATH));
    validator::Validator v(validatorConfig);
    v.initModules();
    validatorModules_ = v.modules();
}

const RendererParams&
Config::rendererParams() const
{
    ASSERT(nullptr != rendererParams_);
    return *rendererParams_;
}

pgpool3::Pool& Config::pool(const std::string& sectionName) const
{
    auto it = poolHolders_.find(sectionName);
    REQUIRE(it != poolHolders_.end(), "Pool section '" << sectionName << "' not intialized");

    ASSERT(it->second);
    return it->second->pool();
}

pgpool3::Pool&
Config::poolCore() const
{
    return pool(POOL_SECTION_NAME_CORE);
}

pgpool3::Pool&
Config::poolSocial() const
{
    return pool(POOL_SECTION_NAME_SOCIAL);
}

pgpool3::Pool&
Config::poolValidation() const
{
    return pool(POOL_SECTION_NAME_VALIDATION);
}

pgpool3::Pool&
Config::poolViewTrunk() const
{
    return pool(POOL_SECTION_NAME_VIEW_TRUNK);
}

pgpool3::Pool&
Config::poolViewStable() const
{
    return pool(POOL_SECTION_NAME_VIEW_STABLE);
}

pgpool3::Pool&
Config::poolView(TBranchId branchId) const
{
    return branchId == revision::TRUNK_BRANCH_ID
        ? poolViewTrunk()
        : poolViewStable();
}

pgpool3::Pool&
Config::poolLabelsTrunk() const
{
    return pool(POOL_SECTION_NAME_LABELS_TRUNK);
}

pgpool3::Pool&
Config::poolLabelsStable() const
{
    return pool(POOL_SECTION_NAME_LABELS_STABLE);
}

pgpool3::Pool&
Config::poolLabels(TBranchId branchId) const
{
    return branchId == revision::TRUNK_BRANCH_ID
        ? poolLabelsTrunk()
        : poolLabelsStable();
}

ThreadPool&
Config::afterCommitThreadPool(AfterCommitPoolType type)
{
    auto it = afterCommitThreadPools_.find(type);
    ASSERT(it != afterCommitThreadPools_.end());
    return it->second;
}

const controller::AsyncTasksSupport&
Config::asyncTasksSupport() const
{
    ASSERT(asyncTasksSupport_);
    return *asyncTasksSupport_;
}

const RateLimiter&
Config::rateLimiter() const
{
    ASSERT(rateLimiter_);
    return *rateLimiter_;
}

//----------------RendererParams----------------
RendererParams::RendererParams(const common::ExtendedXmlDoc& doc)
    : isLabelerEnabled_(doc.getAttr<bool>("/config/services/editor/labeler",
                                           "enabled"))
    , mapPath_(doc.get<std::string>("/config/services/editor/labeler/mapPath"))
    , layersPath_(doc.get<std::string>("/config/services/editor/layersPath", DEFAULT_EDITOR_LAYERS_PATH))
{
    DEBUG() << "RendererParams: labelerEnabled: " << isLabelerEnabled_ << ", "
            << "mapPath: "
            << doc.get<std::string>("/config/services/editor/labeler/mapPath");
}
//---------------------------------------------
bool
RendererParams::isLabelerEnabled()const
{
    return isLabelerEnabled_;
}
//---------------------------------------------
const std::string&
RendererParams::mapPath()const
{
    return mapPath_;
}
//---------------------------------------------
const std::string&
RendererParams::layersPath()const
{
    return layersPath_;
}

//----------------ConfigHolder----------------
static std::unique_ptr<Config> s_cfg_;

ConfigScope::ConfigScope(
        const std::string& filename,
        const std::string& coreSectionName,
        const std::string& poolName,
        const StringSet& allowedExtraPoolSections,
        LongReadAccess lrAccess)
{
    s_cfg_ = make_unique<Config>(
        filename, coreSectionName, poolName, allowedExtraPoolSections, lrAccess);
}

ConfigScope::~ConfigScope()
{
    try {
        s_cfg_->onUnload();
    } catch (const maps::Exception& ex) {
        ERROR() << "unload config : exception: " << ex;
    } catch (const std::exception& ex) {
        ERROR() << "unload config : exception: " << ex.what();
    }
    s_cfg_.reset();
}

Config* cfg()
{
    ASSERT(s_cfg_);
    return s_cfg_.get();
}

} // namespace wiki
} // namespace maps
