#pragma once

#include "maps/wikimap/mapspro/services/editor/src/common.h"
#include "maps/wikimap/mapspro/services/editor/src/objects/object.h"
#include "editor_cfg.h"
#include <maps/wikimap/mapspro/libs/controller/include/async_tasks_support.h>

#include <maps/libs/auth/include/tvm.h>
#include <maps/libs/pgpool/include/pgpool3.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/common/tskv_logger.h>

#include <memory>

/**
* Provides retrieving of parameters from config file
* @param filename   -    config file
*/

//forward declarations

namespace maps {
namespace wiki {

class ModerationLogger;
class Monitoring;
class RateLimiter;
class RendererPool;
class RevisionMetaUpdatersManager;
class SuspiciousUsersTableCleaner;
class SpravTasksUpdater;
class DeletedUsersCacheUpdater;
class ThreadPool;
class ThreadPools;

namespace acl {
class DeletedUsersCache;
}

namespace acl_utils {
class UserGroupsUidsCache;
}

namespace social {
struct ModerationTimeIntervals;
}

namespace validator {
class ModuleInfo;
}

namespace approved_commits {
class Updater;
}

/**
 * Configuration parameters for dynamic labeler
 */
class RendererParams
{
public:
    explicit RendererParams(const common::ExtendedXmlDoc& doc);

    /**
     *@return path to map configuration
     */
    const std::string& mapPath()const;
    /**
     *@return true if labeling enabled
    */
    bool isLabelerEnabled()const;
    /**
    *@todo must be configured
    */
    TZoom maxRendererZoom()const
        { return 23; }

    const std::string& layersPath() const;

private:
    bool isLabelerEnabled_;
    std::string mapPath_;
    std::string layersPath_;
};

enum class AfterCommitPoolType { Core, View };

enum class LongReadAccess { Try, No };

/**
 * Represents servant configuration parameters
 */
class Config
{
public:
    Config(
        const std::string& filename,
        const std::string& coreSectionName,
        const std::string& poolName,
        const StringSet& allowedExtraPoolSections = {},
        LongReadAccess lrAccess = LongReadAccess::No);

    ~Config();

    log8::Level logLevel() const;

    bool tryLongReadAccess() const { return lrAccess_ == LongReadAccess::Try; }

    /**
     * @returns value of element by XPath
     */
    template <class ResultType>
    ResultType get(const std::string& path) const
    {   return doc_.get<ResultType>(path); }
    /**
     * @returns value of element by XPath
     */
    template <class ResultType>
    ResultType get(const std::string& path, const ResultType& default_value) const
    {   return doc_.get<ResultType>(path, default_value); }
    /**
     * @returns value of element's attribute by XPath and attrname.
     */
    template <class ResultType>
    ResultType getAttr(const std::string& path, const std::string& name) const
    {   return doc_.getAttr<ResultType>(path, name); }
    /**
     * @returns value of element's attribute by XPath and attrname.
     */
    template <class ResultType>
    ResultType getAttr(const std::string& path, const std::string& name,
                       const ResultType& default_value) const
    {   return doc_.getAttr<ResultType>(path, name, default_value); }

    /**
       Rendering suppport params
       @return Renderer configuration params
    */
    const RendererParams& rendererParams() const;

    common::TskvLogger* commentsLogger() const { return commentsLogger_.get(); }

    pgpool3::Pool& poolCore() const;
    pgpool3::Pool& poolSocial() const;
    pgpool3::Pool& poolValidation() const;

    pgpool3::Pool& poolViewTrunk() const;
    pgpool3::Pool& poolViewStable() const;
    pgpool3::Pool& poolView(TBranchId branchId) const;

    pgpool3::Pool& poolLabelsTrunk() const;
    pgpool3::Pool& poolLabelsStable() const;
    pgpool3::Pool& poolLabels(TBranchId branchId) const;

    pgpool3::Pool& pool(const std::string& sectionName) const;

    size_t suggestCount() const { return suggestCount_; }
    bool viewPartsEnabled() const { return viewPartsEnabled_; }

    const EditorCfg* editor() const { return editorCfg_.get(); }

    const std::string& projectName() const { return projectName_; }

    const controller::AsyncTasksSupport& asyncTasksSupport() const;

    const acl::DeletedUsersCache& deletedUsersCache() const { return *deletedUsersCache_; }

    const acl_utils::UserGroupsUidsCache& userGroupsUidsCache() const { return *userGroupsUidsCache_; }

    const RateLimiter& rateLimiter() const;

    ThreadPools& threadPools() const { return *threadPools_; }

    ThreadPool& afterCommitThreadPool(AfterCommitPoolType type);

    RendererPool* rendererPool() const { return rendererPool_.get(); }

    approved_commits::Updater* approvedCommitsUpdater() const
    { return approvedCommitsUpdater_.get(); }

    ModerationLogger* moderationLogger() const
    { return moderationLogger_.get(); }

    const std::list<validator::ModuleInfo>& validatorModules() const
    { return validatorModules_; }

    void onLoad();
    void onUnload();

    void initThreadPools(const std::string& programName);
    void initAfterCommitThreadPools();
    void stopAfterCommitThreadPools();
    void initValidatorModules();

    const std::string& tmpDir() const { return tmpDir_; }

    bool branchMaskCleanEmptyEnabled() const { return branchMaskCleanEmptyEnabled_; }
    bool branchMaskMergeDuplicateEnabled() const { return branchMaskMergeDuplicateEnabled_; }

    const std::string& feedbackApiUrl() const { return feedbackApiUrl_; }

    const social::ModerationTimeIntervals& moderationTimeIntervals() const
    { return *moderationTimeIntervals_.get(); }

    std::string getTvmTicket(const std::string& clientAlias) const;
    std::optional<NTvmAuth::TTvmId> getTvmSrcId(const std::string& ticket) const;

private:
    void initMonitoring();
    void initCommentsLogger();
    void initSuspiciousUsersTableCleaner();
    void initTvmClient();
    void initSpravTaskUpdaters();
    void stopSpravTaskUpdaters();
    void initRenderer();
    void initEditorCfg();
    void initUidsByModerationStatusCache();

    common::ExtendedXmlDoc doc_;
    LongReadAccess lrAccess_;
    log8::Level logLevel_;
    std::wstring rendererConfigPath_;
    std::unique_ptr<RendererParams> rendererParams_;

    size_t suggestCount_;
    bool viewPartsEnabled_;

    bool branchMaskCleanEmptyEnabled_;
    bool branchMaskMergeDuplicateEnabled_;

    /**
    * Database connections pool managers
    */
    typedef std::shared_ptr<common::PoolHolder> PoolHolderPtr;

    std::map<std::string, PoolHolderPtr> poolHolders_;

    std::unique_ptr<EditorCfg> editorCfg_;
    std::string projectName_;

    std::unique_ptr<controller::AsyncTasksSupport> asyncTasksSupport_;
    std::unique_ptr<acl::DeletedUsersCache> deletedUsersCache_;
    std::unique_ptr<RateLimiter> rateLimiter_;

    std::unique_ptr<ThreadPools> threadPools_;

    /// Thread pools to execute afterCommit for observers
    std::map<AfterCommitPoolType, ThreadPool> afterCommitThreadPools_;

    std::list<Monitoring> monitoring_;
    std::unique_ptr<common::TskvLogger> commentsLogger_;

    std::unique_ptr<NTvmAuth::TTvmClient> tvmClient_;
    std::unique_ptr<SpravTasksUpdater> notSubmittedTasksUpdater_;
    std::unique_ptr<SpravTasksUpdater> newTasksUpdater_;
    std::unique_ptr<SpravTasksUpdater> inProgressTasksUpdater_;
    std::unique_ptr<SpravTasksUpdater> acceptedTasksUpdater_;

    std::unique_ptr<SuspiciousUsersTableCleaner> suspiciousUsersTableCleaner_;

    std::unique_ptr<RendererPool> rendererPool_;

    std::unique_ptr<RevisionMetaUpdatersManager> revisionMetaUpdater_;
    std::unique_ptr<approved_commits::Updater> approvedCommitsUpdater_;
    std::unique_ptr<DeletedUsersCacheUpdater> deletedUsersCacheUpdater_;
    std::unique_ptr<ModerationLogger> moderationLogger_;

    std::list<validator::ModuleInfo> validatorModules_;

    std::string tmpDir_;
    std::string feedbackApiUrl_;

    std::unique_ptr<social::ModerationTimeIntervals> moderationTimeIntervals_;

    std::unique_ptr<acl_utils::UserGroupsUidsCache> userGroupsUidsCache_;
};

Config* cfg();

class ConfigScope
{
public:
    ConfigScope(
        const std::string& filename,
        const std::string& coreSectionName,
        const std::string& poolName,
        const StringSet& allowedExtraPoolSections = {},
        LongReadAccess lrAccess = LongReadAccess::No);

    ~ConfigScope();
};

} // namespace wiki
} // namespace maps
