#pragma once

#include <maps/wikimap/mapspro/services/editor/src/common.h>
#include <maps/wikimap/mapspro/services/editor/src/utils.h>

#include <yandex/maps/wiki/configs/editor/size.h>
#include <yandex/maps/wiki/configs/editor/zoom_interval.h>

#include <maps/libs/xml/include/xml.h>

#include <boost/noncopyable.hpp>
#include <boost/ptr_container/ptr_map.hpp>

namespace maps {
namespace wiki {

using namespace configs::editor;

// Forward declarations
class GeneralizationMethod;

typedef std::unique_ptr<GeneralizationMethod> GeneralizationMethodPtr;

class GeneralizationMethod
{
public:
    virtual ~GeneralizationMethod();

    /// Factory method
    static GeneralizationMethodPtr create(const maps::xml3::Node& node);

    /**
    * Implementation should provide visibility zoom interval for the object
    * with in provided zoomRange
    */
    virtual ZoomInterval visibilityInterval(
        const GeoObject* object,
        Transaction& work,
        const ZoomInterval& zoomRange) const = 0;

protected:
    GeneralizationMethod(const maps::xml3::Node& node);
};


class GeomGeneralization
{
public:
    GeomGeneralization(const maps::xml3::Node& node);
    ///  @return tentative visibility interval (to be corrected in GeoObject::objectZoomInterval)
    ZoomInterval geomVisibilityInterval(const GeoObject* object, Transaction& work) const;

private:
    GeneralizationMethodPtr method_;
    ZoomInterval zoomInterval_;
};

// Base class representing generalization strategy
class Generalization : public boost::noncopyable
{
public:
    //  @return calculated object visibility zoom interval.
    ZoomInterval objectVisibility(const GeoObject* object, Transaction& work) const;

    virtual ~Generalization() { }

    Generalization(const maps::xml3::Node& node);

    const std::string& id() const { return id_; }

    TZoom zmin() const { return zoomInterval_.zmin(); }
    TZoom zmax() const { return zoomInterval_.zmax(); }

private:
    std::string id_;
    ZoomInterval zoomInterval_;

    typedef boost::ptr_map<std::string, GeomGeneralization> GeomGeneralizations;
    GeomGeneralizations geomGeneralizations_;
};


// Generalization strategy based on area
class GabaritsGeneralization : public GeneralizationMethod
{
    friend GeneralizationMethodPtr GeneralizationMethod::create(const maps::xml3::Node& node);
private:
    // From GeneralizationMethod
    virtual ZoomInterval visibilityInterval(const GeoObject* object, Transaction& work, const ZoomInterval& zoomRange) const;

    GabaritsGeneralization(const maps::xml3::Node& node);

    Size window_;
    Size minSize_;
    double minFraction_;
};

// Generalization strategy for point objects taking collisions with nearby objects into account.
class CollisionGeneralization : public GeneralizationMethod
{
    friend GeneralizationMethodPtr GeneralizationMethod::create(const maps::xml3::Node& node);
private:
    // From GeneralizationMethod
    virtual ZoomInterval visibilityInterval(const GeoObject* object, Transaction& work, const ZoomInterval& zoomRange) const;

    CollisionGeneralization(const maps::xml3::Node& node);
    TZoom goodZoom(double closestNeighbourDstMeters, const ZoomInterval& zoom) const;

    double radius_;
};

// Generalization strategy based on attribute values.
class FunctionalGeneralization : public GeneralizationMethod
{
    friend GeneralizationMethodPtr GeneralizationMethod::create(const maps::xml3::Node& node);
private:
    // From GeneralizationMethod
    virtual ZoomInterval visibilityInterval(const GeoObject* object, Transaction& work, const ZoomInterval& zoomRange) const;

    FunctionalGeneralization(const maps::xml3::Node& node);

    std::string attributeId_;
    typedef std::map<std::string, ZoomInterval> Items;
    Items items_;
};

// Generalization strategy based on attribute presense.
class AttributesGeneralization : public GeneralizationMethod
{
    friend GeneralizationMethodPtr GeneralizationMethod::create(const maps::xml3::Node& node);
private:
    // From GeneralizationMethod
    virtual ZoomInterval visibilityInterval(const GeoObject* object, Transaction& work, const ZoomInterval& zoomRange) const;

    AttributesGeneralization(const maps::xml3::Node& node);

    std::map<std::string, ZoomInterval> attrToRange_;
};
}//namespace wiki
}//namespace maps
