#include "objects_update_restrictions.h"

#include <maps/wikimap/mapspro/services/editor/src/collection.h>
#include <maps/wikimap/mapspro/services/editor/src/exception.h>

#include <maps/wikimap/mapspro/services/editor/src/objects/areal_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/junction.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/linear_element.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/line_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/model_object.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/point_object.h>
#include <maps/wikimap/mapspro/services/editor/src/factory.h>

#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>

#include <yandex/maps/wiki/configs/editor/categories.h>
#include <maps/libs/xml/include/xml.h>

namespace maps {
namespace wiki {

ObjectsUpdateRestrictions::ObjectsUpdateRestrictions(const maps::xml3::Node& nodeEditor)
{
    auto groupRestrictionNodes = nodeEditor.nodes("objects-update-restrictions/restriction");

    for (size_t i = 0; i < groupRestrictionNodes.size(); ++i) {
        auto node = groupRestrictionNodes[i];
        restrictionsByActionId_.emplace(
            node.attr<std::string>("action-id"),
            ObjectsUpdateActionRestrictions {
                node.node("allow-multiple-categories").value<bool>(),
                node.node("allow-junction").value<bool>(),
                node.node("allow-point").value<bool>(),
                node.node("allow-linear-element").value<bool>(),
                node.node("allow-areal").value<bool>(),
                node.node("allow-model3d").value<bool>(),
                node.node("allow-line-object").value<bool>(),
                node.attr<size_t>("max-objects"),
                {}
            }
        );
    }

    for (auto& [action, restrictions] : restrictionsByActionId_) {
        auto permissionNodes = nodeEditor.nodes(
            "permissions/"
            "permission[@id='mpro']/"
            "permission[@id='tools']/"
            "permission[@id='group-operations']/"
            "permission[@id='" + action + "']/"
            "permission", true);
        for (size_t i = 0; i < permissionNodes.size(); ++i) {
            restrictions.categoryIds.insert(permissionNodes[i].attr<std::string>("id"));
        }
    }

    auto groupAttrPermissionNodes = nodeEditor.nodes(
        "permissions/"
        "permission[@id='mpro']/"
        "permission[@id='tools']/"
        "permission[@id='group-operations']/"
        "permission[@id='update-attributes']/"
        "permission");

    for (size_t i = 0; i < groupAttrPermissionNodes.size(); ++i) {
        const auto& categoryId = groupAttrPermissionNodes[i].attr<std::string>("id");
        groupAttrIdsByCategoryId_.emplace(categoryId, StringSet());

        auto attrNodes = groupAttrPermissionNodes[i].nodes(
            "permission[@id='attributes']/"
            "permission",
            true);
        for (size_t j = 0; j < attrNodes.size(); ++j) {
            groupAttrIdsByCategoryId_.at(categoryId).insert(
                attrNodes[j].attr<std::string>("id"));
        }
    }
}

const StringSet&
ObjectsUpdateRestrictions::actionCategoryIds(const std::string& actionId) const
{
    const auto iter = restrictionsByActionId_.find(actionId);
    REQUIRE(iter != restrictionsByActionId_.end(),
        "Action " << actionId << " restrictions not found.");
    return iter->second.categoryIds;
}

size_t
ObjectsUpdateRestrictions::maxObjects(const std::string& actionId) const
{
    const auto iter = restrictionsByActionId_.find(actionId);
    REQUIRE(iter != restrictionsByActionId_.end(),
        "Action " << actionId << " restrictions not found.");
    return iter->second.maxObjects;
}

void
ObjectsUpdateRestrictions::checkObjects(
    const std::string& actionId,
    const GeoObjectCollection& collection,
    CategoriesCheckPolicy categoriesCheckPolicy) const
{
    const auto iter = restrictionsByActionId_.find(actionId);
    REQUIRE(iter != restrictionsByActionId_.end(),
        "Action " << actionId << " restrictions not found.");
    const auto& actionRestrictions = iter->second;

    WIKI_REQUIRE(collection.size() <= actionRestrictions.maxObjects, ERR_BAD_REQUEST,
        "Too many objects for operation: " << actionId);

    std::string lastSeenCategory;
    for (const auto& object : collection) {
        const auto& categoryId = object->categoryId();
        if (!actionRestrictions.allowMultipleCategories) {
            if (lastSeenCategory.empty()) {
                lastSeenCategory = categoryId;
            } else {
                WIKI_REQUIRE(categoryId == lastSeenCategory, ERR_BAD_REQUEST,
                    "Operation "  << actionId << " doesn't allow category mixing.");
            }
        }

        if (!actionRestrictions.categoryIds.empty() &&
            categoriesCheckPolicy == CategoriesCheckPolicy::ListedOnly)
        {
            WIKI_REQUIRE(categoriesCheckPolicy == CategoriesCheckPolicy::Any ||
                actionRestrictions.categoryIds.contains(categoryId), ERR_BAD_REQUEST,
                "Operation " << actionId << " can't be used on category:" << categoryId);
        } else if (is<Junction>(object)) {
            WIKI_REQUIRE(actionRestrictions.allowJunction, ERR_BAD_REQUEST,
                "Operation " << actionId << " can't be used on junctions.");
        } else if (is<LinearElement>(object)) {
            WIKI_REQUIRE(actionRestrictions.allowLinearElement, ERR_BAD_REQUEST,
                "Operation " << actionId << " can't be used on linear elements.");
        } else if (is<ArealObject>(object)) {
            WIKI_REQUIRE(actionRestrictions.allowAreal, ERR_BAD_REQUEST,
                "Operation " << actionId << " can't be used on areal objects.");
        } else if (is<PointObject>(object)) {
            WIKI_REQUIRE(actionRestrictions.allowPoint, ERR_BAD_REQUEST,
                "Operation " << actionId << " can't be used on point objects.");
        } else if (is<Model3dObject>(object)) {
            WIKI_REQUIRE(actionRestrictions.allowModel3d, ERR_BAD_REQUEST,
                "Operation " << actionId << " can't be used on model objects.");
        } else if (is<LineObject>(object)) {
            WIKI_REQUIRE(actionRestrictions.allowLineObject, ERR_BAD_REQUEST,
                "Operation " << actionId << " can't be used on line objects.");
        } else {
            THROW_WIKI_LOGIC_ERROR(ERR_BAD_REQUEST,
                "Unexpected object type for action: " << actionId << ". Category is: " << categoryId);
        }
    }
}

StringSet
ObjectsUpdateRestrictions::allowedCategoryIds(const std::string& actionId) const
{
    const auto iter = restrictionsByActionId_.find(actionId);
    REQUIRE(iter != restrictionsByActionId_.end(),
        "Action " << actionId << " restrictions not found.");
    const auto& actionRestrictions = iter->second;
    const auto& categories = cfg()->editor()->categories();
    StringSet categoryIds;
    for (const auto& [categoryId, _] : categories) {
        const auto& classInfo = GeoObjectFactory::objectClass(categoryId);
        const auto& className = classInfo.className;
        if (actionRestrictions.allowJunction &&
            className == ObjectsClassInfos::junctionClassInfo.className) {
            categoryIds.insert(categoryId);
            continue;
        }
        if (actionRestrictions.allowLinearElement &&
            className == ObjectsClassInfos::linearElementClassInfo.className) {
            categoryIds.insert(categoryId);
            continue;
        }
        if (actionRestrictions.allowAreal &&
            className == ObjectsClassInfos::arealClassInfo.className) {
            categoryIds.insert(categoryId);
            continue;
        }
        if (actionRestrictions.allowPoint &&
            className == ObjectsClassInfos::pointClassInfo.className) {
            categoryIds.insert(categoryId);
            continue;
        }
        if (actionRestrictions.allowModel3d &&
            className == ObjectsClassInfos::model3dClassInfo.className) {
            categoryIds.insert(categoryId);
            continue;
        }
        if (actionRestrictions.allowLineObject &&
            className == ObjectsClassInfos::lineClassInfo.className) {
            categoryIds.insert(categoryId);
            continue;
        }
    }
    return categoryIds;
}

} // namespace wiki
} // namespace maps
