#include "rate_limiter.h"
#include "config.h"
#include <maps/wikimap/mapspro/services/editor/src/exception.h>
#include <maps/wikimap/mapspro/services/editor/src/moderation.h>
#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>

#include <yandex/maps/wiki/common/rate_counter.h>
#include <yandex/maps/wiki/common/robot.h>
#include <yandex/maps/wiki/social/gateway.h>
#include <maps/wikimap/mapspro/libs/acl/include/deleted_users_cache.h>
#include <maps/wikimap/mapspro/libs/acl/include/restricted_users.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/libs/acl_utils/include/moderation.h>
#include <mutex>

namespace maps::wiki {

namespace {

const std::string COMMENTS = "comments";
const std::string EDITS = "edits";
const std::string RPS = "rps";

const std::string CACHE_SIZE = "cacheSize";

const std::chrono::seconds RPS_LIMITER_CACHE_TIME(5);
const size_t RPS_CHECK_MAX_COUNT_DEFAULT = 10; // 2 rps during 5s

bool
isYandexEmployee(const std::string& moderationStatus)
{
    return
        acl_utils::isYandexModerator(moderationStatus) ||
        acl_utils::isCartographer(moderationStatus) ||
        acl_utils::isRobot(moderationStatus);
}

const std::string&
activityTypeString(social::ActivityType type)
{
    switch (type) {
        case social::ActivityType::Comments:
            return COMMENTS;
        case social::ActivityType::Edits:
            return EDITS;
        case social::ActivityType::FeedbackResolve:
            ASSERT(type != social::ActivityType::FeedbackResolve);
            return s_emptyString;
    }
}

std::optional<std::string> loadUserModerationStatus(social::TUid uid)
{
    auto work = cfg()->poolCore().slaveTransaction();
    acl::ACLGateway aclGw(*work);
    auto user = aclGw.user(uid);
    if (user.status() == acl::User::Status::Deleted) {
        return std::nullopt;
    }
    return acl_utils::moderationStatus(aclGw, user);
}

void blockUser(social::TUid uid, const std::string& reason)
{
    auto work = cfg()->poolCore().masterWriteableTransaction();
    acl::ACLGateway aclGw(*work);
    auto user = aclGw.user(uid);
    if (user.status() != acl::User::Status::Deleted) {
        user.setDeleted(acl::User::DeleteReason::DDOS, common::ROBOT_UID);
        acl::restrictUser(*work, uid, reason);
        work->commit();
    }
}

} // namespace

class RateLimiter::RpsCounters
{
public:
    explicit RpsCounters(size_t cacheSize)
        : rateCounter_(cacheSize, RPS_LIMITER_CACHE_TIME)
    {}

    size_t updateCounter(const std::string& key)
    {
        std::lock_guard<std::mutex> lock(mutex_);
        return rateCounter_.add(key);
    }

private:
    std::mutex mutex_;
    common::RateCounter<std::string> rateCounter_;
};

RateLimiter::RateLimiter(const maps::xml3::Node& node)
    : rateLimiter_(node)
{
    if (node.isNull()) {
        INFO() << "Rate limiter is disabled";
        return;
    }

    auto rpsNode = node.node(RPS, true);
    if (!rpsNode.isNull()) {
        auto cacheSize = rpsNode.attr<size_t>(CACHE_SIZE, 0);
        if (cacheSize) {
            INFO() << "Rate limiter : init rps counters, cache size: " << cacheSize;
            rpsCounters_ = std::make_unique<RpsCounters>(cacheSize);
        }
    }
}

RateLimiter::~RateLimiter() = default;

void
RateLimiter::checkUserActivityAndRestrict(
    const BranchContext& branchCtx,
    social::TUid uid,
    social::ActivityType type,
    const std::string& moderationStatus) const
{
    if (isYandexEmployee(moderationStatus)) {
        return;
    }

    social::Gateway gateway(branchCtx.txnSocial());
    auto interval = rateLimiter_.checkLimitExceeded(gateway, uid, type);
    if (interval) {
        const auto& activityTypeStr = activityTypeString(type);
        acl::restrictUser(
            branchCtx.txnCore(),
            uid,
            activityTypeStr + "." + std::to_string(interval->count()));
        branchCtx.txnCore().commit();
        THROW_WIKI_LOGIC_ERROR(ERR_FORBIDDEN, "Too many " + activityTypeStr);
    }
}

void RateLimiter::checkUserReadActivityDeleteAndRestrict(
    social::TUid uid,
    const std::string& handle,
    const std::string& params) const
{
    if (!rpsCounters_) {
        return;
    }

    const auto key = handle + params;
    auto counter = rpsCounters_->updateCounter(std::to_string(uid) + ":" + key);
    DEBUG() << "Rate limiter : check " << uid << " key: " << key << " counter: " << counter;
    if (counter <= RPS_CHECK_MAX_COUNT_DEFAULT) {
        return;
    }

    auto activityStr = "read editor: " + key;

    auto moderationStatus = loadUserModerationStatus(uid);
    if (!moderationStatus) {
        ERROR() << "Too many " << activityStr << " : user already blocked uid: " << uid << " counter: " << counter;
    } else if (isYandexEmployee(*moderationStatus)) {
        WARN() << "Too many " << activityStr
               << " : yandex uid: " << uid
               << " (" << *moderationStatus << ") counter: " << counter;
        return;
    } else {
        ERROR() << "Too many " << activityStr << " : block user by uid: " << uid << " counter: " << counter;
        blockUser(uid, activityStr);
    }
    cfg()->deletedUsersCache().addUser(uid);
    THROW_WIKI_LOGIC_ERROR(ERR_FORBIDDEN, "Too many " << activityStr);
}

} // namespace maps::wiki
