#pragma once

#include "common.h"
#include "view.h"
#include "object_update_data.h"

#include <boost/utility.hpp>

#include <memory>

namespace maps {
namespace wiki {

class GeoObject;
class ObjectsCache;

struct DiffAlertData {
    uint32_t priority;
    std::string message;
};

using DiffAlertsData = std::vector<DiffAlertData>;

class ObjectEditContext
{
public:
    typedef std::vector<TGeoPoint> SplitPoints;
    typedef std::vector<Geom> SplitLines;

    explicit ObjectEditContext(
            View view,
            bool allowIntersections,
            bool allowInvalidContours,
            SplitPoints splitPoints = {},
            SplitLines splitLines = {},
            DiffAlertsData diffAlertsData = {})
        : view_(std::move(view))
        , splitPoints_(std::move(splitPoints))
        , splitLines_(std::move(splitLines))
        , diffAlertsData_(std::move(diffAlertsData))
        , allowIntersections_(allowIntersections)
        , allowInvalidContours_(allowInvalidContours)
    {}

    static bool defaultAllowIntersections() { return true; }
    static bool defaultAllowInvalidContours() { return true; }

    TZoom viewZoom() const { return view_.zoom(); }
    const TGeoPoint& viewCenter() const { return view_.center(); }

    const SplitPoints& splitPoints() const { return splitPoints_; }
    const SplitLines& splitLines() const { return splitLines_; }
    const DiffAlertsData& diffAlertsData() const { return diffAlertsData_; }

    bool allowIntersections() const { return allowIntersections_; }
    bool allowInvalidContours() const { return allowInvalidContours_; }

private:
    View view_;
    SplitPoints splitPoints_;
    SplitLines splitLines_;
    DiffAlertsData diffAlertsData_;

    bool allowIntersections_;
    bool allowInvalidContours_;
};

typedef std::shared_ptr<ObjectEditContext> ObjectEditContextPtr;
typedef std::map<UniqueId, ObjectEditContextPtr> ObjectsEditContextsPtrMap;

/**
* Class Context represents servant call context
* i e various call parameters
* database connection etc
*/
class Context : public boost::noncopyable
{
public:
    Context(ObjectsCache& cache, TUid user);

    ObjectsCache& cache() const;
    TUid user() const;
    void forceSave(bool f);
    bool forceSave() const;

    bool hasEditContext(UniqueId oid) const;
    const ObjectEditContext* editContext(UniqueId oid) const;
    void addEditContext(UniqueId oid, ObjectEditContextPtr newEditContext);

    const ObjectsEditContextsPtrMap& editContexts() const { return objectsEditContexts_; }

    TZoom maxViewZoom() const;

private:
    ObjectsCache& cache_;
    TUid user_;
    bool forceSave_;

    ObjectsEditContextsPtrMap objectsEditContexts_;
};

} // namespace wiki
} // namespace maps
