#pragma once

#include "common.h"

#include <maps/libs/common/include/exception.h>

#include <util/system/compiler.h>

#include <string>

namespace maps::wiki {

#define DECLARE_ERR_CODE(code) const std::string code = #code

DECLARE_ERR_CODE( INTERNAL_ERROR );
DECLARE_ERR_CODE( ERR_ACTION_NOT_ALLOWED_FOR_CATEGORY );
DECLARE_ERR_CODE( ERR_ADDRESSES_LIMIT_EXCEEDED );
DECLARE_ERR_CODE( ERR_ALREADY_REVERTED_COMMITS );
DECLARE_ERR_CODE( ERR_AMBIGUOUS_OBJECTS );
DECLARE_ERR_CODE( ERR_AMBIGUOUS_RELATIONS_RESOLUTION );
DECLARE_ERR_CODE( ERR_ATTRIBUTE_VALUE_DUPLICATE );
DECLARE_ERR_CODE( ERR_ATTRIBUTE_VALUE_INVALID );
DECLARE_ERR_CODE( ERR_BAD_DATA );
DECLARE_ERR_CODE( ERR_BAD_REQUEST );
DECLARE_ERR_CODE( ERR_BAD_RICH_CONTENT );
DECLARE_ERR_CODE( ERR_BRANCH_ALREADY_FINISHED );
DECLARE_ERR_CODE( ERR_BRANCH_IN_PROGRESS );
DECLARE_ERR_CODE( ERR_BRANCH_LOCKED );
DECLARE_ERR_CODE( ERR_BRANCH_NOT_EXISTS );
DECLARE_ERR_CODE( ERR_BRANCH_READ_ONLY );
DECLARE_ERR_CODE( ERR_BRANCH_UNAVAILABLE );
DECLARE_ERR_CODE( ERR_BUILDINGS_LIMIT_EXCEEDED );
DECLARE_ERR_CODE( ERR_CATEGORY_MISMATCH );
DECLARE_ERR_CODE( ERR_COMMENT_SIZE_LIMIT_EXCEEDED );
DECLARE_ERR_CODE( ERR_CONFLICTING_OPERATION_IN_PROGRESS );
DECLARE_ERR_CODE( ERR_ENTRANCES_LIMIT_EXCEEDED );
DECLARE_ERR_CODE( ERR_FORBIDDEN );
DECLARE_ERR_CODE( ERR_INDOOR_GEOMETRY_OVERLAY_FORBIDDEN );
DECLARE_ERR_CODE( ERR_INDOOR_GEOMETRY_OVERLAY_REQUIRED );
DECLARE_ERR_CODE( ERR_INITIALIZATION );
DECLARE_ERR_CODE( ERR_INVALID_OPERATION );
DECLARE_ERR_CODE( ERR_JUNCTION_ALREADY_EXISTS );
DECLARE_ERR_CODE( ERR_LIMIT_EXCEEDED );
DECLARE_ERR_CODE( ERR_MANDATORY_ATTRIBUTE_MISSING );
DECLARE_ERR_CODE( ERR_MASTER_REQUIRED );
DECLARE_ERR_CODE( ERR_MASTERS_COUNT );
DECLARE_ERR_CODE( ERR_MERGE_WITH_RELATIONS );
DECLARE_ERR_CODE( ERR_MISSING_OBJECT );
DECLARE_ERR_CODE( ERR_MODERATION_CONFLICT );
DECLARE_ERR_CODE( ERR_MULTIPLE_CATEGORIES_NOT_SUPPORTED );
DECLARE_ERR_CODE( ERR_NO_COUNTRY_IN_AD_HIERARCHY );
DECLARE_ERR_CODE( ERR_NO_BUILDINGS );
DECLARE_ERR_CODE( ERR_NO_LANG );
DECLARE_ERR_CODE( ERR_NOT_FOUND );
DECLARE_ERR_CODE( ERR_PARENT_RELATIONS_LOOP );
DECLARE_ERR_CODE( ERR_PROHIBITED_CATEGORY );
DECLARE_ERR_CODE( ERR_RD_CROSSES_AD_BORDER );
DECLARE_ERR_CODE( ERR_RD_OUTSIDE_PARENT );
DECLARE_ERR_CODE( ERR_RELATIONS_TO_MISSING_OBJECTS );
DECLARE_ERR_CODE( ERR_RESULT_IS_NOT_POLYGON );
DECLARE_ERR_CODE( ERR_REVERT_IMPOSSIBLE );
DECLARE_ERR_CODE( ERR_REVERT_NOT_DRAFT_COMMIT );
DECLARE_ERR_CODE( ERR_REVERT_NOT_OWNER );
DECLARE_ERR_CODE( ERR_REVERT_TOO_COMPLEX );
DECLARE_ERR_CODE( ERR_ROUTING_IMPOSSIBLE_BUILD_ROUTE );
DECLARE_ERR_CODE( ERR_ROUTING_IMPOSSIBLE_FIND_PATH );
DECLARE_ERR_CODE( ERR_ROUTING_IMPOSSIBLE_RESTORE_ROUTE );
DECLARE_ERR_CODE( ERR_ROUTING_IMPOSSIBLE_SNAP_STOP );
DECLARE_ERR_CODE( ERR_ROUTING_SEARCH_AREA_TOO_LARGE );
DECLARE_ERR_CODE( ERR_ROUTING_TOO_MANY_ELEMENTS );
DECLARE_ERR_CODE( ERR_ROUTING_UNSUPPORTED_ELEMENT_CATEGORY );
DECLARE_ERR_CODE( ERR_ROUTING_UNSUPPORTED_ROUTE_CATEGORY );
DECLARE_ERR_CODE( ERR_SELF_REFERENCE );
DECLARE_ERR_CODE( ERR_SLAVES_COUNT );
DECLARE_ERR_CODE( ERR_SPORT_TRACK_OUTSIDE_PARENT );
DECLARE_ERR_CODE( ERR_TASK_POINTS_LIMIT_EXCEEDED );
DECLARE_ERR_CODE( ERR_TOPO_AMBIGUOUS_SNAPPING );
DECLARE_ERR_CODE( ERR_TOPO_CLOSED_EDGE_FORBIDDEN );
DECLARE_ERR_CODE( ERR_TOPO_CREATE_INTERSECTION_IS_NOT_PERMITTED );
DECLARE_ERR_CODE( ERR_TOPO_DEGENERATE_EDGE );
DECLARE_ERR_CODE( ERR_TOPO_DEGENERATE_SEGMENT );
DECLARE_ERR_CODE( ERR_TOPO_INTERSECTS_180_MERIDIAN );
DECLARE_ERR_CODE( ERR_TOPO_INVALID_FACE );
DECLARE_ERR_CODE( ERR_TOPO_INVALID_GEOMETRY );
DECLARE_ERR_CODE( ERR_TOPO_INVALID_RESTRICTIONS );
DECLARE_ERR_CODE( ERR_TOPO_INVALID_TOPOLOGY );
DECLARE_ERR_CODE( ERR_TOPO_JC_OVERLAP );
DECLARE_ERR_CODE( ERR_TOPO_MERGE_OF_OVERLAPPED_EDGES_FORBIDDEN );
DECLARE_ERR_CODE( ERR_TOPO_NO_GEOMETRY );
DECLARE_ERR_CODE( ERR_TOPO_NODE_DELETION_FORBIDDEN);
DECLARE_ERR_CODE( ERR_TOPO_NOT_CONTINUOUS );
DECLARE_ERR_CODE( ERR_TOPO_OBJECTS_HAVE_INTERSECTIONS );
DECLARE_ERR_CODE( ERR_TOPO_OBJECTS_HAVE_OVERLAPS );
DECLARE_ERR_CODE( ERR_TOPO_SELF_INTERSECTION );
DECLARE_ERR_CODE( ERR_TOPO_SELF_OVERLAP );
DECLARE_ERR_CODE( ERR_TOPO_SPIKES );
DECLARE_ERR_CODE( ERR_TOPO_TOO_LARGE );
DECLARE_ERR_CODE( ERR_TOPO_TOO_LONG );
DECLARE_ERR_CODE( ERR_TOPO_TOO_LONG_EDGE );
DECLARE_ERR_CODE( ERR_TOPO_TOO_LONG_SEGMENT );
DECLARE_ERR_CODE( ERR_TOPO_TOO_MANY_INTERSECTIONS_WITH_ELEMENT );
DECLARE_ERR_CODE( ERR_TOPO_TOO_MANY_INTERSECTIONS_WITH_NETWORK );
DECLARE_ERR_CODE( ERR_TOPO_TOO_MANY_VERTEXES );
DECLARE_ERR_CODE( ERR_TOPO_TOO_SMALL );
DECLARE_ERR_CODE( ERR_TOPO_UNEXPECTED_INTERSECTION );
DECLARE_ERR_CODE( ERR_TOPO_UNSPECIFIED );
DECLARE_ERR_CODE( ERR_TOPO_UNSUPPORTED_OPERATION );
DECLARE_ERR_CODE( ERR_TOPO_WRONG_GEOMETRY_TYPE );
DECLARE_ERR_CODE( ERR_UNSUPPORTED_FORMAT );
DECLARE_ERR_CODE( ERR_USER_NOT_EXISTS );
DECLARE_ERR_CODE( ERR_VERSION_CONFLICT );
DECLARE_ERR_CODE( ERR_WRONG_CATEGORY_SWITCHING );
DECLARE_ERR_CODE( ERR_WRONG_FORMAT_SETTING );
DECLARE_ERR_CODE( ERR_WRONG_RELATION_ASSOCIATION );

class Exception : public maps::Exception
{
public:
    explicit Exception(const std::string& status)
    {
        attrs().insert(std::make_pair("status", status));
    }
    explicit Exception(const char* status) = delete;

    virtual ~Exception() {}

    virtual const char* type() const { return "WikiMaps Error"; }

    std::string toXml() const = delete;
};

class LogicException: public wiki::Exception
{
public:
    explicit LogicException(const std::string& status) : wiki::Exception(status) {}

    const char* type() const override { return "WikiMaps Logic Error"; }

    virtual const TGeoPoint* location() const { return nullptr; }
};

class LogicExceptionWithLocation: public wiki::LogicException
{
public:
    LogicExceptionWithLocation(const std::string& status, const TGeoPoint& location)
        : wiki::LogicException(status)
        , location_(location)
    {}

    const TGeoPoint* location() const override { return &location_; }

private:
    TGeoPoint location_;
};

class InitializationException: public wiki::Exception
{
public:
    InitializationException() : wiki::Exception(ERR_INITIALIZATION) {}

    const char* type() const override { return "Wiki Editor Initialization Error"; }
};

class InternalErrorException: public wiki::Exception
{
public:
    InternalErrorException() : wiki::Exception(INTERNAL_ERROR) {}

    const char* type() const override { return "Wiki Editor Internal Error"; }
};

class VersionConflictException: public wiki::LogicException
{
public:
    VersionConflictException() : wiki::LogicException(ERR_VERSION_CONFLICT) {}

    const char* type() const override { return "Wiki Editor Version Conflict"; }
};

class UnknownObjectCategoryException: public wiki::InternalErrorException {};


void exceptionToStream(std::ostream& out, const maps::Exception& ex, bool withBacktrace);

} // namespace maps::wiki

#define THROW_WIKI_LOGIC_ERROR(status, cause) \
    throw maps::wiki::LogicException(status) << cause

#define THROW_WIKI_INIT_ERROR(cause) \
    throw maps::wiki::InitializationException() << cause

#define THROW_WIKI_INTERNAL_ERROR(cause) \
    throw maps::wiki::InternalErrorException() << cause

#define WIKI_REQUIRE(cond, status, msg) \
    if (Y_LIKELY(cond)) {} else \
        THROW_WIKI_LOGIC_ERROR(status, msg)

#define WIKI_REQUIRE_WITH_LOCATION(cond, status, msg, point) \
    if (Y_LIKELY(cond)) {} else \
        throw maps::wiki::LogicExceptionWithLocation(status, point) << msg;

#define CHECK_REQUEST_PARAM(param) \
    WIKI_REQUIRE(param, ERR_BAD_REQUEST, "Invalid request parameter: " #param)

#define CHECK_NON_EMPTY_REQUEST_PARAM(param) \
    WIKI_REQUIRE(!(param).empty(), ERR_BAD_REQUEST, "Empty request parameter: " #param)

#define UNSUPPORTED_FORMAT(formatType, taskName) \
    throw maps::wiki::LogicException(ERR_UNSUPPORTED_FORMAT) \
        << formatType << " format not supported for " << taskName
