#include "last_commits.h"

#include "commit.h"

#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/filters.h>

#include <unordered_map>

namespace maps {
namespace wiki {

namespace {

std::unordered_map<TOid, TCommitIds> findHistoryCommits(
    pqxx::transaction_base& work,
    const TOIds& objectIds,
    TCommitId minCommitId,
    TBranchId branchId)
{
    std::unordered_map<TOid, TCommitIds> objectIdToCommitIds;
    if (objectIds.empty()) {
        return objectIdToCommitIds;
    }

    auto branch = revision::BranchManager(work).load(branchId);
    revision::RevisionsGateway gateway(work, branch);
    auto snapshot = gateway.historicalSnapshot(minCommitId, gateway.headCommitId());

    auto notRelationFilter = (
        revision::filters::ObjRevAttr::objectId().in(objectIds) &&
        revision::filters::ObjRevAttr::isNotRelation()
    );
    for (const auto& revisionId: snapshot.revisionIdsByFilter(notRelationFilter)) {
        objectIdToCommitIds[revisionId.objectId()].insert(revisionId.commitId());
    }


    std::vector<std::string> commitAttrs;
    for (const auto& objectId: objectIds) {
        commitAttrs.push_back(objectEditNotesKey(objectId));
        commitAttrs.push_back(primaryObjectKey(objectId));
    };
    auto commitsFilter = (
        revision::filters::CommitAttribute::definedAny(commitAttrs) &&
        revision::filters::CommitAttr::isVisible(branch) &&
        revision::filters::CommitAttr::id() > minCommitId
    );
    auto commits = revision::Commit::load(work, commitsFilter);

    for (const auto& commit : commits) {
        auto primaryObjId = primaryObjectId(commit);
        if (primaryObjId) {
            objectIdToCommitIds[*primaryObjId].insert(commit.id());
        }
        for (const auto& objectId: oidsWithNotes(commit)) {
            objectIdToCommitIds[objectId].insert(commit.id());
        }
    }

    return objectIdToCommitIds;
}


} // namespace

TCommitIds evalLastCommitIds(
    pqxx::transaction_base& work,
    const std::list<revision::Commit>& commits,
    TBranchId branchId)
{
    TCommitId minCommitId = 0;
    TOIds primaryObjectIds;
    for (const auto& commit: commits) {
        auto primaryObjId = primaryObjectId(commit);
        if (primaryObjId) {
            primaryObjectIds.insert(*primaryObjId);
        }

        if (minCommitId == 0) {
            minCommitId = commit.id();
        } else {
            minCommitId = std::min(minCommitId, commit.id());
        }
    }

    auto objectIdToCommitIds = findHistoryCommits(work, primaryObjectIds, minCommitId, branchId);

    TCommitIds lastCommitIds;
    for (const auto& commit: commits) {
        auto primaryObjId = primaryObjectId(commit);
        if (primaryObjId) {
            if (objectIdToCommitIds.count(*primaryObjId)) {

                const auto& commitIds = objectIdToCommitIds[*primaryObjId];
                auto lastCommitId = *commitIds.rbegin();
                if (lastCommitId == commit.id()) {
                    lastCommitIds.insert(commit.id());
                }
            } else {
                lastCommitIds.insert(commit.id());
            }
        } else {
            lastCommitIds.insert(commit.id());
        }
    }
    return lastCommitIds;
}


} // namespace wiki
} // namespace maps
