#include "moderation.h"
#include "acl_role_info.h"
#include "configs/config.h"

#include <maps/libs/common/include/exception.h>
#include <maps/wikimap/mapspro/libs/acl/include/policy.h>
#include <maps/wikimap/mapspro/libs/acl/include/role.h>
#include <yandex/maps/wiki/common/moderation.h>
#include <yandex/maps/wiki/configs/editor/category_groups.h>
#include <yandex/maps/wiki/configs/editor/geoobject_templates.h>

#include <boost/lexical_cast.hpp>
#include <algorithm>

namespace maps::wiki::moderation {

namespace {

const moderation::String2ModerationMode MODERATION_MODE_RESOLVER = {
    {common::MODERATION_STATUS_MODERATOR, social::ModerationMode::Moderator},
    {common::MODERATION_STATUS_YANDEX_MODERATOR, social::ModerationMode::SuperModerator},
    {common::MODERATION_STATUS_CARTOGRAPHER, social::ModerationMode::Supervisor}
};

const int MIN_FC_TO_AUTOAPPROVE = 7;

const std::unordered_set<std::string> RD_EL_AUTOAPPROVE_ATTRS = {
    ATTR_RD_EL_DR,
    ATTR_RD_EL_F_ZLEV,
    ATTR_RD_EL_FOW,
    ATTR_RD_EL_PAVED,
    ATTR_RD_EL_SPEED_LIMIT,
    ATTR_RD_EL_SPEED_LIMIT_T,
    ATTR_RD_EL_SPEED_LIMIT_TRUCK,
    ATTR_RD_EL_SPEED_LIMIT_TRUCK_T,
    ATTR_RD_EL_STRUCT_TYPE,
    ATTR_RD_EL_T_ZLEV,
    ATTR_RD_EL_TOLL,
    ATTR_RD_EL_SPEED_CAT,
    ATTR_SYS_IMPORT_SOURCE
};

const StringSet COND_VIA_ROLES = {
    ROLE_FROM,
    ROLE_LOCATED_AT,
    ROLE_PLACED_AT,
    ROLE_VIA
};

} // namespace

const String2ModerationMode&
getModerationModeResolver()
{
    return MODERATION_MODE_RESOLVER;
}

std::string
toAclRoleName(social::ModerationMode mode)
{
    for (const auto& str2mode : getModerationModeResolver()) {
        if (str2mode.second == mode) {
            return str2mode.first;
        }
    }
    throw RuntimeError()
        << "Unknown moderation mode: " << static_cast<int>(mode);
}

StringSet getAutoModeratedCategoryIds(
    const acl::User& user,
    const TOIds& aoiIds)
{
    if (aoiIds.empty()) {
        return {};
    }

    StringSet categoryGroupIds;
    for (const auto& policy : user.allPolicies()) {
        if (!aoiIds.count(policy.aoiId())) {
            continue;
        }

        moderation::AclRoleInfo roleInfo(policy.role().name());
        if (roleInfo.roleName() == common::MODERATION_STATUS_AUTO_APPROVE_MODERATOR) {
            categoryGroupIds.insert(
                roleInfo.categoryGroupIds().begin(),
                roleInfo.categoryGroupIds().end());
        }
    }

    return cfg()->editor()->categoryGroups().categoryIdsByGroups(categoryGroupIds);
}

bool isAutoApprovable(const ObjectPtr& primaryObj)
{
    const auto& categoryId = primaryObj->categoryId();
    if (categoryId == CATEGORY_RD_JC) {
        auto condViaRange = primaryObj->masterRelations().range(COND_VIA_ROLES);
        if (!condViaRange.empty()) {
            return false;
        }
        auto rdElRange = primaryObj->masterRelations().range(StringSet{ROLE_START, ROLE_END});
        for (const auto& rdEl : rdElRange) {
            if (boost::lexical_cast<int>(rdEl.relative()->attributes().value(ATTR_RD_EL_FC)) <
                MIN_FC_TO_AUTOAPPROVE) {
                    return false;
            }
        }
        return true;
    }
    if (categoryId != CATEGORY_RD_EL) {
        return true;
    }
    auto& attrs = primaryObj->attributes();
    auto fc = boost::lexical_cast<int>(attrs.value(ATTR_RD_EL_FC));
    if (fc < MIN_FC_TO_AUTOAPPROVE || primaryObj->isModifiedLinksToMasters()) {
        return false;
    }
    if (primaryObj->hasExistingOriginal()) {
        auto prevAttrs = primaryObj->original()->attributes();
        auto prevFc = boost::lexical_cast<int>(prevAttrs.value(ATTR_RD_EL_FC));
        if (prevFc < MIN_FC_TO_AUTOAPPROVE || abs(fc - prevFc) > 1) {
            return false;
        }
    }

    auto rdElTemplates =
        cfg()->editor()->externals().geoObjectTemplates().findTemplates(CATEGORY_RD_EL);
    return std::any_of(rdElTemplates.begin(), rdElTemplates.end(),
        [&](const GeoObjectTemplate& template_) {
            for (const auto& attr : attrs) {
                const auto& attrName = attr.name();
                const auto& attrValue = attr.value();
                if (RD_EL_AUTOAPPROVE_ATTRS.count(attrName)) {
                    continue;
                }
                if (template_.attrValues().count(attrName)) {
                    if (attrValue != template_.attrValues().at(attrName)) {
                        return false;
                    }
                } else {
                    if (!attrValue.empty() &&
                        attrValue != cfg()->editor()->attribute(attrName)->defaultValue()) {
                            return false;
                    }
                }
            }
            return true;
        });
}

ResolveLockedByOther
canResolveLockedByOther(bool isUserCartographer)
{
    return
        isUserCartographer
        ? moderation::ResolveLockedByOther::Yes
        : moderation::ResolveLockedByOther::No;
}

} // namespace maps::wiki::moderation
