#include "object_update_data.h"

#include "configs/config.h"

namespace maps {
namespace wiki {

namespace {
const size_t UUID_LENGHT = 36;

void
checkRequiredAttributesPairs(const StringMultiMap& attributesValues)
{
    const auto editorCfg = cfg()->editor();
    for (const auto& attr : attributesValues) {
        if (attr.second.empty() || !editorCfg->isAttributeDefined(attr.first)) {
            continue;
        }
        const auto & requiredAttrId =
            editorCfg->attribute(attr.first)->requiredAttributeId();
        if (requiredAttrId.empty()) {
            continue;
        }
        const auto range = attributesValues.equal_range(requiredAttrId);
        auto pred = [] ( const StringMultiMap::value_type& mapPair)
        {
            return !mapPair.second.empty();
        };
        if (std::find_if(range.first, range.second, pred) == range.second) {
            THROW_WIKI_LOGIC_ERROR(ERR_ATTRIBUTE_VALUE_INVALID,
                " Attribute " << attr.first << " with value: " << attr.second
                << " requires attribute " << requiredAttrId);
        }
    }
}

} // namespace

bool isValidUUID(const std::string& str)
{
    return !str.empty() && str.length() == UUID_LENGHT;
}

ObjectUpdateData::ObjectUpdateData(
        UniqueId id,
        TRevisionId revId,
        std::string categoryId,
        Geom geom,
        AttributesValues attributes,
        TableAttributesValues tableAttributes,
        boost::optional<std::string> richContent,
        RelativesDiffByRoleMap slavesDiff,
        RelativesDiffByRoleMap mastersDiff)
    : id_(id)
    , revisionId_(std::move(revId))
    , categoryId_(std::move(categoryId))
    , geom_(std::move(geom))
    , attributes_(std::move(attributes))
    , tableAttributes_(std::move(tableAttributes))
    , richContent_(std::move(richContent))
    , slavesDiff_(std::move(slavesDiff))
    , mastersDiff_(std::move(mastersDiff))
{
    if (!geom_.isNull()) {
        const auto error = findGeomError(geom_);
        WIKI_REQUIRE_WITH_LOCATION(!error,
            ERR_TOPO_INVALID_GEOMETRY,
            "Input Geometry " << geom_.dump() <<  " is not valid.",
            *error);
        WIKI_REQUIRE(geom_->isSimple(),
            ERR_TOPO_INVALID_GEOMETRY,
            "Input Geometry " << geom_.dump() <<  " is not simple.");
    }
    checkRequiredAttributesPairs(attributes_);
    const auto& tableAttrs = tableAttributes_.attrNames();
    for (const auto& attrId : tableAttrs) {
        const auto& valuesRows = tableAttributes_.find(attrId);
        for (size_t row = 0; row < valuesRows.numRows(); ++row) {
            checkRequiredAttributesPairs(valuesRows.values(row));
        }
    }
}

size_t
ObjectUpdateData::attrValuesCount(const std::string& name) const
{
    return attributes_.count(name);
}

const StringMultiMap&
ObjectUpdateData::attributes() const
{
    return attributes_;
}

const TableAttributesValues&
ObjectUpdateData::tableAttributes() const
{
    return tableAttributes_;
}

void
ObjectUpdateData::setAttrValue(const std::string& id, const std::string& value)
{
    attributes_.erase(id);
    attributes_.insert({id, value});
}

const std::string&
ObjectUpdateData::attrValue(const std::string& name) const
{
    auto it = attributes_.find(name);
    if (it != attributes_.end()) {
        return it->second;
    }
    return s_emptyString;
}

bool
ObjectUpdateData::hasReassignedSlaves() const
{
    for (const auto&  diff : slavesDiff_) {
        if (!diff.second.added().empty() || !diff.second.removed().empty()) {
            return true;
        }
    }
    return false;
}

bool
contains(const RelativesDiffByRoleMap& relativesDiffMap, const UniqueId& id)
{
    for (const auto& diff : relativesDiffMap) {
        if (diff.second.contains(id)) {
            return true;
        }
    }
    return false;
}

ObjectsUpdateDataCollection::ObjectsUpdateDataCollection(const ObjectUpdateData& object)
    : primaryObjectIds_{object.id()}
{
    objects_.push_back(object);
}

const ObjectUpdateData&
ObjectsUpdateDataCollection::operator [] (const UniqueId& id) const
{
    auto it = objects_.findByKey(id);

    REQUIRE(it != objects_.end(), "Object with id " << id << " not found.");

    return *it;
}

ObjectsUpdateDataCollection::Iterator
ObjectsUpdateDataCollection::find(const UniqueId& id) const
{
    return objects_.findByKey(id);
}

void
ObjectsUpdateDataCollection::replaceId(const UniqueId& oldId, const UniqueId& newId)
{
    DEBUG() << "Replacing " << oldId << " with " << newId;
    auto objectData = objects_.extractByKey(oldId);
    REQUIRE(objectData, "Object with id " << oldId <<  " not found.");

    objectData->setId(newId);
    objects_.push_back(std::move(*objectData));

    for (auto& object : objects_) {
        object.replaceIdInRelativesDiff(oldId, newId);
    }

    auto it = primaryObjectIds_.find(oldId);
    if (it != primaryObjectIds_.end()) {
        primaryObjectIds_.erase(it);
        REQUIRE(primaryObjectIds_.insert(newId).second,
            "Object " << newId << " already present in primaries");
    }
}

void
ObjectsUpdateDataCollection::updateObjectRelations(
    const UniqueId& id,
    RelativesDiffByRoleMap slavesDiff,
    RelativesDiffByRoleMap mastersDiff)
{
    auto it = objects_.findByKey(id);
    REQUIRE(it != objects_.end(), "Object " << id << " not found");

    ObjectUpdateData& object = *it;
    object.setRelations(std::move(slavesDiff), std::move(mastersDiff));
}


ObjectIdToParentIdsMap
reversedObjectRelationsGraph(const ObjectsUpdateDataCollection& objects)
{
    ObjectIdToParentIdsMap result;

    auto processRelativesDiffMap =
        [&result, objects] (const UniqueId& objectId, const RelativesDiffByRoleMap& diff)
    {
        auto processRelativesDiff =
            [&result, objects, objectId] (const UniqueIdSet& childrenIds)
            {
                for (const auto& childId : childrenIds) {
                    if (!objects.isPrimary(childId)) {
                        result[childId].insert(objectId);
                    }
                }
            };

        for (const auto& roleDiff : diff) {
            const auto& roleRelativesDiff = roleDiff.second;
            processRelativesDiff(roleRelativesDiff.added());
            processRelativesDiff(roleRelativesDiff.changed());
            processRelativesDiff(roleRelativesDiff.removed());
        }
    };

    for (const auto& object : objects) {
        result.insert({object.id(), {}});
        processRelativesDiffMap(object.id(), object.slavesDiff());
        processRelativesDiffMap(object.id(), object.mastersDiff());
    }

    return result;
}

} // namespace wiki
} // namespace maps
