#pragma once

#include "tablevalues.h"
#include "common.h"
#include "exception.h"

#include <yandex/maps/wiki/configs/editor/unique_list.h>
#include <maps/libs/xml/include/xml.h>
#include <maps/libs/json/include/value.h>

#include <list>
#include <map>

namespace maps {
namespace wiki {

typedef std::string UUID;

bool isValidUUID(const std::string& str);

class UniqueId
{
public:
    UniqueId(const TRevisionId& revisionId, UUID uuid)
    {
        if (revisionId.valid()) {
            objectId_ = revisionId.objectId();
        } else {
            objectId_ = 0;
            REQUIRE(!uuid.empty(), "Empty object id and empty uuid");
        }
        uuid_ = std::move(uuid);
    }

    explicit UniqueId(TOid objectId)
        : objectId_(objectId)
    {
        REQUIRE(objectId_, "Invalid object id");
    }

    TOid objectId() const { return objectId_; }
    const UUID& uuid() const { return uuid_; }

    bool operator < (const UniqueId& otherId) const
    {
        if (objectId_ && otherId.objectId_) {
            return objectId_ < otherId.objectId_;
        }
        if (!objectId_ && !otherId.objectId_) {
            return uuid_ < otherId.uuid_;
        }
        if (objectId_) {
            return false;
        }
        return true;
    }

    bool operator == (const UniqueId& otherId) const
    {
        if (objectId_ && otherId.objectId_) {
            return objectId_ == otherId.objectId_;
        }
        return uuid_ == otherId.uuid_;
    }

    std::string str() const
    {
        return std::to_string(objectId_) + ":" + uuid_;
    }

private:
    TOid objectId_;
    UUID uuid_;
};

typedef std::set<UniqueId> UniqueIdSet;

} // namespace wiki
} // namespace maps

namespace std {

inline std::ostream& operator << (std::ostream& os, const maps::wiki::UniqueId& id)
{
    os << id.str();
    return os;
}

} // namespace std

namespace maps {
namespace wiki {

class RelativesDiff
{
public:
    RelativesDiff() = default;
    RelativesDiff(UniqueIdSet added, UniqueIdSet changed, UniqueIdSet removed)
        : added_(std::move(added))
        , changed_(std::move(changed))
        , removed_(std::move(removed))
    {}

    bool contains(const UniqueId& id) const
    {
        return added_.count(id) || changed_.count(id) || removed_.count(id);
    }

    bool empty() const
    {
        return added_.empty() && changed_.empty() && removed_.empty();
    }

    /// Invalidates references to oldId
    void replaceId(const UniqueId& oldId, const UniqueId& newId)
    {
        auto replacer = [&] (UniqueIdSet& ids) {
            auto it = ids.find(oldId);
            if (it != ids.end()) {
                ids.erase(it);
                REQUIRE(ids.insert(newId).second, "Key " << newId << " already present");
            }
        };

        replacer(added_);
        replacer(changed_);
        replacer(removed_);
    }

    const UniqueIdSet& added() const { return added_; }
    const UniqueIdSet& changed() const { return changed_; }
    const UniqueIdSet& removed() const { return removed_; }

private:
    UniqueIdSet added_;
    UniqueIdSet changed_;
    UniqueIdSet removed_;
};

typedef std::map<std::string, RelativesDiff> RelativesDiffByRoleMap;

bool contains(const RelativesDiffByRoleMap& relativesDiffMap, const UniqueId& id);

inline bool
isEmptyValues(const StringMultiMap& valueMap)
{
    for (const auto& it : valueMap) {
        if (!it.second.empty()) {
            return false;
        }
    }
    return true;
}

class ObjectUpdateData
{
public:
    typedef StringMultiMap AttributesValues;

    ObjectUpdateData() = delete;

    explicit ObjectUpdateData(
        UniqueId id,
        TRevisionId revId = TRevisionId(),
        std::string categoryId = std::string(),
        Geom geom = Geom(),
        AttributesValues attributes = AttributesValues(),
        TableAttributesValues tableAttributes = TableAttributesValues(),
        boost::optional<std::string> richContent = boost::none,
        RelativesDiffByRoleMap slavesDiff = RelativesDiffByRoleMap(),
        RelativesDiffByRoleMap mastersDiff = RelativesDiffByRoleMap());

    operator const UniqueId& () const { return id_; }

    const UniqueId& id() const { return id_; }
    TRevisionId revision() const { return revisionId_; }
    const std::string& categoryId() const { return categoryId_; }

    bool isGeometryDefined() const { return !geom_.isNull(); }
    const Geom& geometry() const { return geom_; }

    const boost::optional<std::string>& richContent() const { return richContent_; }

    const StringMultiMap& attributes() const;
    const TableAttributesValues& tableAttributes() const;
    size_t attrValuesCount(const std::string& name) const;

    /**
    * Change existing attribute value or add new attribute if not exists
    */
    void setAttrValue(const std::string& id, const std::string& value);

    /**
    * Retrieve attribute value if defined or empty string
    */
    const std::string& attrValue(const std::string& name) const;

    const RelativesDiffByRoleMap& slavesDiff() const { return slavesDiff_; }
    const RelativesDiffByRoleMap& mastersDiff() const { return mastersDiff_; }

    bool isEmpty() const
    {
        return !isGeometryDefined()
            && attributes().empty() && tableAttributes().empty()
            && (richContent() && richContent()->empty())
            && slavesDiff().empty()
            && mastersDiff().empty();
    }

    bool isReference() const
    {
        if (!isEmpty()) {
            return false;
        }
        return revision().valid() || categoryId().empty();
    }

    void setId(const UniqueId& newId)
    {
        REQUIRE(newId.objectId(), "Only valid object id can be set, newId " << newId);
        id_ = newId;
    }

    void replaceIdInRelativesDiff(const UniqueId& oldId, const UniqueId& newId)
    {
        auto replacer = [&] (RelativesDiffByRoleMap& diffByRoles) {
            for (auto& roleDiff : diffByRoles) {
                roleDiff.second.replaceId(oldId, newId);
            }
        };

        replacer(slavesDiff_);
        replacer(mastersDiff_);
    }

    void setRelations(
        RelativesDiffByRoleMap slavesDiff, RelativesDiffByRoleMap mastersDiff)
    {
        slavesDiff_ = std::move(slavesDiff);
        mastersDiff_ = std::move(mastersDiff);
    }

    bool hasReassignedSlaves() const;

private:
    UniqueId id_;
    TRevisionId revisionId_;
    std::string categoryId_;

    Geom geom_;
    AttributesValues attributes_;
    TableAttributesValues tableAttributes_;
    boost::optional<std::string> richContent_;

    RelativesDiffByRoleMap slavesDiff_;
    RelativesDiffByRoleMap mastersDiff_;
};

typedef std::map<TOid, const ObjectUpdateData*> ObjectsDataMap;

class ObjectsUpdateDataCollection
{
public:
    typedef UniqueList<ObjectUpdateData, UniqueId> List;
    typedef List::const_iterator Iterator;
    ObjectsUpdateDataCollection() = default;
    explicit ObjectsUpdateDataCollection(const ObjectUpdateData& object);

    template <typename... ObjectConstructorArgs>
    UniqueId add(
        bool isPrimary,
        const TRevisionId& revisionId,
        UUID uuid,
        ObjectConstructorArgs... args);

    /**
     * Sets relatives diffs for object identified by id
     *   and recomputes primary objects set with regards to the updated relations.
     */
    void updateObjectRelations(
        const UniqueId& id,
        RelativesDiffByRoleMap slavesDiff,
        RelativesDiffByRoleMap mastersDiff);

    const ObjectUpdateData& operator [] (const UniqueId& id) const;
    Iterator find(const UniqueId& id) const;

    Iterator begin() const { return objects_.begin(); }
    Iterator end() const { return objects_.end(); }

    bool empty () const { return objects_.empty(); }
    size_t size() const { return objects_.size(); }

    bool isPrimary(const UniqueId& id) const { return primaryObjectIds_.count(id); }

    boost::optional<UniqueId> primaryUniqueId() const
    {
        return
            primaryObjectIds_.size() == 1
            ? boost::optional<UniqueId>(*primaryObjectIds_.begin())
            : boost::none;
    };

    /**
     * Replaces oldId with newId wherever it is mentioned: as object id
     *   or as relative id in relations of all objects in collection.
     */
    void replaceId(const UniqueId& oldId, const UniqueId& newId);

    bool isMultisave() const { return multisave_; }
    void setMultisave() { multisave_ = true; }

private:
    List objects_;

    /**
     * Root objects of diff trees:
     *   those not referenced by any other object in collection.
     */
    UniqueIdSet primaryObjectIds_;

    /**
    * All saved objects are all primary and no relations changed.
    */
    bool multisave_ = false;
};


// dicentra TODO check whether it is reference and not object
// HACK if object is met first time then it is saved
// otherwise it is ignored
template <typename... ObjectConstructorArgs>
UniqueId
ObjectsUpdateDataCollection::add(
    bool isPrimary,
    const TRevisionId& revisionId,
    UUID uuid,
    ObjectConstructorArgs... args)
{
    WIKI_REQUIRE(
        revisionId.valid() || !uuid.empty(),
        ERR_BAD_DATA,
        "Neither valid revision id nor uuid set for object");

    UniqueId id(revisionId, std::move(uuid));

    ObjectUpdateData object(
            id, revisionId, std::forward<ObjectConstructorArgs>(args)...);

    auto it = objects_.findByKey(id);

    if (it != objects_.end()) {
        const ObjectUpdateData& existingData = *it;
        WIKI_REQUIRE(
            existingData.isReference() || object.isReference(),
            ERR_BAD_DATA,
            "Object " << id << " data is met more than once in input");
        if (!existingData.isReference()
            || (existingData.isReference() && object.isReference()))
        {
            return existingData.id();
        }
        *it = std::move(object);
    } else {
        objects_.push_back(std::move(object));
    }

    if (isPrimary) {
        primaryObjectIds_.insert(id);
    }

    return id;
}


typedef std::map<UniqueId, UniqueIdSet> ObjectIdToParentIdsMap;

ObjectIdToParentIdsMap
reversedObjectRelationsGraph(const ObjectsUpdateDataCollection& objects);

} // namespace wiki
} // namespace maps
