#include "renderer.h"

#include <maps/wikimap/mapspro/services/editor/src/common.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/config.h>
#include <maps/wikimap/mapspro/services/editor/src/collection.h>
#include <maps/wikimap/mapspro/services/editor/src/rendererpool.h>
#include <maps/wikimap/mapspro/services/editor/src/objects_cache.h>
#include <maps/wikimap/mapspro/services/editor/src/branch_helpers.h>
#include <maps/wikimap/mapspro/services/editor/src/sync/db_helpers.h>

#include <yandex/maps/wiki/common/pg_utils.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/renderer5/postgres/IPostgresTransactionProvider.h>

#include <thread>

namespace maps {
namespace wiki {

namespace {
const size_t PLACE_LABELS_BEFORE_COMMIT_LIMIT = 1;
}

class RendererObserver::RendererContextData : public Observer::ContextData
{
public:
    virtual ~RendererContextData() {}

    TOIds ids;
};

//------------------------------------------------------------------------

RendererObserver::RendererObserver()
    : labeledCategories_(cfg()->rendererPool()->labeledCategories())
{}

//------------------------------------------------------------------------

Observer::ContextDataPtr
RendererObserver::beforeCommit(
    ObjectsCache& cache,
    const GeoObjectCollection&,
    UserContext& /*userContext*/,
    const CommitContext& /*commitContext*/) const
{
    auto deleteLabelsFilter = [] (const GeoObject* object) -> bool
    {
        return object->original() &&
            (object->isUpdatePropertiesRequested() || object->isModified()) &&
            !object->isPrivate();
    };
    auto placeLabelsFilter = [] (const GeoObject* object) -> bool
    {
        return !object->isDeleted() &&
            (object->isUpdatePropertiesRequested() || object->isModified()) &&
            !object->isPrivate();
    };
    auto idsToDelete = computeDeletedLabelIds(cache, deleteLabelsFilter);
    auto& branchCtx = cache.branchContext();
    deleteLabels(idsToDelete, branchCtx.txnLabels());

    auto ids = computePlacedLabelIds(cache, placeLabelsFilter);
    if (ids.size() > PLACE_LABELS_BEFORE_COMMIT_LIMIT) {
        auto context = make_unique<RendererContextData>();
        context->ids.swap(ids);
        return std::move(context);
    }

    if (!ids.empty()) {
        placeLabels(ids, branchCtx.txnView(), branchCtx.txnLabels(), branchCtx.branch.id());
    }
    return {};
}

//------------------------------------------------------------------------

void
RendererObserver::afterCommit(
        TBranchId branchId,
        Observer::ContextData& data) const
{
    const RendererContextData* rendererData =
        dynamic_cast<const RendererContextData*>(&data);
    REQUIRE(nullptr != rendererData, BOOST_CURRENT_FUNCTION << ": wrong afterCommit context type");

    auto workView = BranchContextFacade::acquireWorkWriteViewOnly(branchId);
    auto workLabels = BranchContextFacade::acquireWorkWriteLabelsOnly(branchId);
    placeLabels(rendererData->ids, *workView, *workLabels, branchId);
    workLabels->commit();
}

//------------------------------------------------------------------------

bool
RendererObserver::isNeedLabel(const GeoObject* object) const
{
    if (object->geom().isNull()) {
        return false;
    }

    const std::string& cat_id = object->categoryId(); // fix for multi categories
    return labeledCategories_.find(cat_id) != labeledCategories_.end();
}

//------------------------------------------------------------------------


TOIds
RendererObserver::computeDeletedLabelIds(ObjectsCache& cache,
    ObjectPredicate filter) const
{
    auto completeFilter = [&] (const GeoObject* object) -> bool
    {
        return isNeedLabel(object) && filter(object);
    };

    GeoObjectCollection collection = cache.find(completeFilter);
    DEBUG() << BOOST_CURRENT_FUNCTION << " GeoCollection size:" << collection.size();

    TOIds ids;
    for (const auto& objectPtr : collection) {
        ids.insert(objectPtr->id());
    }
    return ids;
}

//------------------------------------------------------------------------

void
RendererObserver::deleteLabels(const TOIds& ids, Transaction& workLabels)
{
    if (ids.empty()) {
        return;
    }

    DEBUG() << BOOST_CURRENT_FUNCTION << " DELETE labels count: "
            << ids.size() << " for ids: " << common::join(ids, ',');

    auto condition = common::whereClause("objectid", ids);
    workLabels.exec(
        "SELECT id FROM labels"
        "  WHERE " + condition + " ORDER BY 1 FOR UPDATE;\n"
        "DELETE FROM labels WHERE " + condition
    );
}

//------------------------------------------------------------------------

TOIds
RendererObserver::computePlacedLabelIds(ObjectsCache& cache,
    ObjectPredicate filter) const
{
    auto completeFilter = [&] (const GeoObject* object) -> bool
    {
        return isNeedLabel(object) && filter(object) && !object->renderLabel().empty();
    };

    GeoObjectCollection collection = cache.find(completeFilter);
    DEBUG() << BOOST_CURRENT_FUNCTION << " GeoCollection size:" << collection.size();

    TOIds ids;
    for (const auto& objectPtr : collection) {
        ids.insert(objectPtr->id());
    }
    return ids;
}

//------------------------------------------------------------------------

namespace {

using namespace renderer5::postgres;

class TransactionHolder: public IPQXXTransactionHolder
{
public:
    explicit TransactionHolder(Transaction& work) : work_(work) {}

    PQXXTransaction& get() override { return work_; }

private:
    Transaction& work_;
};

class TransactionProvider : public IPostgresTransactionProvider
{
public:
    explicit TransactionProvider(PQXXTransactionHolderPtr holder)
        : holder_(std::move(holder))
    {}

    PQXXTransactionHolderPtr getTransaction() override { return holder_; }

private:
    PQXXTransactionHolderPtr holder_;
};

class TransactionProviderEx: public PostgresTransactionProviderEx
{
public:
    TransactionProviderEx(
            Transaction& workView,
            Transaction& workLabels,
            TBranchId branchId)
        : providerView_(new TransactionProvider(PQXXTransactionHolderPtr(new TransactionHolder(workView))))
        , providerLabels_(new TransactionProvider(PQXXTransactionHolderPtr(new TransactionHolder(workLabels))))
    {
        setQueryNamePrefix(std::to_string(branchId));
        setObjectsTransaction(providerView_);
        setLabelsTransaction(providerLabels_);
    }

    PQXXTransactionHolderPtr getTransaction() override
    {
        throw maps::RuntimeError("Wrong using transaction provider");
    }

private:
    PostgresTransactionProviderPtr providerView_;
    PostgresTransactionProviderPtr providerLabels_;
};

class OperationProgress : public tilerenderer4::IOperationProgress
{
public:
    explicit OperationProgress(const std::string& logPrefix)
        : logPrefix_(logPrefix)
    {}

    bool cancelled() override { return false; }

    void progress(double percentage) override
    {
        INFO() << logPrefix_ << ' ' << percentage << '%';
    }

private:
    const std::string logPrefix_;
};

} // namespace

bool
RendererObserver::placeLabels(
    const TOIds& ids, Transaction& workView, Transaction& workLabels, TBranchId branchId)
{
    if (ids.empty()) {
        return true;
    }

    auto pool = cfg()->rendererPool();
    RendererPool::HolderPtr pHolder = pool->acquire();
    renderer5::postgres::PostgresTransactionProviderExPtr transactionProvider(
        new TransactionProviderEx(workView, workLabels, branchId));

    bool success = true;
    try {
        pHolder->renderer()->placeLabels({ids.begin(), ids.end()}, transactionProvider);
    } catch (const std::exception& ex) {
        WARN() << "Renderer exception: " << ex.what();
        success = false;
    }
    DEBUG() << BOOST_CURRENT_FUNCTION << pool->stat();
    return success;
}

void
RendererObserver::placeAllLabels(
    Transaction& workView, Transaction& workLabels, TBranchId branchId)
{
    auto taskPtr = tilerenderer4::BulkLabelTask::create();
    taskPtr->maxThreadsToUse = std::thread::hardware_concurrency();

    OperationProgress progressIndicator(
        "Place all labels for branchId " + std::to_string(branchId));

    auto pool = cfg()->rendererPool();
    RendererPool::HolderPtr pHolder = pool->acquire();
    renderer5::postgres::PostgresTransactionProviderExPtr transactionProvider(
        new TransactionProviderEx(workView, workLabels, branchId));

    pHolder->renderer()->placeLabels(
        progressIndicator,
        *taskPtr,
        transactionProvider);
}

} // namespace wiki
} // namespace maps
