#include "trust_level_calculator.h"
#include "envelope_info.h"
#include "diffalert_runner_proxy.h"

#include <maps/wikimap/mapspro/services/editor/src/acl_utils.h>
#include <maps/wikimap/mapspro/services/editor/src/objects/junction.h>

#include <yandex/maps/wiki/configs/editor/categories.h>

namespace maps::wiki {

TrustLevelCalculator::TrustLevelCalculator(
    const BranchContext& branchCtx,
    const acl::User& user,
    const TOIds& aoiIds,
    const EnvelopeInfo& envelopeInfo,
    const revision::Commit& commit,
    DiffAlertRunnerProxy& diffAlertRunner)
    : branchCtx_(branchCtx)
    , user_(user)
    , aoiIds_(aoiIds)
    , envelopeInfo_(envelopeInfo)
    , commit_(commit)
    , diffAlertRunner_(diffAlertRunner)
{

}

moderation::TrustLevel
TrustLevelCalculator::calculate(const GeoObjectCollection& collection)
{
    const auto primaryObj = collection.primaryObject();
    auto trustLevel = calculateInternal(primaryObj);
    if (!primaryObj && trustLevel != moderation::TrustLevel::SkipModeration) {
        bool canSkipModeration = true;
        for (const auto& collection : collection) {
            if (moderation::TrustLevel::SkipModeration !=
                calculateInternal(collection))
            {
                canSkipModeration = false;
                break;
            }
        }
        if (canSkipModeration) {
            trustLevel = moderation::TrustLevel::SkipModeration;
        }
    }
    return trustLevel;
}

moderation::TrustLevel
TrustLevelCalculator::calculateInternal(const ObjectPtr& primaryObj)
{
    if (!commit_.inTrunk()) {
        return moderation::TrustLevel::SkipModeration;
    }

    if (user_.status() != acl::User::Status::Active) {
        return moderation::TrustLevel::NotTrusted;
    }

    if (primaryObj && is<Junction>(primaryObj) &&
        primaryObj->isDeleted() &&
        diffAlertRunner_.messages().empty() &&
        moderation::isAutoApprovable(primaryObj))
    {
        return moderation::TrustLevel::SkipModeration;
    }

    auto commitTrustLevel = moderation::TrustLevel::NotTrusted;
    boost::optional<bool> isAutoApprovableObject;
    if (!userPolicies_) {
        userPolicies_ = user_.allPolicies();
    }
    for (const auto& policy : *userPolicies_) {
        const auto aoiId = policy.aoiId();
        if (aoiId && !aoiIds_.contains(aoiId)) {
            continue;
        }
        const auto roleInfo = getRoleInfo(policy);
        if (roleInfo.trustLevel() == moderation::TrustLevel::NotTrusted) {
            continue;
        }

        const bool canProcessPrimary =
            primaryObj &&
            roleInfo.canProcessCategory(primaryObj->categoryId());
        if (roleInfo.trustLevel() == moderation::TrustLevel::SkipModeration &&
            (roleInfo.canProcessAllGroups() || canProcessPrimary))
        {
            return moderation::TrustLevel::SkipModeration;
        }
        if (roleInfo.canProcessAllGroups() ||
            !primaryObj && roleInfo.hasAccessToCommonTasks())
        {
            commitTrustLevel = moderation::TrustLevel::AcceptAndModerate;
        }
        if (canProcessPrimary) {
            if (roleInfo.roleName() == common::MODERATION_STATUS_AUTO_APPROVE_EDIT) {
                if (!isAutoApprovableObject) {
                    isAutoApprovableObject = moderation::isAutoApprovable(primaryObj);
                }
                if (*isAutoApprovableObject && diffAlertRunner_.messages().empty()) {
                    return moderation::TrustLevel::SkipModeration;
                }
            }
            commitTrustLevel = moderation::TrustLevel::AcceptAndModerate;
        }
    }
    if (commitTrustLevel == moderation::TrustLevel::AcceptAndModerate &&
        primaryObj && primaryObj->category().autoApprove() &&
        !isCommitInImportantRegion(branchCtx_, primaryObj->categoryId(),
            envelopeInfo_.geomDiffObjects(), envelopeInfo_.envelope()) &&
        !isCommitWithBigArea(primaryObj->categoryId(), diffAlertRunner_.messages())) {
            return moderation::TrustLevel::SkipModeration;
    }
    return commitTrustLevel;
}

const moderation::AclRoleInfo&
TrustLevelCalculator::getRoleInfo(const acl::Policy& policy) {
    const auto roleName = policy.role().name();
    const auto roleInfoIt = roleInfos_.find(roleName);
    if (roleInfoIt != roleInfos_.end()) {
        return roleInfoIt->second;
    }
    return roleInfos_.try_emplace(roleName, roleName).first->second;
}

} // namespace maps::wiki
