#include "polyline_sticker.h"

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/distance.h>

namespace maps::wiki::polyline_sticker {

namespace {

void
removeSequentialDuplicates(std::vector<size_t>& lineString)
{
    if (lineString.size() < 2) {
        return;
    }
    size_t i = 0;
    for (const auto x : lineString) {
        if (x != lineString[i]) {
            lineString[++i] = x;
        }
    }
    lineString.resize(i + 1);
}

} // namespace

PolylineSticker::PolylineSticker(
    double vertexToVertex,
    double vertexToEdge,
    std::optional<double> endpointToEndpoint)
    : vertexToVertex_(vertexToVertex)
    , vertexToEdge_(vertexToEdge)
    , endpointToEndpoint_(endpointToEndpoint ? *endpointToEndpoint : vertexToVertex)
{
}

void
PolylineSticker::addPolyline(const IPolyline* polyline, bool fixed)
{
    ASSERT(!modified_);
    MeshPolyline meshPoly {
        .fixed = fixed,
        .closed = polyline->isClosed()
    };
    for (size_t lineString = 0; lineString < polyline->numLineStrings(); ++lineString) {
        std::vector<size_t> lineStringIdx;
        bool lastVertexIsSame = (
            polyline->lineStringVertex(lineString, 0) ==
            polyline->lineStringVertex(lineString, polyline->numLineStringVertexes(lineString) - 1));
        ASSERT(lastVertexIsSame || !polyline->isClosed());
        size_t stopVertexNum = polyline->numLineStringVertexes(lineString) -
            ((lastVertexIsSame && polyline->isClosed()) ? 1 : 0);
        for (size_t v = 0; v < stopVertexNum; ++v)
        {
            lineStringIdx.push_back(addVertex(polyline->lineStringVertex(lineString, v), fixed));
        }
        if (polyline->isClosed()) {
            lineStringIdx.push_back(lineStringIdx.front());
        } else {
            vertexes_[lineStringIdx.front()].endpoint = true;
            vertexes_[lineStringIdx.back()].endpoint = true;
        }
        meshPoly.lineStringVertexesIdx.push_back(lineStringIdx);
    }
    polylines_.emplace(polyline->id(), meshPoly);
}

bool
PolylineSticker::processMesh()
{
    ASSERT(!modified_);
    bool repeat = true;
    while (repeat) {
        repeat = false;
        // Collapse vertexes
        for (size_t v = 0 ; v < vertexes_.size(); ++v) {
            if (vertexes_[v].fixed || vertexes_[v].unused) {
                continue;
            }
            for (size_t vo = 0; vo < vertexes_.size(); ++vo) {
                if (v == vo || vertexes_[vo].unused) {
                    continue;
                }
                const auto vertexTolerance =
                    vertexes_[vo].endpoint && vertexes_[v].endpoint
                        ? endpointToEndpoint_
                        : vertexToVertex_;
                if (geolib3::distance(vertexes_[vo].coord, vertexes_[v].coord) < vertexTolerance) {
                    repeat = true;
                    replaceVertex(v, vo);
                }
            }
        }
        // Insert vertexes to edges
        for (auto& [id, poly] : polylines_) {
            for (auto& lineString : poly.lineStringVertexesIdx) {
                for (auto edgeB = lineString.begin(); edgeB != lineString.end();) {
                    auto edgeE = edgeB;
                    ++edgeE;
                    if (edgeE == lineString.end()) {
                        break;
                    }

                    auto closestVertex = findClosestVertexOnEdge(poly, *edgeB, *edgeE);
                    if (closestVertex) {
                        repeat = true;
                        if (!vertexes_[*closestVertex].fixed) {
                            moveVertexToEdge(*edgeB, *edgeE, *closestVertex);
                        }
                        modified_ = true;
                        lineString.insert(edgeE, *closestVertex);
                        edgeB = lineString.begin();
                    } else {
                         ++edgeB;
                    }
                }
            }
        }
    }
    if (modified_) {
        for (auto& [_, p] : polylines_) {
            for (auto& lineString : p.lineStringVertexesIdx) {
                removeSequentialDuplicates(lineString);
            }
        }
    }
    return modified_;
}

size_t
PolylineSticker::addVertex(const geolib3::Point2& coord, bool fixed)
{
    auto value = vertexes_.size();
    vertexes_.emplace_back(
        MeshVertex {
            .fixed = fixed,
            .coord = coord
        }
    );
    return value;
}

void
PolylineSticker::replaceVertex(size_t v, size_t vo)
{
    vertexes_[v].unused = true;
    modified_ = true;
    for (auto& [_, p] : polylines_) {
        for (auto& lineString : p.lineStringVertexesIdx) {
            for (auto& lineStringVertex : lineString) {
                if (lineStringVertex == v) {
                    lineStringVertex =  vo;
                }
            }
        }
    }
}

namespace {
bool areSameLineStringVertexes(
    const std::vector<std::vector<size_t>>& lineStringVertexesIdx,
    size_t v1, size_t v2, size_t v3)
{
    for (const auto& lineStringIdx : lineStringVertexesIdx) {
        if (std::find(lineStringIdx.begin(), lineStringIdx.end(), v1) != lineStringIdx.end() &&
            std::find(lineStringIdx.begin(), lineStringIdx.end(), v2) != lineStringIdx.end() &&
            std::find(lineStringIdx.begin(), lineStringIdx.end(), v3) != lineStringIdx.end())
        {
            return true;
        }
    }
    return false;
}
} // namespace

std::optional<size_t>
PolylineSticker::findClosestVertexOnEdge(const MeshPolyline& edgeOwner, size_t vB, size_t vE) const
{
    geolib3::Segment2 edge(vertexes_[vB].coord, vertexes_[vE].coord);
    std::optional<size_t> resultVertex;
    std::optional<double> minDistance;
    bool vBAndVeFixed = vertexes_[vB].fixed && vertexes_[vE].fixed;
    for (size_t v = 0; v < vertexes_.size(); ++v) {
        if (v == vB || v == vE || vertexes_[v].unused) {
            continue;
        }
        if (vertexes_[v].fixed && vBAndVeFixed) {
            continue;
        }
        if (geolib3::distance(vertexes_[vB].coord, vertexes_[v].coord) < vertexToVertex_ ||
            geolib3::distance(vertexes_[vE].coord, vertexes_[v].coord) < vertexToVertex_)
        {
            continue;
        }
        const auto dist = geolib3::distance(edge, vertexes_[v].coord);
        if (dist > vertexToEdge_ || (minDistance && dist > *minDistance)) {
            continue;
        }
        if (areSameLineStringVertexes(edgeOwner.lineStringVertexesIdx, v, vB, vE)) {
            continue;
        }
        resultVertex = v;
        minDistance = dist;
    }
    return resultVertex;
}

void
PolylineSticker::moveVertexToEdge(size_t vB, size_t vE, size_t v)
{
    modified_ = true;
    const geolib3::Segment2 edge(vertexes_[vB].coord, vertexes_[vE].coord);
    vertexes_[v].coord = geolib3::closestPoint(edge, vertexes_[v].coord);
}

std::optional<ResultPolyline>
PolylineSticker::resultPolyline(ID id) const
{
    auto it = polylines_.find(id);
    if (it == polylines_.end()) {
        return std::nullopt;
    }
    std::vector<std::vector<geolib3::Point2>> lineStringsVertexes;
    for (const auto& idxLineString : it->second.lineStringVertexesIdx) {
        std::vector<geolib3::Point2> lineStringVertexes;
        if ((it->second.closed && idxLineString.size() < 4) ||
            idxLineString.size() < 2) {
            return std::nullopt;
        }
        lineStringVertexes.reserve(idxLineString.size());
        for (const auto& vIdx : idxLineString) {
            ASSERT(!vertexes_[vIdx].unused);
            lineStringVertexes.emplace_back(vertexes_[vIdx].coord);
        }
        lineStringsVertexes.emplace_back(std::move(lineStringVertexes));
    }
    return ResultPolyline(id, lineStringsVertexes, it->second.closed);
}
} // maps::wiki::polyline_sticker
