#pragma once

#include <maps/libs/geolib/include/closest_point.h>
#include <maps/libs/geolib/include/point.h>
#include <map>

namespace maps::wiki::polyline_sticker {
using ID = uint64_t;

class IPolyline
{
public:
    virtual ~IPolyline() {};
    virtual ID id() const = 0;
    virtual size_t numLineStrings() const = 0;
    virtual size_t numLineStringVertexes(size_t lineString) const = 0;
    virtual geolib3::Point2 lineStringVertex(size_t lineString, size_t v) const = 0;
    virtual bool isClosed() const = 0;
};

class ResultPolyline final : public IPolyline
{
public:
    ID id() const override { return id_; }
    virtual size_t numLineStrings() const override { return lineStringsVertexes_.size(); }
    virtual size_t numLineStringVertexes(size_t lineString) const override { return lineStringsVertexes_[lineString].size(); }
    virtual geolib3::Point2 lineStringVertex(size_t lineString, size_t v) const override
    { return lineStringsVertexes_[lineString][v]; }
    virtual bool isClosed() const override { return closed_; }

private:
    friend class PolylineSticker;
    ResultPolyline(ID id, std::vector<std::vector<geolib3::Point2>> lineStringsVertexes, bool closed)
        : id_(id)
        , lineStringsVertexes_(std::move(lineStringsVertexes))
        , closed_(closed)
    {
    }
    ID id_;
    std::vector<std::vector<geolib3::Point2>> lineStringsVertexes_;
    bool closed_;
};

class PolylineSticker
{
private:
    struct MeshVertex
    {
        bool fixed = false;
        bool unused = false;
        bool endpoint = false;
        geolib3::Point2 coord;
    };
    using MeshVertexes = std::vector<MeshVertex>;

    struct MeshPolyline
    {
        std::vector<std::vector<size_t>> lineStringVertexesIdx;
        bool fixed = false;
        bool closed = true;
    };
    using MeshPolylines = std::map<ID, MeshPolyline>;

public:
    PolylineSticker(
        double vertexToVertex,
        double vertexToEdge,
        std::optional<double> endpointToEndpoint = std::nullopt);
    void addPolyline(const IPolyline* polyline, bool fixed);
    bool processMesh();
    std::optional<ResultPolyline> resultPolyline(ID id) const;

private:
    size_t addVertex(const geolib3::Point2&, bool fixed);
    void replaceVertex(size_t v, size_t vo);
    std::optional<size_t> findClosestVertexOnEdge(const MeshPolyline& edgeOwner, size_t vB, size_t vE) const;
    void moveVertexToEdge(size_t vB, size_t vE, size_t v);

    const double vertexToVertex_;
    const double vertexToEdge_;
    const double endpointToEndpoint_;

    MeshVertexes vertexes_;
    MeshPolylines polylines_;
    bool modified_ = false;
};

} // namespace maps::wiki::polyline_sticker
