#include "relations_manager.h"

#include "objects/object.h"
#include "objects/relation_object.h"
#include "collection.h"
#include "objects_cache.h"
#include "revisions_facade.h"
#include "factory.h"
#include "configs/config.h"

#include <yandex/maps/wiki/common/batch.h>
#include <yandex/maps/wiki/configs/editor/categories.h>

namespace maps {
namespace wiki {

namespace {

const size_t OBJECT_IDS_BULK_SIZE = 1000;

const std::vector<const RelationObject*> EMPTY;

} // namespace

RelationsManager::RelationsManager(ObjectsCache& cache)
    : cache_(cache)
{}

void
RelationsManager::loadRelations(RelationType type, const TOIds& ids)
{
    GeoObjectCollection objects = cache_.get(ids);
    TOIds toLoad;
    for (const auto& obj : objects) {
        if (!obj->relations(type).areAllRolesLoaded()) {
            toLoad.insert(obj->id());
        }
    }
    if (toLoad.empty()) {
        return;
    }
    // no runtime data for relations
    GeoObjectCollection relations = cache_.revisionsFacade().loadRelations(type, toLoad);
    processRelations(type, relations);

    for (auto& obj : objects) {
        auto relationsRefContainer = obj->relationsHolderByType(type).relations();
        for (auto& roleRefContainer : relationsRefContainer) {
            roleRefContainer.links->loaded = true;
        }
    }
}


const std::vector<const RelationObject*>&
RelationsManager::cachedRelations(RelationType type, TOid id) const
{
    const auto& relations =
        type == RelationType::Master
        ? cachedMasterRelations_
        : cachedSlaveRelations_;

    auto it = relations.find(id);
    if (it != relations.end()) {
        return it->second;
    }
    return EMPTY;
}

void
RelationsManager::loadRelations(RelationType type, const TOIds& ids, const StringSet& roleIds)
{
    GeoObjectCollection objects = cache_.get(ids);
    TOIds toLoad;
    StringSet rolesToLoad;
    for (const auto& obj : objects) {
        const auto& relationsHolder = obj->relationsHolderByType(type);
        StringSet objRolesToLoad = relationsHolder.rolesToLoad(roleIds);
        if (!objRolesToLoad.empty()) {
            toLoad.insert(obj->id());
            rolesToLoad.insert(objRolesToLoad.begin(), objRolesToLoad.end());
        }
    }
    if (toLoad.empty() || rolesToLoad.empty()) {
        return;
    }
    // no runtime data for relation objects
    GeoObjectCollection relations =
        cache_.revisionsFacade().loadRelations(type, toLoad, rolesToLoad);
    processRelations(type, relations);

    for (auto& obj : objects) {
        auto relationsRefContainer = obj->relationsHolderByType(type).relations(rolesToLoad);
        for (auto& roleRefContainer : relationsRefContainer) {
            roleRefContainer.links->loaded = true;
        }
    }
}

void
RelationsManager::loadRelations(
    RelationType type, const std::map<TOid, StringSet>& oidsToRolesMap)
{
    std::map<StringSet, TOIds> oidsByRolesMap;

    for (const auto& [id, roles] : oidsToRolesMap) {
        oidsByRolesMap[roles].insert(id);
    }
    for (const auto& kv : oidsByRolesMap) {
        const auto& roles = kv.first;
        const auto& ids = kv.second;
        common::applyBatchOp(
            ids,
            OBJECT_IDS_BULK_SIZE,
            [&](const auto& batchIds) {
                loadRelations(type, batchIds, roles);
            });
    }
}

void
RelationsManager::loadRelations(RelationType type, TOid id, RelativesLimit limit)
{
    ObjectPtr object = cache_.getExisting(id);
    if (object->relations(type).areAllRolesLoaded()) {
        return;
    }
    ASSERT(limit.type == RelativesLimit::Type::PerRole);
    loadRelationsWithPerRoleLimit(type, *object, limit.value);
}

void
RelationsManager::loadRelations(RelationType type, TOid id, const StringSet& roleIds, RelativesLimit limit)
{
    ObjectPtr object = cache_.getExisting(id);
    StringSet rolesToLoad = object->relationsHolderByType(type).rolesToLoad(roleIds);
    if (rolesToLoad.empty()) {
        return;
    }
    if (limit.type == RelativesLimit::Type::All) {
        loadRelationsWithCumulativeLimit(type, *object, rolesToLoad, limit.value);
    } else {
        ASSERT(limit.type == RelativesLimit::Type::PerRole);
        loadRelationsWithPerRoleLimit(type, *object, rolesToLoad, limit.value);
    }
}

void
RelationsManager::loadRelationsWithCumulativeLimit(
    RelationType type, GeoObject& object, const StringSet& roleIds, size_t limit)
{
    auto relationsContainer =
        cache_.revisionsFacade().loadRelations(type, object.id(), roleIds, limit);
    if (!relationsContainer) {
        return;
    }
    GeoObjectCollection relations = *relationsContainer;
    processRelations(type, relations);
    auto relationsRefContainer = object.relationsHolderByType(type).relations(roleIds);
    for (auto& roleRefContainer : relationsRefContainer) {
        roleRefContainer.links->loaded = true;
    }
}

void
RelationsManager::loadRelationsWithPerRoleLimit(
    RelationType type, GeoObject& object, size_t limit)
{
    StringSet relativeRoleIds = object.category().roleIds(type, roles::filters::All);
    loadRelationsWithPerRoleLimit(type, object, relativeRoleIds, limit);
}

void
RelationsManager::loadRelationsWithPerRoleLimit(
    RelationType type, GeoObject& object, const StringSet& roleIds, size_t limit)
{
    StringSet rolesToLoad = object.relationsHolderByType(type).rolesToLoad(roleIds);
    if (rolesToLoad.empty()) {
        return;
    }
    GeoObjectCollection loadedRelations;
    StringSet loadedRoles;
    for (const auto& roleId : rolesToLoad) {
        auto roleRelations =
            cache_.revisionsFacade().loadRelations(type, object.id(), StringSet{roleId}, limit);
        if (!roleRelations) {
            continue;
        }
        loadedRelations.append(*roleRelations);
        loadedRoles.insert(roleId);
    }
    processRelations(type, loadedRelations);
    auto relationsRefContainer = object.relationsHolderByType(type).relations(loadedRoles);
    for (auto& roleRefContainer : relationsRefContainer) {
        roleRefContainer.links->loaded = true;
    }
}

void
RelationsManager::processRelations(RelationType type, GeoObjectCollection& relations)
{
    GeoObjectCollection unprocessedRelations;
    TOIds relativesIds;
    for (const auto& relObj : relations) {
        RelationObject* relation = as<RelationObject>(relObj);
        if (cachedRelations_.find(relation) == cachedRelations_.end()) {
            relativesIds.insert(type == RelationType::Master
                ? relation->masterId()
                : relation->slaveId());
            unprocessedRelations.add(relObj);
        }
    }
    GeoObjectCollection relatives = cache_.get(relativesIds);
    TOIds danglingRelationIds;
    for (const auto& relObj : unprocessedRelations) {
        RelationObject* relation = as<RelationObject>(relObj);
        auto relativeId = type == RelationType::Master
            ? relation->masterId()
            : relation->slaveId();
        if (!relatives.getById(relativeId).get()) {
            if (cache_.policy().danglingRelationsPolicy == DanglingRelationsPolicy::Check) {
                THROW_WIKI_LOGIC_ERROR(
                    ERR_RELATIONS_TO_MISSING_OBJECTS, "Object id: " << relativeId);
            }
            danglingRelationIds.insert(relation->id());
        }
    }
    for (auto hangingRelationId : danglingRelationIds) {
        unprocessedRelations.remove(hangingRelationId);
    }

    updateRelations(cache_.add(unprocessedRelations));
}

RelationsManager::RelationKey::RelationKey(const RelationObject* obj)
    : masterId(obj->masterId())
    , slaveId(obj->slaveId())
    , roleId(obj->role())
    , masterCategoryId(obj->masterCategoryId())
    , slaveCategoryId(obj->slaveCategoryId())
{}

RelationsManager::RelationKey::RelationKey(
        TOid masterId, TOid slaveId,
        const std::string& roleId,
        const std::string& masterCategoryId,
        const std::string& slaveCategoryId)
    : masterId(masterId)
    , slaveId(slaveId)
    , roleId(roleId)
    , masterCategoryId(masterCategoryId)
    , slaveCategoryId(slaveCategoryId)
{}

bool
RelationsManager::RelationKey::operator < (const RelationKey& other) const
{
    return
        std::tie(masterId, slaveId, roleId,
            masterCategoryId, slaveCategoryId) <
        std::tie(other.masterId, other.slaveId, other.roleId,
            other.masterCategoryId, other.slaveCategoryId);
}

/**
 * Relations are already cached in common map
 */
void
RelationsManager::updateRelations(const GeoObjectCollection& relations)
{
    DEBUG() << BOOST_CURRENT_FUNCTION << " relations count: " << relations.size();
    for (auto& obj : relations) {
        RelationObject* rel = as<RelationObject>(obj);
        RelationKey key(rel);
        ObjectPtr master = cache_.getExisting(rel->masterId());
        ObjectPtr slave = cache_.getExisting(rel->slaveId());
        const std::string role = key.roleId;
        RoleInfosContainer& linksInMaster = *master->slaveRelationsHolder_.roleRelations(role);
        RoleInfosContainer& linksInSlave = *slave->masterRelationsHolder_.roleRelations(role);
        RelationInfo linkInMaster(RelationType::Slave, slave.get(), rel);
        RelationInfo linkInSlave(RelationType::Master, master.get(), rel);
        RelationInfoKey masterKey(master->id(), role, master->categoryId());
        RelationInfoKey slaveKey(slave->id(), role, slave->categoryId());
        if (!rel->isDeleted()) {
            master->slaveRelationsHolder_.deleted().removeByKey(slaveKey);
            slave->masterRelationsHolder_.deleted().removeByKey(masterKey);
            linksInSlave.relations.push_back(linkInSlave);
            linksInMaster.relations.push_back(linkInMaster);
            if (!rel->hasExistingOriginal()) {
                master->slaveRelationsHolder_.added().push_back(linkInMaster);
                slave->masterRelationsHolder_.added().push_back(linkInSlave);
            }
        } else {
            linksInSlave.relations.removeByKey(master->id());
            linksInMaster.relations.removeByKey(slave->id());
            master->slaveRelationsHolder_.added().removeByKey(slaveKey);
            slave->masterRelationsHolder_.added().removeByKey(masterKey);
            if (rel->hasExistingOriginal()) {
                master->slaveRelationsHolder_.deleted().push_back(linkInMaster);
                slave->masterRelationsHolder_.deleted().push_back(linkInSlave);
            }
        }
        cachedRelations_.emplace(std::move(key), rel);
        cachedMasterRelations_[rel->slaveId()].push_back(rel);
        cachedSlaveRelations_[rel->masterId()].push_back(rel);
        DEBUG() << BOOST_CURRENT_FUNCTION << " Processed relation:"
            << " relation id: " << rel->id()
            << ", master id: " << master->id()
            << ", slave id: " << slave->id()
            << ", role id: " << role
            << ", relation is created: " << rel->isCreated()
            << ", relation is deleted: " << rel->isDeleted();
    }
}

bool
RelationsManager::isRelationExists(TOid masterId, TOid slaveId, const std::string& roleId) const
{
    ObjectPtr master = cache_.getExisting(masterId);
    ObjectPtr slave = cache_.getExisting(slaveId);
    auto it = cachedRelations_.find(RelationKey(
        masterId, slaveId, roleId, master->categoryId(), slave->categoryId()));
    if (it != cachedRelations_.end()) {
        return !it->second->isDeleted();
    }
    const auto& existingRels = cache_.revisionsFacade().findRelation(masterId, slaveId,
                    roleId, master->categoryId(), slave->categoryId());
    return !existingRels.empty();
}

void
RelationsManager::createRelation(TOid masterId, TOid slaveId, const std::string& roleId)
{
    ObjectPtr relation;
    ObjectPtr master = cache_.getExisting(masterId);
    ObjectPtr slave = cache_.getExisting(slaveId);
    auto it = cachedRelations_.find(RelationKey(
        masterId, slaveId, roleId, master->categoryId(), slave->categoryId()));
    if (it != cachedRelations_.end()) {
        if (!it->second->isDeleted()) {
            return;
        }
        relation = cache_.getExisting(it->second->id());
        relation->setState(GeoObject::State::Draft);
    } else {
        std::vector<ObjectPtr> existingRels;
        if (master->original() && slave->original()) {
            DEBUG() << "createRelation: checking if relation already exists, "
                << "master " << masterId << ", slave " << slaveId << ", role " << roleId;
            existingRels =
                cache_.revisionsFacade().findRelation(masterId, slaveId,
                    roleId, master->categoryId(), slave->categoryId());
        }
        if (existingRels.empty()) {
            GeoObjectFactory factory(cache_);
            relation = factory.createNewObject(
                ObjectsClassInfos::relationClassInfo,
                CATEGORY_RELATION);
        } else {
            relation = cache_.add(existingRels[0]);
            for (size_t i = 1; i < existingRels.size(); ++i) {
               auto excessRel = cache_.add(existingRels[i]);
               excessRel->setState(GeoObject::State::Deleted);
            }
        }
        RelationObject* relPtr = as<RelationObject>(relation);
        relPtr->setRelation(
            masterId, master->categoryId(), slaveId, slave->categoryId(), roleId, 0);
    }
    GeoObjectCollection col;
    col.add(relation);
    updateRelations(col);
}

void
RelationsManager::deleteRelation(TOid masterId, TOid slaveId, const std::string& roleId)
{
    ObjectPtr relation;
    ObjectPtr master = cache_.getExisting(masterId);
    ObjectPtr slave = cache_.getExisting(slaveId);
    auto it = cachedRelations_.find(RelationKey(
        masterId, slaveId, roleId, master->categoryId(), slave->categoryId()));
    if (it != cachedRelations_.end()) {
        if (!it->second->isDeleted()) {
            relation = cache_.getExisting(it->second->id());
            relation->setState(GeoObject::State::Deleted);
        }
    }
    DEBUG() << "deleteRelation: checking if relation already exists, "
        << "master " << master->categoryId() << " " << masterId
        << ", slave " << slave->categoryId() << " " << slaveId
        << ", role " << roleId;
    auto existingRels =
        cache_.revisionsFacade().findRelation(masterId, slaveId,
            roleId, master->categoryId(), slave->categoryId());
    for (const auto& existingRel :  existingRels) {
        auto excessRel = cache_.add(existingRel);
        excessRel->setState(GeoObject::State::Deleted);
        if (!relation) {
            relation = excessRel;
        }
    }
    if (relation) {
        GeoObjectCollection col;
        col.add(relation);
        updateRelations(col);
    }
}

} // namespace wiki
} // namespace maps
